package gov.va.cpss.controller;

import static gov.va.cpss.ESAPI.EsapiValidationType.CROSS_SITE_SCRIPTING_PERSISTENT;
import static gov.va.cpss.ESAPI.EsapiValidationType.SAFE_STRING;
import static gov.va.cpss.ESAPI.EsapiValidator.validateStringInput;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.ModelAndView;

import gov.va.cpss.model.BatchItem;
import gov.va.cpss.model.BatchJob;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.BatchStatus;
import gov.va.cpss.service.BatchService;
import gov.va.cpss.service.CbssAccess;
import gov.va.cpss.service.SchedulingService;
import gov.va.cpss.util.CpssUtils;

/**
 * APPS Controller class.
 * 
 * Copyright HPE / DXC / VA
 * 
 * @author Unknown
 * @version 1.0.0
 * 
 */
@Controller
public class APPSController {

	private Logger logger = Logger.getLogger(APPSController.class.getCanonicalName());

	private BatchService batchService;
	private SchedulingService schedulingService;

	public SchedulingService getSchedulingService() {
		return schedulingService;
	}

	public void setSchedulingService(SchedulingService schedulingService) {
		this.schedulingService = schedulingService;
	}

	public BatchService getBatchService() {
		return batchService;
	}

	public void setBatchService(BatchService batchService) {
		this.batchService = batchService;
	}

	@RequestMapping(value = "/annualProcess")
	public ModelAndView annualProcessBatch(ModelAndView model) {
		logger.info("request annual batch.");
		
		// Get the latest list of batch items from the Batch Service.
		List<BatchItem> batchItemList = batchService.getAPPSLatestBatchItemsList();
		List<BatchItem> validatedBatchItemList = new ArrayList<BatchItem>();

		for (BatchItem batchItem : batchItemList) {
			BatchItem validatedBatchItem = new BatchItem(validateBatchJob(batchItem.getJob()), batchItem.getRun());
			validatedBatchItemList.add(validatedBatchItem);
		}

		model.addObject("batchItemL", validatedBatchItemList);
		model.addObject("version", CpssUtils.getVersion());
		model.addObject("timeoutGlobalUrl", CbssAccess.getCbssAccessRedirectUrl());
		model.setViewName("annualProcess");
		return model;
	}

	// JSP ajax to populate the batch run dialog.
	// Requires operator to have ADMIN privileges.
	@RequestMapping(value = "appsBatch/appsRunDialog", params = { "id" }, method = RequestMethod.GET)
	public ModelAndView runDialog(@RequestParam(value = "id", required = true) int id, ModelAndView model) {

		BatchItem bi = batchService.getLatestBatchItemById(id);
		if ((bi.getRun() != null) && bi.getRun().getBatchStatus().getJobStatus() == BatchStatus.JobStatus.RUNNING) {
			model.addObject("batchJob", null);
		} else {
			model.addObject("batchJob", bi.getJob());
		}
		model.setViewName("batch/AppsBatchRunDialog");
		return model;
	}

	@RequestMapping(value = "/appsBatchStatus/{id}", method = RequestMethod.GET)
	public ModelAndView batchStatus(@PathVariable(value = "id") int id, ModelAndView model) {
		// Get the desired batch job by id.

		model.addObject("batchItem", validatedBatchItem(batchService.getLatestBatchItemById(id)));

		model.setViewName("appsBatchStatus");

		return model;
	}

	// JSON ajax to update the main cpss batch html table.
	// Requires operator to have ADMIN privileges.
	@RequestMapping(value = "appsBatch/updateAppsJobStatusTable", method = RequestMethod.GET, headers = "Accept=application/json")
	public @ResponseBody List<BatchItem> updateAppJobStatusTable() {
		// Get most recent run of the batch jobs.
		return batchService.getAPPSLatestBatchItemsList();
	}

	// JSP ajax to populate the batch configuration dialog.
	// Requires operator to have ADMIN privileges.
	@RequestMapping(value = "appsBatch/appsConfigDialog", params = { "id" }, method = RequestMethod.GET)
	public ModelAndView configDialog(@RequestParam(value = "id", required = true) int id, ModelAndView model) {
		// Get the list of the batch types to populate the radio buttons.
		model.addObject("rts", batchService.getBatchTypeList());
		// Get the desired batch job by id.
		model.addObject("batchJob", batchService.getBatchJobById(id));

		model.setViewName("batch/AppsBatchConfigDialog");
		return model;
	}

	// Handle saving of batch updates.
	// Requires operator to have ADMIN privileges.
	@RequestMapping(value = "appsBatch/saveBatch", method = RequestMethod.POST)
	public ModelAndView saveBatch(@ModelAttribute BatchJob batchJob) {
		schedulingService.saveBatchJob(batchJob);
		return new ModelAndView("redirect:/annualProcess");
	}

	// Manually run the specified job.
	// Requires operator to have ADMIN privileges.
	@RequestMapping(value = "appsBatch/runAppsJob", method = RequestMethod.POST)
	public ModelAndView runJob(@ModelAttribute BatchJob batchJob) {
		// Use the Scheduling Service to run the job.
		schedulingService.runJob(batchJob.getId());
		return new ModelAndView("redirect:/annualProcess");
	}

	public BatchItem validatedBatchItem(BatchItem batchItem) {
		BatchItem validatedBatchItem = new BatchItem(validateBatchJob(batchItem.getJob()), batchItem.getRun());
		return validatedBatchItem;
	}

	public BatchJob validateBatchJob(BatchJob job) {
		BatchJob validatedBatchJob = new BatchJob();
		validatedBatchJob.setDescription(validateStringInput(job.getDescription(), CROSS_SITE_SCRIPTING_PERSISTENT));
		validatedBatchJob.setEmail(validateStringInput(job.getEmail(), CROSS_SITE_SCRIPTING_PERSISTENT));
		validatedBatchJob.setName(validateStringInput(job.getName(), CROSS_SITE_SCRIPTING_PERSISTENT));
		validatedBatchJob.setId(
				Integer.valueOf(validateStringInput(Integer.toString(job.getId()), CROSS_SITE_SCRIPTING_PERSISTENT)));
		validatedBatchJob.setSchedule(validateStringInput(job.getSchedule(), CROSS_SITE_SCRIPTING_PERSISTENT));
		validatedBatchJob.setTypeId(Integer
				.valueOf(validateStringInput(Integer.toString(job.getTypeId()), CROSS_SITE_SCRIPTING_PERSISTENT)));
		validatedBatchJob.setScheduleDate(validateStringInput(job.formatScheduleDate(), CROSS_SITE_SCRIPTING_PERSISTENT));
		return validatedBatchJob;
	}

	public BatchRun validateBatchRun(BatchRun run) {
		BatchRun validatedBatchRun = new BatchRun();
		validatedBatchRun.setBatchStatus(validateBatchStatus(run.getBatchStatus()));
		validatedBatchRun.setEndDate(run.getEndDate());
		validatedBatchRun.setId(
				Integer.valueOf(validateStringInput(Integer.toString(run.getId()), CROSS_SITE_SCRIPTING_PERSISTENT)));
		validatedBatchRun.setJobId(Integer
				.valueOf(validateStringInput(Integer.toString(run.getJobId()), CROSS_SITE_SCRIPTING_PERSISTENT)));
		validatedBatchRun.setMessage(validateStringInput(run.getMessage(), SAFE_STRING));
		validatedBatchRun.setStartDate(run.getStartDate());
		validatedBatchRun.setStatusId(Integer
				.valueOf(validateStringInput(Integer.toString(run.getStatusId()), CROSS_SITE_SCRIPTING_PERSISTENT)));

		return validatedBatchRun;

	}

	public BatchStatus validateBatchStatus(BatchStatus status) {
		BatchStatus validatedBatchStatus = new BatchStatus();
		validatedBatchStatus
				.setDescription(validateStringInput(status.getDescription(), CROSS_SITE_SCRIPTING_PERSISTENT));
		validatedBatchStatus.setId(Integer
				.valueOf(validateStringInput(Integer.toString(status.getId()), CROSS_SITE_SCRIPTING_PERSISTENT)));
		validatedBatchStatus.setJobStatus(status.getJobStatus());
		validatedBatchStatus.setName(validateStringInput(status.getName(), CROSS_SITE_SCRIPTING_PERSISTENT));

		return validatedBatchStatus;
	}

}
