package gov.va.cpss.cobol;

import java.io.Serializable;
import java.text.DecimalFormat;
import java.util.HashMap;
import java.util.Map;

/**
 * Money class to handle money related operations.
 * 
 * Copyright HPE / DXC / VA
 * 
 * @author Unknown
 * @version 1.0.0
 * 
 */
@SuppressWarnings("nls")
public class Money implements Serializable {

	//private static final long serialVersionUID = 1L;
	private static final long serialVersionUID = -2705311519737531799L;

	public enum SignConversion {
		OPEN(1, "0", "{"), A(1, "1", "A"), B(1, "2", "B"), C(1, "3", "C"), D(1, "4", "D"), E(1, "5", "E"), F(1, "6",
				"F"), G(1, "7", "G"), H(1, "8", "H"), I(1, "9", "I"), CLOSE(-1, "0", "}"), J(-1, "1", "J"), K(-1, "2",
						"K"), L(-1, "3", "L"), M(-1, "4", "M"), N(-1, "5",
								"N"), O(-1, "6", "O"), P(-1, "7", "P"), Q(-1, "8", "Q"), R(-1, "9", "R");

		private static final Map<String, SignConversion> KEY_LOOKUP = new HashMap<>();
		private static final Map<String, SignConversion> POSITIVE_VALUE_LOOKUP = new HashMap<>();
		private static final Map<String, SignConversion> NEGATIVE_VALUE_LOOKUP = new HashMap<>();

		private int sign;
		private String value;
		private String key;

		public int getSign() {
			return this.sign;
		}

		public String getValue() {
			return this.value;
		}

		public String getKey() {
			return this.key;
		}

		private SignConversion(int inSign, String inValue, String inKey) {
			this.sign = inSign;
			this.value = inValue;
			this.key = inKey;
		}

		public static SignConversion lookupByKey(String key) {
			return KEY_LOOKUP.get(key);
		}

		public static SignConversion lookupByPositiveValue(String value) {
			return POSITIVE_VALUE_LOOKUP.get(value);
		}

		public static SignConversion lookupByNegativeValue(String value) {
			return NEGATIVE_VALUE_LOOKUP.get(value);
		}

		static {
			for (SignConversion t : SignConversion.values()) {
				KEY_LOOKUP.put(t.getKey(), t);
				if (t.getSign() < 0) {
					NEGATIVE_VALUE_LOOKUP.put(t.getValue(), t);
				} else {
					POSITIVE_VALUE_LOOKUP.put(t.getValue(), t);
				}
			}
		}
	}

	public static final String DOUBLE_STRING_FORMAT = "0.00";

	private String cobolValue;
	private long longValue;
	private double doubleValue;

	public Money(String cobol) {
		this.cobolValue = cobol;
		initializeFromCobol();
	}

	public Money(final double inDoubleValue, final int digitsLength) {
		this.doubleValue = inDoubleValue;
		initializeFromDouble(digitsLength);
	}

	private void initializeFromCobol() {

		// Parse the cobol value for the sign of the number (Overpunched sign is
		// the last character of the string).
		final String signLookup = this.cobolValue.substring(this.cobolValue.length() - 1);

		// Identify the associated enumerated type based on the overpunched sign
		// value.
		SignConversion sign = Money.SignConversion.lookupByKey(signLookup);

		// If sign is an unexpected value then value is invalid and throw
		// exception.
		if (sign == null) {
			throw new NumberFormatException(
					"Unexpected Overpunched Cobol Format. Last character of " + this.cobolValue + " not in {,},or A-R.");
		}

		// Convert to long representation.
		String valueS = new StringBuilder(this.cobolValue.substring(0, this.cobolValue.length() - 1)).append(sign.getValue())
				.toString();
		this.longValue = Long.valueOf(valueS).longValue() * sign.getSign();

		// Convert to double representation.
		valueS = new StringBuilder(valueS).insert(valueS.length() - 2, ".").toString();
		this.doubleValue = Double.valueOf(valueS).doubleValue() * sign.getSign();
	}

	private void initializeFromDouble(int digitLength) {

		// Get string builder containing the string format of the absolute value
		// of the double value.
		StringBuilder sb = new StringBuilder(new DecimalFormat("0.00").format(Math.abs(this.doubleValue)));

		// Get absolute long representation value.
		final String absLongS = sb.deleteCharAt(sb.length() - 3).toString();
		this.longValue = Long.valueOf(absLongS).longValue();

		// Verify size is large enough. If not then the number is invalid and
		// throw and exception.
		if (absLongS.length() > digitLength) {
			throw new IndexOutOfBoundsException("Cobol format requires larger size than specified. Size for "
					+ this.doubleValue + " requires at least " + absLongS.length() + " digits.");
		}

		// Get cobol sign lookup value based on the last digit of the number.
		final String valueLookup = String.valueOf(sb.charAt(sb.length() - 1));

		// Preformat the cobol value with specified digit length by prepadding
		// with leading zeroes.
		this.cobolValue = String.format("%%0%dd", Integer.valueOf(digitLength));
		this.cobolValue = String.format(this.cobolValue, Long.valueOf(this.longValue));
		this.cobolValue = this.cobolValue.substring(0, this.cobolValue.length() - 1);

		// Set final representation values based on the sign.
		if (this.doubleValue < 0) {
			this.longValue *= -1;
			this.cobolValue = new StringBuilder(this.cobolValue)
					.append(Money.SignConversion.lookupByNegativeValue(valueLookup).getKey()).toString();
		} else {
			this.cobolValue = new StringBuilder(this.cobolValue)
					.append(Money.SignConversion.lookupByPositiveValue(valueLookup).getKey()).toString();
		}
	}

	public String getCobol() {
		return this.cobolValue;
	}

	public long getLong() {
		return this.longValue;
	}

	public double getDouble() {
		return this.doubleValue;
	}

	public String getDoubleAsString() {
		return new DecimalFormat(DOUBLE_STRING_FORMAT).format(this.doubleValue);
	}
	
	public String getDoubleAsAbsoluteString() {
		return new DecimalFormat(DOUBLE_STRING_FORMAT).format(Math.abs(this.doubleValue));
	}
	
	public String getDoubleWithSignSuffixAsString() {
		DecimalFormat format = new DecimalFormat(DOUBLE_STRING_FORMAT);
		format.setNegativePrefix("");
		format.setNegativeSuffix("-");
		return format.format(this.doubleValue);
	}

	@Override
	public String toString() {
		return "Money [cobolValue=" + this.cobolValue + ", convertedRawValue=" + this.longValue + ", doubleValue=" + this.doubleValue
				+ "]";
	}
	
	@Override
	public boolean equals(Object o) {
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;
		
		Money money = (Money) o;
		
		if (this.doubleValue != money.doubleValue) return false;
		
		if (this.cobolValue == null) {
			if (money.cobolValue != null) return false;
		} else if (! this.cobolValue.equals(money.cobolValue)) return false;
		
		return true;
	}
	
	@Override
	public int hashCode() {
		return Double.valueOf(this.doubleValue).hashCode() + ((this.cobolValue == null) ? 0 : this.cobolValue.hashCode());
	}

}
