Option Explicit


Function fnGetFoldersStartingWith(sPattern)
'*********************************************************************
' Purpose	:	Gets an array of folder names whose path starts with specified pattern 
' Parameters:	sPattern	- Pattern to match in the search   
' Returns:    	arrResult 	- An array of folder names whose path starts with specified pattern.
'							  If sPattern parameter refers to a folder ("\\[ServerName]" or ends with "\"),
'							  this function will return all existing subfolders. If no matches, 
'							  an empty string will be returned
'							  NOTE: Only one level in path hierarchy is examined. 
' Revision History:
' 06/08/10    ZPG	Original creation 
'*********************************************************************	

	Dim arrResult
	Dim bQualified
	Dim sNewPattern
	Dim l_objFSO
	Dim arrRet
	Dim l_objFolder
	Dim sParentFolder
	Dim iCount
	
	On Error Resume Next
	
	ReDim arrResult(0)
	arrResult(0)=""
	
	'Handle the case when the pattern is a fully qualified path
	If Left(sPattern, 2) = "\\" Then
		sNewPattern=Right(sPattern, Len(sPattern)-2)
		bQualified=vbTrue
	Else
		sNewPattern=sPattern
	End If
	
	arrRet=Split(sNewPattern, "\")
	
	Set l_objFSO = CreateObject("Scripting.FileSystemObject")
	
	'Handle the case when sPattern parameter refers to a folder
	If arrRet(Ubound(arrRet))=""  Or (bQualified=vbTrue And Ubound(arrRet)=0) Then
		If l_objFSO.FolderExists(sPattern) Then
			For Each l_objFolder in l_objFSO.GetFolder(sPattern).Subfolders
				arrResult(UBound(arrResult))=l_objFolder.Path
				ReDim Preserve arrResult(UBound(arrResult)+1)
			Next
			'Remove the last array item because it is empty
			If UBound(arrResult)>0 Then ReDim Preserve arrResult(UBound(arrResult)-1) 
		End If
		fnGetFoldersStartingWith=arrResult
		Exit Function
	End If
	
	'Determine the parent folder
	For iCount = 0 To UBound(arrRet)-1 
		sParentFolder=sParentFolder & arrRet(iCount) & "\" 
	Next
	If bQualified=vbTrue Then sParentFolder="\\" & sParentFolder
	
	'Return an empty array if the parent folder doesn't exist
	If Not l_objFSO.FolderExists(sParentFolder) Then 
		fnGetFoldersStartingWith=arrResult
		Exit Function
	End If
	
	'Build an array of folder names matching the pattern
	For Each l_objFolder In l_objFSO.GetFolder(sParentFolder).Subfolders
		If InStr(1, l_objFolder.Path, sPattern, vbTextCompare)>0 Then
			arrResult(UBound(arrResult))=l_objFolder.Path
			ReDim Preserve arrResult(UBound(arrResult)+1)
		End If
	Next
	'Remove the last array item because it is empty
	If UBound(arrResult)>0 Then ReDim Preserve arrResult(UBound(arrResult)-1) 
	
	fnGetFoldersStartingWith=arrResult
	
	'Clean up
	On Error Resume Next
	Set l_objFolder = Nothing
	Set l_objFSO = Nothing
	
End Function


Function fnConstructLogName(sFeedName, sDate)
'*********************************************************************
' Purpose	:	Constructs a log file name for the specified feed and date 
' Parameters:	sFeedName	- Feed name that will be prepended to the log file name
'				sDate		- Date that will be used to construct the log file name   
' Returns:    	sResult 	- Log file name for the specified feed and date
'							  on success, empty string on failure
' Revision History:
' 06/16/10    ZPG	Original creation 
'*********************************************************************	

	Dim sResult
	Dim sName

	On Error Resume Next
	
	'Assume failure
	sResult = ""
	
	'Construct the date part of the log name
	sName = Year(sDate)
	If Month(sDate) > 9 Then
		sName = sName & Month(sDate)
	Else
		sName = sName & "0" & Month(sDate)
	End If
	If Day(sDate) > 9 Then
		sName = sName & Day(sDate)
	Else
		sName = sName & "0" & Day(sDate)
	End If
	
	'Finish constructing the name if the date part is formed correctly
	If Len(sName) = 8 Then 
		sName = sFeedName & sName & ".log"
		sResult = sName
	End If
	
	fnConstructLogName = sResult
	
End Function

Function fnExcludeTransmittedToday(p_objFile, p_bExcludeTransmittedToday)
'*********************************************************************
' Purpose	:	Determines whether a file should be excluded if it was
'						transmitted today. This function supports fnArchiveIncomingFiles
' Parameters:	p_objFile										- File that needs to be checked
'							p_bExcludeTransmittedToday	-	Flag indicating whether files 
'																						transmitted doday should be archived
' Returns:		bResult 	- False if the file was either transmitted before today 
'													or not excluded because it was transmitted today, 
'													True otherwise.
'													False will also be returned on error
' Revision History:
' 10/07/10    ZPG	Original creation 
'*********************************************************************	

	Dim bResult

	On Error Resume Next

	If p_bExcludeTransmittedToday = vbFalse Then
		bResult = vbFalse
	Else
		If DateDiff("d", p_objFile.DateLastModified, Now) > 0 Then
			bResult = vbFalse
		Else
			bResult = vbTrue
		End If
	End If	
	
	fnExcludeTransmittedToday = bResult
	
End Function

Function fnFTPCompleted(p_objFile)
'*********************************************************************
' Purpose	:	Determines whether a file is still being transmitted
' Parameters:	p_objFile	- File that needs to be checked
' Returns:		bResult 	- False if the file is still being transmitted, 
'													True otherwise.
'													False will also be returned on error
' Revision History:
' 07/23/10    ZPG	Original creation 
'*********************************************************************	

	Dim bResult
	Dim lFileSize

	On Error Resume Next

	If p_objFile.DateLastModified = p_objFile.DateCreated Then
		'Confirm first indication that the file is still being transferred
		lFileSize = p_objFile.Size
		wscript.Sleep constFTPDelay * 1000
		If p_objFile.Size > lFileSize Then
			bResult = vbFalse
		Else
			bResult = vbTrue
		End If
	Else
		bResult = vbTrue
	End If
	
	fnFTPCompleted = bResult
	
End Function

Function fnEstablishPath(p_sPath)
'*********************************************************************
' Purpose	:	Create path defined in the argument
' Parameters:	p_sPath	- Path that needs to be established. If it already
'												exists, function simply reports success
' Returns:		sResult - Empty string on success, |-delimited error code  
'												and error description on failure
' Revision History:
' 08/23/10    ZPG	Original creation 
'*********************************************************************	

	Dim sResult
	Dim l_objFSO
	Dim l_iCount
	Dim iMin
	Dim iMax
	Dim sRequestedPath
	Dim sCurrentPath

	On Error Resume Next

	Set l_objFSO = CreateObject("Scripting.FileSystemObject")
	
	'Add trailing backslash to the requested path if it's missing
	If Right(p_sPath, 1) = "\" Then
		sRequestedPath = p_sPath
	Else
		sRequestedPath = p_sPath & "\"
	End If

	iMax = UBound(Split(sRequestedPath, "\"))
	If Left(sRequestedPath, 2) = "\\" Then
		sCurrentPath = "\\" & Split(sRequestedPath, "\")(2) 
		iMin = 3
	Else
		sCurrentPath = Split(sRequestedPath, "\")(0)
		iMin = 1
	End If
	For l_iCount = iMin To iMax
		sCurrentPath = sCurrentPath & "\" & Split(sRequestedPath, "\")(l_iCount)
		If Not l_objFSO.FolderExists(sCurrentPath) Then
			Err.Clear
			l_objFSO.CreateFolder sCurrentPath
			If Not Err.Number = 0 Then
				sResult = Err.Number & "|" & "Unable to create folder " & sCurrentPath & " (" & Err.Description & ")"
				Exit For
			End If
		End If
	Next
	
	fnEstablishPath = sResult
	
	'Clean up
	Set l_objFSO = Nothing
	
End Function

Function fnMoveFile(p_File, p_strDestination, p_iTimeout)
'*********************************************************************
' Purpose	:	Moves a file
' Parameters:	p_File						- File that needs to be moved. May be either full path
'																	or a reference to a File object
'							p_strDestination	- Destination. May be either a full folder path or
'																	full file path. 
'							p_iTimeout				-	Timeout in minutes 
' Returns:		sResult 					- Empty string on success, |-delimited error code  
'																	and error description on failure
' Revision History:
' 11/04/10	ZPG	Original creation 
' 02/17/11	ZPG	1. Corrected move timeout code	
'								2. Corrected returned value
'*********************************************************************	

	Dim sResult
	Dim l_objFSO
	Dim bIsFile
	Dim bIsFolder
	Dim sStartTime
	Dim sOriginalFilePath

	On Error Resume Next
	
	'Ensure that destination path ends with a backslash if it's a folder
	Set l_objFSO = CreateObject("Scripting.FileSystemObject")
	If l_objFSO.FolderExists(p_strDestination) Then
		If Not Right(p_strDestination, 1) = "\" Then p_strDestination = p_strDestination & "\"
	End If
	
	sStartTime = Now
	Select Case TypeName(p_File) 
		Case "String"
			If Trim(p_File) = "" Then
					sResult = ""
			Else
				Err.Clear
				Do While objFSO.FileExists(p_File)
					l_objFSO.MoveFile p_File, p_strDestination
					If Err.Number = 0 Then
						sResult = ""
					Else
						If DateDiff("n", sStartTime, Now) < p_iTimeout Then
							Err.Clear
							wscript.Sleep 30000	'30 seconds
						Else
							sResult = Err.Number & "|" & "Unable to move file " & p_File & " (" & Err.Description & ")"
							Exit Do
						End If
					End If
				Loop 
			End If
		Case "File"
			sOriginalFilePath = p_File.Path
			If Not sOriginalFilePath = "" Then
				Do While sOriginalFilePath = p_File.Path
					p_File.Move p_strDestination
					If Err.Number = 0 Then
						sResult = ""
					Else
						If DateDiff("n", sStartTime, Now) > p_iTimeout Then
							Err.Clear
							wscript.Sleep 30000	'30 seconds
						Else
							sResult = Err.Number & "|" & "Unable to move file " & sOriginalFilePath & " (" & Err.Description & ")"
							Exit Do
						End If
					End If
				Loop 
			End If
		Case Else
			sResult = "0|" & "Invalid argument type '" & TypeName(p_File) & "' was passed istead of string or file object" 
	End Select
	
	fnMoveFile = sResult
	
End Function

Function fnReadReg(p_sRegPath)
'*********************************************************************
' Purpose	:	Retrieves a key from the registry based on a registry path
' Parameters:	p_sRegPath	-	Registry path
' Returns:		arrResult	-	Array containing 3 elements:
'								(0) Key value   
'								(1) Error code (will be 0 on success)
'								(2) Error description (will be blank on success)
' Revision History:
' 02/08/11	ZPG	Original creation 
'*********************************************************************	

	Dim objRegistry
	Dim arrResult
	
	On Error Resume Next

	ReDim arrResult(2)
	
	Set objRegistry = CreateObject("Wscript.shell")
	Err.Clear
	arrResult(0) = objRegistry.RegRead(p_sRegPath)
	arrResult(1) = Err.Number
	arrResult(2) = Err.Description

	fnReadReg = arrResult
	
	Set objRegistry  = Nothing
	
End Function

Function fnFormatDateTime(p_sDateTime)
'*********************************************************************
' Purpose	:	Returns date and time formatted as MM/DD/YYYY HH:MM AM/PM
' Parameters:	p_sDateTime - date and time	
' Returns:		Formatted date and time on success, empty string on error		
' Revision History:
' 04/04/11	ZPG	Original creation 
'*********************************************************************	

Dim sFormattedDateTime

	If IsDate(p_sDateTime) Then
		If Hour(p_sDateTime) < 12 Then
			sFormattedDateTime = Date & " " & Hour(p_sDateTime) & ":" & Right("0" & Minute(p_sDateTime), 2) & " AM"
		Else
			sFormattedDateTime = Date & " " & Hour(p_sDateTime) - 12 & ":" & Right("0" & Minute(p_sDateTime), 2) & " PM"
		End If
	Else
		sFormattedDateTime = ""
	End If
	
	fnFormatDateTime = sFormattedDateTime
	
End Function

Function fnGetScriptDir()
'*********************************************************************
' Purpose	:	Returns the directory path for the VB Script file
' Parameters:	None	
' Returns:		VB Script file path (with a trailing slash)		
' Revision History:
' 07/01/11	ZPG	Original creation 
'*********************************************************************	

Dim l_sPath

	l_sPath = WScript.ScriptFullName
	l_sPath = Left(l_sPath, InStrRev(l_sPath, WScript.ScriptName) - 1)

	fnGetScriptDir = l_sPath
	
End Function
