Option Explicit

'SCHEDULING: This script should be run once a day

'Script parameters
Const constFeedName = "Data Transformation"
Const constRecordLayout = "OncoRecordLayout-A.txt" 
Const constRulesFile = "CCRRules.txt" 
Const constProblemFilesFolder = ".\Problem_Files\"	'IMPORTANT: Include trailing backslash
Const constProcessingFolder = ".\Processing\"		'IMPORTANT: Include trailing backslash
Const constPaddingChar = " "
Const constDateFormat = "YYYYMMDD"
Const constRejectFilesWithIncompleteRecords = True
'Const constRejectFilesWithIncompleteRecords = False
Const constRejectFilesOnInvalidFieldValue = True
'Const constRejectFilesOnInvalidFieldValue = False
Const constRejectNullFieldValueInCondition = True
'Const constRejectNullFieldValueInCondition = False

'Database parameters
Const constDBProvider = "Microsoft.Jet.OLEDB.4.0"
'Const constDBProvider = "Microsoft.ACE.OLEDB.12.0" 
Const constDBName = "ZIPCountyCrossref.mdb"
'Const constDBName = "ZIPCountyCrossref.accdb"

'ADO constants
'Constants - ADO
Const adOpenStatic = 3 
Const adLockOptimistic = 3 
Const adStateClosed = 0

'Environment-specific feed processing parameters are defined immediately after processing INCLUDES

Dim objConn
Dim objRS
Dim objFSO
Dim objFile
Dim colFiles
Dim arrIncludes
Dim iCount
Dim sText
Dim sLog
Dim sErrMsg
Dim sProcName
Dim arrFolders
Dim arrFields
Dim arrRecords
Dim arrRules
Dim sRet
Dim bContinue

ReDim arrIncludes(2)
arrIncludes(0) = "GlobalParms.vbs" 'This file must be included first
arrIncludes(1) = "Logger.vbs"
arrIncludes(2) = "GeneralFunctions.vbs"

For iCount = 0 To UBound(arrIncludes)
	sText = fnIncludeScript(arrIncludes(iCount))
	If Not sText = "" Then
		wscript.Echo "Unable to include script " & arrIncludes(iCount) & " (Err " & Split(sText, "|")(0) & "): " & Split(sText, "|")(1)
		wscript.Quit
	End If
Next

Select Case constEnvironment
	Case "DEVL"
		ExecuteGlobal Replace("Const constFileSourceLocation=#.\Source_Files\#", "#", Chr(34))		'IMPORTANT: Include trailing backslash
		ExecuteGlobal Replace("Const constFileTargetLocation=#.\Ready_For_Upload\#", "#", Chr(34))	'IMPORTANT: Include trailing backslash
		ExecuteGlobal Replace("Const constFileCompletedLocation=#.\Completed\#", "#", Chr(34))		'IMPORTANT: Include trailing backslash
	Case "TEST"
		ExecuteGlobal Replace("Const constFileSourceLocation=#.\Source_Files\#", "#", Chr(34))		'IMPORTANT: Include trailing backslash
		ExecuteGlobal Replace("Const constFileTargetLocation=#.\Ready_For_Upload\#", "#", Chr(34))	'IMPORTANT: Include trailing backslash
		ExecuteGlobal Replace("Const constFileCompletedLocation=#.\Completed\#", "#", Chr(34))		'IMPORTANT: Include trailing backslash
	Case "PROD"
		ExecuteGlobal Replace("Const constFileSourceLocation=#.\Source_Files\#", "#", Chr(34))		'IMPORTANT: Include trailing backslash
		ExecuteGlobal Replace("Const constFileTargetLocation=#.\Ready_For_Upload\#", "#", Chr(34))	'IMPORTANT: Include trailing backslash
		ExecuteGlobal Replace("Const constFileCompletedLocation=#.\Completed\#", "#", Chr(34))		'IMPORTANT: Include trailing backslash
	Case Else
		MsgBox "Unrecognized environment <" & constEnvironment & ">!"
		wscript.Quit
End Select

On Error Resume Next

bContinue = True

'Construct error log file name
sLog = constLogFolder & fnConstructLogName(constFeedName, Date)

'Establish required global objects
Set objFSO = CreateObject("Scripting.FileSystemObject")

'Establish required folders
ReDim arrFolders(3)
arrFolders(0) = constFileTargetLocation
arrFolders(1) = constProblemFilesFolder
arrFolders(2) = constProcessingFolder
arrFolders(3) = constFileCompletedLocation
For iCount = 0 To UBound(arrFolders)
	sRet = fnEstablishPath(arrFolders(iCount))
	If Not sRet = "" Then
		sErrMsg = Split(sRet, "|")(1)
		fnLogEntry sLog, sErrMsg, constError & " " & Split(sRet, "|")(0)
		bContinue = False
		Exit For
	End If
Next

'Delete all unfinished files in Processing Folder
Err.Clear
objFSO.DeleteFile constProcessingFolder & "*.*", True	'Delete read-only files too
If Not Err.Number = 0 Then
	sErrMsg = "Unable to delete unfinished files in " & constProcessingFolder & " folder (" & Err.Description & ")"
	fnLogEntry sLog, sErrMsg, constError & " " & Err.Number
	bContinue = False
End If

'Establish DB connection 
Set objConn = CreateObject("ADODB.Connection")
Set objRS= CreateObject("ADODB.Recordset")
Err.Clear
objConn.Open "Provider=" & constDBProvider & "; Data Source=" & constDBName & ";" 
If Not Err.Number = 0 Then 
	sErrMsg = "Unable to establish connection to " & constDBName & " (" & Err.Description & ")"
	'Make an error log entry
	fnLogEntry sLog, sErrMsg, constError & " " & Err.Number	
End If

If bContinue Then
	'Retrieve record layout
	If fnReadRecordDefinition Then
		'Retrieve data transformation rules
		If fnReadTransformationRules Then
			'Process each file in the monitored folder
			Err.Clear
			Set colFiles = objFSO.GetFolder(constFileSourceLocation).Files
			If Err.Number = 0 Then
				For Each objFile In colFiles
					If fnFTPCompleted(objFile) Then
						'Check if this file has already been processed
						'This is done to handle file manipulation errors
						'from previous script execution sessions
						If objFSO.FileExists(constFileTargetLocation & "\" & objFile.Name) Then
							'Remove processed file from the source folder
							sRet = fnMoveFile(objFile, constFileCompletedLocation, constFileMoveDelay)
							If Not sRet = "" Then
								sErrMsg = "Unable to move processed file " & objFile & " to its archival location "
								sErrMsg = sErrMsg & constFileCompletedLocation & " (" & Split(sRet, "|")(1) & ")"
								fnLogEntry sLog, sErrMsg, constError & " " & Split(sRet, "|")(0)
							End If
						Else
							If fnReadRecords(objFile) Then
								'Validate records
							 	If fnCheckRequiredValues Then
									'Apply data transformations
									If fnApplyTransformations Then
										'Write out transformed set of records to a file
										If fnOutputRecords(constProcessingFolder & objFile.Name) Then
											'Move transformed data file to the target location
											sRet = fnMoveFile(constProcessingFolder & objFile.Name, constFileTargetLocation, constFileMoveDelay)
											If sRet = "" Then
												'Remove processed file from the source folder
												sRet = fnMoveFile(objFile, constFileCompletedLocation, constFileMoveDelay)
												If Not sRet = "" Then
													sErrMsg = "Unable to move processed file to its archival location "
													sErrMsg = sErrMsg & constFileCompletedLocation & " (" & Split(sRet, "|")(1) & ")"
													fnLogEntry sLog, sErrMsg, constError & " " & Split(sRet, "|")(0)
												End If
											Else
												sErrMsg = "Unable to move transformed file " & constProcessingFolder & objFile.Name
												sErrMsg = sErrMsg & " to its target location " & constFileTargetLocation & " (" & Split(sRet, "|")(1) & ")"
												fnLogEntry sLog, sErrMsg, constError & " " & Split(sRet, "|")(0)
											End If
										End If
									Else
										'Move the file to the Problem folder
										sRet = fnMoveFile(objFile, constProblemFilesFolder, constFileMoveDelay)
										If Not sRet = "" Then
											sErrMsg = "Data transformation failed on file " & objFile.Name
											sErrMsg = sErrMsg & ". Unable to move it to " & constProblemFilesFolder & " (" & Split(sRet, "|")(1) & ")"
											fnLogEntry sLog, sErrMsg, constError & " " & Split(sRet, "|")(0)
										End If
									End If
								Else
									'Move the file to the Problem folder
									sRet = fnMoveFile(objFile, constProblemFilesFolder, constFileMoveDelay)
									If Not sRet = "" Then
										sErrMsg = "Record validation failed on file " & objFile.Name
										sErrMsg = sErrMsg & ". Unable to move it to " & constProblemFilesFolder & " (" & Split(sRet, "|")(1) & ")"
										fnLogEntry sLog, sErrMsg, constError & " " & Split(sRet, "|")(0)
									End if
								End If
							Else
								'Move the file to the Problem folder
								sRet = fnMoveFile(objFile, constProblemFilesFolder, constFileMoveDelay)
								If Not sRet = "" Then
									sErrMsg = "Records couldn't be parsed in file " & objFile.Name
									sErrMsg = sErrMsg & ". Unable to move it to " & constProblemFilesFolder & " (" & Split(sRet, "|")(1) & ")"
									fnLogEntry sLog, sErrMsg, constError & " " & Split(sRet, "|")(0)
								End If
							End If
						End If
					End If
				Next
			Else
				sErrMsg = "Unable to retrieve collection of files in " & constFileSourceLocation & " folder (" & Err.Description & ")"
				fnLogEntry sLog, sErrMsg, constError & " " & Err.Number
			End If
		End If
	End If

End If

'Clean up

On Error Resume Next

objRS.Close
Set objRS = Nothing
objConn.Close
Set objConn = Nothing
Set colFiles = Nothing
objFile.Close
Set objFile = Nothing
Set objFSO = Nothing

Function fnIncludeScript(strScriptFile)
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
''    Author:     ebgreen (http://www.visualbasicscript.com/m29285.aspx)
''    Purpose:    Implements Include functionality
''    Takes:      strScriptFile - The name of the file containing the script
''                    			  to import. Must be a valid path.
''	  Returns:    sResult 	- Empty string on success,
'                             |-delimited string containing the error code
'                             and error description on failure.
''    Revision History:
''        06/03/10    ZG	Modified error handling
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

	Dim objFSO
	Dim objFile
	Dim strFile

	On Error Resume Next

	Set objFSO = CreateObject ("Scripting.FileSystemObject")

	'Check to see if the INCLUDE file exists. If it does not, return an error
	If objFSO.FileExists(strScriptFile) Then
		'Open the file to be loaded. If the open fails, return an error.
		Err.Clear
		Set objFile = objFSO.OpenTextFile(strScriptFile)
		If Err.Number = 0 Then
			'The file object is created so read all of the file and ececute it globally to load the class for use
			strFile = objFile.ReadAll
			ExecuteGlobal strFile
			If Err.Number = 0 Then
				fnIncludeScript = ""
			Else
				fnIncludeScript = Err.Number & "|" & "The specified INCLUDE file " & strScriptFile & " could not be loaded: " & Err.Description
			End If
		Else
			fnIncludeScript = Err.Number & "|" & "The specified INCLUDE file " & strScriptFile & " could not be opened for reading: " & Err.Description
		End If
	Else
		fnIncludeScript = "53" & "|" & "The specified INCLUDE file " & strScriptFile & " does not exist"
	End If

	objFile.Close
	Set objFile = Nothing
	Set objFSO = Nothing

End Function

Function fnReadRecordDefinition()
'*********************************************************************
' Purpose	:	Read record definition file. Columns in this  file are:
'				1. Field_Number
'				2. Start_Position
'				3. Length
'				4. Name
'				5. Required_Value
' Parameters:	None
' Returns:		True on success, false on failure
' Revision History:
' 07/01/11    ZPG	Original creation
'*********************************************************************

	Dim l_bResult
	Dim l_objRecordLayout
	Dim l_sLine
	Dim l_arrField

	'Assume success
	l_bResult = True

	If objFSO.FileExists(constRecordLayout) Then
		Set l_objRecordLayout = objFSO.OpenTextFile(constRecordLayout, ForReading)
		'Verify that the first line contains correct column headers
		If l_objRecordLayout.AtEndOfStream Then
			l_bResult = False
			sErrMsg = "Missing column header in the record layout definition file " & constRecordLayout
			fnLogEntry sLog, sErrMsg, constError
		Else
			l_sLine = l_objRecordLayout.ReadLine() 'Read a new line of file list
			'Parse column names in the header
			l_arrField = Split(l_sLine, vbTab)
			If UBound(l_arrField) < 4 Then
				l_bResult = False
				sErrMsg = "Invalid column header in the record layout definition file " & constRecordLayout
				fnLogEntry sLog, sErrMsg, constError
			Else
				If l_arrField(0) = "Field_Number" And l_arrField(1) = "Start_Position" And l_arrField(2) = "Length" And l_arrField(3) = "Name" And l_arrField(4) = "Required_Value" Then
					'Save column headers as the first element of the array (this will make it 1-based for data storage)
					ReDim arrFields(4, 0)
					arrFields(0, 0) = l_arrField(0)	'Field_Number
					arrFields(1, 0) = l_arrField(1)	'Start_Position
					arrFields(2, 0) = l_arrField(2)	'Length
					arrFields(3, 0) = l_arrField(3)	'Name
					arrFields(4, 0) = l_arrField(4)	'Required_Value
					'Read attributes for each field
					Do While Not l_objRecordLayout.AtEndOfStream
						l_sLine = l_objRecordLayout.ReadLine() 'Read a new line of file
						If Not Trim(l_sLine) = "" Then
							ReDim Preserve arrFields(4, UBound(arrFields, 2) + 1)
							l_arrField = Split(l_sLine, vbTab)
							If UBound(l_arrField) < 4 Then
								l_bResult = False
								sErrMsg = "Invalid definition for field " & UBound(arrFields, 2) & " in the record layout definition file " & constRecordLayout
								fnLogEntry sLog, sErrMsg, constError
							Else
								If IsNumeric(l_arrField(0)) And IsNumeric(l_arrField(1)) And IsNumeric(l_arrField(2)) Then
									'Save field attributes
									arrFields(0, UBound(arrFields, 2)) = l_arrField(0)	'Field_Number
									arrFields(1, UBound(arrFields, 2)) = l_arrField(1)	'Start_Position
									arrFields(2, UBound(arrFields, 2)) = l_arrField(2)	'Length
									arrFields(3, UBound(arrFields, 2)) = l_arrField(3)	'Name
									arrFields(4, UBound(arrFields, 2)) = l_arrField(4)	'Required_Value
								Else
									l_bResult = False
									sErrMsg = "Invalid definition for field " & UBound(arrFields, 2) & " in the record layout definition file " & constRecordLayout
									fnLogEntry sLog, sErrMsg, constError
								End If
							End If
						End If
					Loop
				Else
					l_bResult = False
					sErrMsg = "Invalid column name in the header of the record layout definition file " & constRecordLayout
					fnLogEntry sLog, sErrMsg, constError
				End If
			End If
		End If
	Else
		l_bResult = False
		sErrMsg = "Record layout definition file " & constRecordLayout & " is not found"
		fnLogEntry sLog, sErrMsg, constError
	End If

	fnReadRecordDefinition = l_bResult

	'Clean up
	On Error Resume Next
	l_objRecordLayout.Close
	Set l_objRecordLayout = Nothing

End Function

Function fnReadRecords(p_objFile)
'*********************************************************************
' Purpose	:	Read records from specified file. Fields will be parsed
'				according to previously retrieved record layout
'				(see fnReadRecordDefinition)
' Parameters:	p_objFile - File containing records to be read
' Returns:		True on success, false on failure
'NOTE: 	Depending on constRejectFilesWithIncompleteRecords, a file containing
'		an incomplete record will either be rejected in its entirety (return FALSE)
'		or accepted with just the incomplete record discarded (return TRUE)
' Revision History:
' 07/01/11    ZPG	Original creation
'*********************************************************************

	Dim l_bResult
	Dim l_objData
	Dim l_iRecLength
	Dim l_sRecord
	Dim l_iCount

	'Assume success
	l_bResult = True

	'Calculate record length based on record definition
	l_iRecLength = CInt(arrFields(1, UBound(arrFields, 2))) + CInt(arrFields(2, UBound(arrFields, 2)) - 1)

	'Prepare array that will hold records field by field
	ReDim arrRecords(UBound(arrFields, 2), 0)

	If objFSO.FileExists(p_objFile.Path) Then
		Set l_objData = objFSO.OpenTextFile(p_objFile.Path, ForReading)
		Do While Not l_objData.AtEndOfStream
			'Read record
			l_sRecord = l_objData.ReadLine() 
' 			l_sRecord = l_objData.Read(l_iRecLength)
			If Not (Len(l_sRecord) = l_iRecLength) Then 
				If constRejectFilesWithIncompleteRecords Then
					l_bResult = False
					sErrMsg = "Record of incorrect length (" & Len(l_sRecord) & " bytes instead of expected "
					sErrMsg = sErrMsg & l_iRecLength & ") is found in file " & p_objFile.Path
					fnLogEntry sLog, sErrMsg, constError
				End If
			Else
				'Parse record into individual fields
				ReDim Preserve arrRecords(UBound(arrFields, 2), UBound(arrRecords, 2) + 1)
				For l_iCount = 1 to UBound(arrFields, 2)
					arrRecords(l_iCount, UBound(arrRecords, 2)) = Mid(l_sRecord, arrFields(1, l_iCount), arrFields(2, l_iCount))
				Next
			End If
		Loop
	Else
		l_bResult = False
		sErrMsg = "Data file " & p_objFile.Path & " is not found"
		fnLogEntry sLog, sErrMsg, constError
	End If

	fnReadRecords = l_bResult

	'Clean up
	On Error Resume Next
	l_objData.Close
	Set l_objData = Nothing

End Function

Function fnReadTransformationRules()
'*********************************************************************
' Purpose	:	Read data transformation rules
'		NOTE:	Expected rule layout is <Condition>|<Transformation>|<Comment>
'				Comment is optional
' Parameters:	None
' Returns:		True on success, false on failure
' Revision History:
' 07/01/11    ZPG	Original creation
'*********************************************************************

	Dim l_objRules
	Dim l_bResult
	Dim l_arrRule
	Dim l_sLine

	'Assume success
	l_bResult = True

	ReDim arrRules(1, 0)

	If objFSO.FileExists(constRulesFile) Then
		Set l_objRules = objFSO.OpenTextFile(constRulesFile, ForReading)
		'Read each rule
		Do While Not l_objRules.AtEndOfStream
			l_sLine = l_objRules.ReadLine() 'Read a new line of file
			If Not Trim(l_sLine) = "" Then
				ReDim Preserve arrRules(1, UBound(arrRules, 2) + 1)
				l_arrRule = Split(l_sLine, "|")
				If (UBound(l_arrRule) = 1) Or (UBound(l_arrRule) = 2) Then
					arrRules(0, UBound(arrRules, 2)) = l_arrRule(0) 'Condition
					arrRules(1, UBound(arrRules, 2)) = l_arrRule(1) 'Transformation
				Else
					l_bResult = False
					sErrMsg = "Invalid definition for rule #" & UBound(arrRules, 2) & " found in file " & constRulesFile
					fnLogEntry sLog, sErrMsg, constError
					Exit Do
				End If
			End If
		Loop
	Else
		l_bResult = False
		sErrMsg = "File containing data transformation rules " & constRulesFile & " is not found"
		fnLogEntry sLog, sErrMsg, constError
	End If

	fnReadTransformationRules = l_bResult

	'Clean up
	On Error Resume Next
	l_objData.Close
	Set l_objData = Nothing

End Function

Function fnApplyTransformations()
'*********************************************************************
' Purpose	:	Apply transformations to current set of records
' Parameters:	None
' Returns:		True on success, false on failure
' Revision History:
' 07/05/11    ZPG	Original creation
'*********************************************************************

	Dim l_iCount
	Dim l_iCount1
	Dim l_iCount2
	Dim l_arrConditions
	Dim l_arrTransformations
	Dim l_bRet
	Dim l_iRet
	Dim l_bResult
	Dim l_sCondition
	Dim l_sTransformation
	Dim l_sCountyCode

	'Initialize
	l_bResult = True

	'Scroll through the set of records
	For l_iCount = 1 To UBound(arrRecords, 2)
		'Scroll through the set of rules
		For l_iCount1 = 1 To UBound(arrRules, 2)
			'Extract a set of conditions from the current rule
			l_arrConditions = Split(arrRules(0, l_iCount1), ",")
			'Check whether current rule applies to the current record
			l_bRet = True
			For l_iCount2 = 0 To UBound(l_arrConditions)
				l_iRet = fnCheckCondition(l_iCount, l_arrConditions(l_iCount2))
				If l_iRet = 2 Then
					l_bResult = False
					Exit For	'l_iCount2 loop
				Else
					l_bRet = l_bRet And CBool(l_iRet)
				End If
			Next
			If l_bResult = False Then
				Exit For	'l_iCount1 loop iterating rules
			Else
				If l_bRet Then
					'Extract a set of transformations from the current rule
					l_arrTransformations = Split(arrRules(1, l_iCount1), ",")
					'Transform record according to the rule
					For l_iCount2 = 0 To UBound(l_arrTransformations)
						l_bResult = l_bResult And fnTransformField(l_iCount, l_arrTransformations(l_iCount2))
						If l_bResult = False Then Exit For 'l_iCount2 loop iterating transformations
					Next
				End If
			End If
		Next
		If l_bResult = False Then Exit For	'l_iCount loop iterating records
		
		'Apply ZIP to County conversion
		l_sCondition = "15=[998][999]'"
		l_iRet = fnCheckCondition(l_iCount, l_sCondition)
		If CBool(l_iRet) Then
			l_sCountyCode = fnGetCountyCode(arrRecords(14, l_iCount))
			If Not l_sCountyCode = "" Then
				l_sTransformation = "15=" & l_sCountyCode
				fnTransformField l_iCount, l_sTransformation
			End If
		End If
	Next

	fnApplyTransformations = l_bResult

End Function

Function fnCheckCondition(p_iRecNum, p_sCondition)
'*********************************************************************
' Purpose	:	Check whether the record meets specified condition
'				See l_arrOperators for the list of recognized operators
' Parameters:	p_iRecNum 		- Record number
'				p_sCondition 	- Condition
' Returns:		0 if the record doesn't meet specified condition
'				1 if the record meets specified condition
'				2 if unable to determine due to error
' NOTE:	Depending on constRejectFilesOnInvalidFieldValue, a field value that is
'		invalid in the context of the condition being checked will either cause 
'		an error(return 2) or will simply fail to match (return 0). 
'		The error will be logged either way.
' NOTE:	Depending on constRejectNullFieldValueInCondition, null field values in date 
'		and numeric comparisons will either cause an error (return 2) or 
'		will simply fail to match (return 0). 
' Revision History:
' 07/02/11    ZPG	Original creation
'*********************************************************************

	Dim l_iResult
	Dim l_arrOperators
	Dim l_arrCondition
	Dim l_iFieldNum
	Dim l_sDataType
	Dim l_sValue	'String
	Dim l_cValue	'Numeric
	Dim l_dValue	'Date
	Dim l_sOperator
	Dim l_iCount
	Dim l_sExpression
	Dim l_sFieldValue
	Dim l_bInClause
	Dim l_arrInClause
	Dim l_bValid

	sProcName = "fnCheckCondition"

	'Define recognized operators in specified condition
	'NOTE: Operators including other operators must be listed first
	ReDim l_arrOperators(5)
	l_arrOperators(0) = ">="
	l_arrOperators(1) = "<="
	l_arrOperators(2) = "<>"
	l_arrOperators(3) = "="
	l_arrOperators(4) = ">"
	l_arrOperators(5) = "<"

	'Assume error
	l_iResult = 2

	'Validate specified record number
	If p_iRecNum < 1 Or p_iRecNum > UBound(arrRecords, 2) Then
		sErrMsg = "[" & objFile.Name & "] " & sProcName & ": Invalid record number " & p_iRecNum & " specified for condition " & p_sCondition
		sErrMsg = sErrMsg & ". Valid record numbers are 1 through " & UBound(arrRecords, 2) & "."
		fnLogEntry sLog, sErrMsg, constError
	Else
		'Determine operator
		For l_iCount = 0 To UBound(l_arrOperators)
			If InStr(p_sCondition, l_arrOperators(l_iCount)) > 0 Then
				l_sOperator = l_arrOperators(l_iCount)
				Exit For
			End If
		Next
		If l_sOperator = "" Then
			sErrMsg = "[" & objFile.Name & "] " & sProcName & ": No recognized comparison operator in specified condition " & p_sCondition
			fnLogEntry sLog, sErrMsg, constError
		Else
			'Parse condition specification
			l_arrCondition = Split(p_sCondition, l_sOperator)
			If UBound(l_arrCondition) = 1 Then
				l_iFieldNum = CInt(Trim(l_arrCondition(0)))
				l_sValue = Trim(l_arrCondition(1))
				'Validate field number
				If l_iFieldNum < 1 Or l_iFieldNum > UBound(arrFields, 2) Then
					sErrMsg = "[" & objFile.Name & "] " & sProcName & ": Invalid field number " & l_iFieldNum & " specified for condition "
					sErrMsg = sErrMsg & p_sCondition & ". Valid field numbers are 1 through " & UBound(arrFields, 2) & "."
					fnLogEntry sLog, sErrMsg, constError
				Else
					'Determine value data type
					If IsNumeric(l_sValue) Then
						l_bInClause = False
						l_sDataType = "Numeric"
						l_cValue = CCur(l_sValue)
					Else
						If Left(l_sValue, 1) = "'" And Right(l_sValue, 1) = "'" Then
							l_bInClause = False
							l_sDataType = "String"
							l_sValue = Mid(l_sValue, 2, Len(l_sValue)-2)
						ElseIf Left(l_sValue, 1) = "#" And Right(l_sValue, 1) = "#" Then
							l_sValue = Mid(l_sValue, 2, Len(l_sValue)-2)
							If IsDate(l_sValue) Then
								l_bInClause = False
								l_sDataType = "Date"
								l_dValue = CDate(l_sValue)
							Else
								sErrMsg = "[" & objFile.Name & "] " & sProcName & ": Invalid data type specified for condition " & p_sCondition
								fnLogEntry sLog, sErrMsg, constError
							End If
						ElseIf Left(l_sValue, 1) = "[" And Right(l_sValue, 1) = "]" Then
							'Parse the In clause
							l_arrInClause = Split(Mid(l_sValue, 2, Len(l_sValue)-2), "][")
							'Validate values listed in the In clause
							For l_iCount = 0 To UBound(l_arrInClause)
								l_bValid = True
								If Not IsNumeric(l_arrInClause(l_iCount)) Then
									l_bValid = False
									Exit For
								End If
							Next
							'Set data type and In Clause flag
							If l_bValid Then
								l_bInClause = True
								l_sDataType = "Numeric"
							End If
						ElseIf Left(l_sValue, 1) = "[" And Right(l_sValue, 2) = "]'" Then
							'Parse the In clause
							l_arrInClause = Split(Mid(l_sValue, 2, Len(l_sValue)-3), "][")
							'Set data type and In Clause flag
							l_bInClause = True
							l_sDataType = "String"
						ElseIf Left(l_sValue, 1) = "[" And Right(l_sValue, 2) = "]#" Then
							'Parse the In clause
							l_arrInClause = Split(Mid(l_sValue, 2, Len(l_sValue)-3), "][")
							'Validate values listed in the In clause
							For l_iCount = 0 To UBound(l_arrInClause)
								l_bValid = True
								If Not IsDate(l_arrInClause(l_iCount)) Then
									l_bValid = False
									Exit For
								End If
							Next
							'Set data type and In Clause flag
							If l_bValid Then
								l_bInClause = True
								l_sDataType = "Date"
							End If
						Else
							sErrMsg = "[" & objFile.Name & "] " & sProcName & ": Invalid data type specified for condition " & p_sCondition
							fnLogEntry sLog, sErrMsg, constError
						End If
					End If
					'Remove characters that pad actual field value to fixed length
					l_sFieldValue = fnTrimPadded(arrRecords(l_iFieldNum, p_iRecNum))
					l_sExpression = ""
					Select Case l_sDataType
						Case "Numeric"
							'Validate the value in the field
							If IsNumeric(l_sFieldValue) Then
								'Construct expression for the condition
								If l_bInClause Then
									For l_iCount = 0 To UBound(l_arrInClause)
										l_sExpression = l_sExpression & "(" & l_sFieldValue & l_sOperator & l_arrInClause(l_iCount) & ") Or "
									Next
									l_sExpression = Left(l_sExpression, Len(l_sExpression) - Len(" Or "))
								Else
									l_sExpression = l_sFieldValue & l_sOperator & l_cValue
								End If
							Else
								If l_sFieldValue = "" And Not constRejectNullFieldValueInCondition Then
									l_iResult = 0
								Else
									sErrMsg = "[" & objFile.Name & "] " & sProcName & ": Numeric comparison " & p_sCondition & " specified for record number "								
									sErrMsg = sErrMsg & p_iRecNum & ", but the field value <" & l_sFieldValue & "> is not numeric"
									fnLogEntry sLog, sErrMsg, constError
									If Not constRejectFilesOnInvalidFieldValue Then
										l_iResult = 0
									End If
								End If
							End If
						Case "String"
							'Construct expression for the condition - case insensitive operation
							If l_bInClause Then
								For l_iCount = 0 To UBound(l_arrInClause)
									l_sExpression = l_sExpression & "(" & Chr(34) & UCase(l_sFieldValue) & Chr(34) & l_sOperator & Chr(34)
									l_sExpression = l_sExpression & UCase(l_arrInClause(l_iCount)) & Chr(34) & ") Or "
								Next
								l_sExpression = Left(l_sExpression, Len(l_sExpression) - Len(" Or "))
							Else
								l_sExpression = Chr(34) & UCase(l_sFieldValue) & Chr(34) & l_sOperator & Chr(34)
								l_sExpression = l_sExpression & UCase(l_sValue) & Chr(34)
							End If
						Case "Date"
							'Validate the value in the field
							If fnFormatDate(l_sFieldValue) = "" Then
								If l_sFieldValue = "" And Not constRejectNullFieldValueInCondition Then
									l_iResult = 0
								Else
									sErrMsg = "[" & objFile.Name & "] " & sProcName & ": Date comparison " & p_sCondition & " specified for record number "
									sErrMsg = sErrMsg & p_iRecNum & ", but the field value <" & l_sFieldValue & "> is not a date in the format " & constDateFormat
									fnLogEntry sLog, sErrMsg, constError
									If Not constRejectFilesOnInvalidFieldValue Then
										l_iResult = 0
									End If
								End If
							Else
								'Construct expression for the condition
								If l_bInClause Then
									For l_iCount = 0 To UBound(l_arrInClause)
										l_sExpression = l_sExpression & "(" & CLng(CDate(fnFormatDate(l_sFieldValue))) & l_sOperator
										l_sExpression = l_sExpression & CLng(l_arrInClause(l_iCount)) & ") Or "
									Next
									l_sExpression = Left(l_sExpression, Len(l_sExpression) - Len(" Or "))
								Else
									l_sExpression = CLng(CDate(fnFormatDate(l_sFieldValue))) & l_sOperator & CLng(l_dValue)
								End If
							End If
					End Select
					If Len(l_sExpression) > 0 Then
						l_iResult = Abs(CInt(Eval(l_sExpression)))
					End If
				End If
			Else
				sErrMsg = "[" & objFile.Name & "] " & sProcName & ": Invalid condition " & p_sCondition & " specified for record number " & p_iRecNum
				fnLogEntry sLog, sErrMsg, constError
			End If
		End If
	End If

	fnCheckCondition = l_iResult

End Function

Function fnTransformField(p_iRecNum, p_sTransformation)
'*********************************************************************
' Purpose	:	Apply specified transformation to the field of a record
' Parameters:	p_iRecNum 			- Record number
'				p_sTransformation	- Transformation describing the field and its new value
' Returns:		True on success, false on failure
' Revision History:
' 07/02/11    ZPG	Original creation
'*********************************************************************

	Dim l_bResult
	Dim l_arrTransformation
	Dim l_iFieldNum
	Dim l_sNewValue

	sProcName = "fnTransformField"

	'Assume failure
	l_bResult = False

	'Validate specified record number
	If p_iRecNum < 1 Or p_iRecNum > UBound(arrRecords, 2) Then
		sErrMsg = "[" & objFile.Name & "] " & sProcName & ": Invalid record number " & p_iRecNum & " specified for transformation "
		sErrMsg = sErrMsg & p_sTransformation & ". Valid record numbers are 1 through " & UBound(arrRecords, 2) & "."
		fnLogEntry sLog, sErrMsg, constError
	Else
		'Parse transformation specification
		l_arrTransformation = Split(p_sTransformation, "=")
		If UBound(l_arrTransformation) = 1 Then
		l_iFieldNum = CInt(l_arrTransformation(0))
			l_sNewValue = l_arrTransformation(1)
			'Validate field number
			If l_iFieldNum < 1 Or l_iFieldNum > UBound(arrFields, 2) Then
				sErrMsg = "[" & objFile.Name & "] " & sProcName & ": Invalid field number " & l_iFieldNum & " specified for transformation "
				sErrMsg = sErrMsg & p_sTransformation & ". Valid field numbers are 1 through " & UBound(arrFields, 2) & "."
				fnLogEntry sLog, sErrMsg, constError
			Else
				If UCase(l_sNewValue) = "NULL" Then
					'Set record to Null (fill entire field with padding characters)
					arrRecords(l_iFieldNum, p_iRecNum) = String(arrFields(2, l_iFieldNum), constPaddingChar)
					l_bResult = True
				Else
					'Validate new value for the field
					If Len(l_sNewValue) > CInt(arrFields(2, l_iFieldNum)) Then
						sErrMsg = "[" & objFile.Name & "] " & sProcName & ": Value in specified transformation "
						sErrMsg = sErrMsg & p_sTransformation & " is too long. Valid field length is " & arrFields(2, l_iFieldNum) & "."
						fnLogEntry sLog, sErrMsg, constError
					Else
						'Apply transformation to the record
						arrRecords(l_iFieldNum, p_iRecNum) = Left(l_sNewValue & String(arrFields(2, l_iFieldNum), constPaddingChar), arrFields(2, l_iFieldNum))
						l_bResult = True
					End If
				End If
			End If
		Else
			sErrMsg = "[" & objFile.Name & "] " & sProcName & ": Invalid transformation " & p_sTransformation & " specified for record number " & p_iRecNum
			fnLogEntry sLog, sErrMsg, constError
		End If
	End If

	fnTransformField = l_bResult

End Function

Function fnFormatDate(p_sDate)
'*********************************************************************
' Purpose	:	Reformats specified date from the format described in
'				constDateFormat to the format matching current system locale setting
' Parameters:	p_sDate	- Date to be reformatted
' Returns:		Properly formatted string on success, empty string on failure
' Revision History:
' 07/02/11	ZPG	Original creation
' 08/23/11	ZPG	Added handling of incomplete dates (just YYYY or YYYYMM)
'*********************************************************************

	Dim l_sResult
	Dim l_sDate
	Dim l_iPosition
	Dim l_dTest
	
	Dim l_sYear
	Dim l_sMonth
	Dim l_sDay

	On Error Resume Next

	'Assume failure
	l_sResult = ""
	
	'Parse year
	l_iPosition = InStr(UCase(constDateFormat), "YYYY")
	If l_iPosition > 0 Then
		l_sYear = Mid(p_sDate, l_iPosition, 4)
	Else
		l_iPosition = InStr(UCase(constDateFormat), "YY")
		If l_iPosition > 0 Then
			l_sYear = Mid(p_sDate, l_iPosition, 2)
		End If
	End If
	
	'Parse month
	l_iPosition = InStr(UCase(constDateFormat), "MM")
	If l_iPosition > 0 Then
		l_sMonth = Mid(p_sDate, l_iPosition, 2)
		If fnTrimPadded(l_sMonth) = "" Then l_sMonth = "01"
	End If
	
	'Parse day
	l_iPosition = InStr(UCase(constDateFormat), "DD")
	If l_iPosition > 0 Then
		l_sDay = Mid(p_sDate, l_iPosition, 2)
		If fnTrimPadded(l_sDay) = "" Then l_sDay = "01"
	End If
	
	'Convert result to date
	l_dTest = CDate(l_sMonth & "/" & l_sDay & "/" & l_sYear)
	If Err.Number = 0 Then
		l_sResult = CStr(l_dTest)
	End If

	fnFormatDate = l_sResult

End Function

Function fnTrimPadded(p_sValue)
'*********************************************************************
' Purpose	:	Removes padding characters from the field value.
'				The value is assumed to be left-justified, i.e padded
'				to fixed langth on the right.
' Parameters:	p_sValue	- Fixed length field value
' Returns:		Field value with padding characters removed
' Revision History:
' 07/05/11    ZPG	Original creation
'*********************************************************************

	Dim l_sTrimmedValue

	l_sTrimmedValue = p_sValue

	If Not l_sTrimmedValue = "" Then
		If l_sTrimmedValue = String(Len(l_sTrimmedValue), constPaddingChar) Then
			l_sTrimmedValue = ""
		Else
			Do While Right(l_sTrimmedValue, 1) = constPaddingChar
				l_sTrimmedValue = Left(l_sTrimmedValue, Len(l_sTrimmedValue)-1)
			Loop
		End If
	End If

	fnTrimPadded = l_sTrimmedValue

End Function

Function fnOutputRecords(p_sTargetFile)
'*********************************************************************
' Purpose	:	Writes current set of records to a file
' Parameters:	p_sTargetFile	- Full path and name of the file where
'								  records will be written out
' Returns:		True on success, false on failure
' Revision History:
' 07/05/11    ZPG	Original creation
'*********************************************************************

	Dim l_bResult
	Dim l_objOutputFile
	Dim l_iCount
	Dim l_iCount1
	Dim l_sText

	'Assume success
	l_bResult = True

	'Create output file
	Err.Clear
	Set l_objOutputFile = objFSO.CreateTextFile(p_sTargetFile, True)	'Overwrite if file already exists
	If Err.Number = 0 Then
		'Scroll through all existing records in current set
		For l_iCount = 1 To UBound(arrRecords, 2)
			l_sText = ""
			'Append each field value to the new line of output file
			For l_iCount1 = 1 To UBound(arrFields, 2)
				l_sText = l_sText & arrRecords(l_iCount1, l_iCount)
			Next
			'Save resulting record to file
			Err.Clear
			l_objOutputFile.WriteLine(l_sText)
			If Not Err.Number = 0 Then
				l_bResult = False
				sErrMsg = "Unable to save transformed record #" & l_iCount & " to output file " 
				sErrMsg = sErrMsg & p_sTargetFile & " (" & Err.Description & ")"
				fnLogEntry sLog, sErrMsg, constError & " " & Err.Number
				Exit For
			End If
		Next
	Else
		l_bResult = False
		sErrMsg = "Unable to create a transformed record  output file " & p_sTargetFile & " (" & Err.Description & ")"
		fnLogEntry sLog, sErrMsg, constError & " " & Err.Number
	End If

	fnOutputRecords = l_bResult

	'Clean up
	On Error Resume Next
	l_objOutputFile.Close
	Set l_objOutputFile = Nothing

End Function

Function fnGetCountyCode(p_sZIPCode)
'*********************************************************************
' Purpose	:	Looks up county code by ZIP code
' Parameters:	p_sZIPCode	- ZIP code that needs to be translated to 
'							the county code (5 digits + optional 4 digits)
' Returns:		County code on success, empty string on failure
' Revision History:
' 07/12/11    ZPG	Original creation
'*********************************************************************

	Dim l_sSQL
	Dim l_sResult

	sProcName = "fnGetCountyCode"

	'Initialize
	l_sResult = ""

	'Close recordset if it was previously opened
	If Not objRS.State = adStateClosed Then objRS.Close

	'Look up the county code
	l_sSQL="Select CountyCode from ZIP_To_County where ZIP='" & Left(p_sZIPCode, 5) & "'"
	objRS.Open l_sSQL, objConn, adOpenStatic, adLockOptimistic
	If Err.Number = 0 Then 
		If objRS.RecordCount > 0 Then
			l_sResult = objRS("CountyCode")
		End If
	Else
		sErrMsg = sProcName & ": Unable to open recordset " & l_sSQL & " in " & constDBName & " (" & Err.Description & ")"
		fnLogEntry sLog, sErrMsg, constError & " " & Err.Number
	End If

	fnGetCountyCode = l_sResult

End Function

Function fnCheckRequiredValues()
'*********************************************************************
' Purpose	:	Validates current set of records against the list of 
'           required values that are always expected in a correctly 
'           formed record
' Parameters:	None (this function uses global arrays arrFields
'             and arrRecords for validation) 
' Returns:		True on success, false on failure
' Revision History:
' 07/20/11    ZPG	Original creation
'*********************************************************************

  Dim l_iCount
  Dim l_iCount1
  Dim l_bRet
  
  l_bRet = True

  For l_iCount = 1 To UBound(arrFields, 2)
    If Not arrFields(4, l_iCount) = "" Then
    	For l_iCount1 = 1 to UBound(arrRecords, 2)
    		If Not arrRecords(l_iCount, l_iCount1) = arrFields(4, l_iCount) Then
    			l_bRet = False
				sErrMsg = objFile.Name & ": Record " & l_iCount1 & " has value <" & arrRecords(l_iCount, l_iCount1)  
				sErrMsg = sErrMsg & "> instead of expected <" & arrFields(4, l_iCount) & "> in field " & l_iCount 
				sErrMsg = sErrMsg & " (" & arrFields(3, l_iCount) & ")"
				fnLogEntry sLog, sErrMsg, constError    			
    		End If
    	Next 
    End If
  Next
  
  fnCheckRequiredValues = l_bRet
  
End Function

