﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using System.Text;
using System.IO;
using System.Diagnostics;
using System.Configuration;
using System.Reflection;
using gov.va.medora.mdo;
using gov.va.medora.mdo.api;
using gov.va.medora.mdo.dao;
using gov.va.medora.mdo.dao.vista;
using gov.va.medora.mdo.src;
using gov.va.medora.mdo.src.mdo;
using gov.va.medora.mdo.utils;

namespace GetOncology
{
    class Program
    {
        //constants
		const string constScriptName = "DataTransformation.vbs";
		const string constLogPrefix = "OncoTraX_Data_Extraction";
		const string constReportPrefix = "OncoTraX";

        //enumerators
        public enum status {Present, Added, Removed};
        public enum type {New = 1, Updated};
		public enum entryTypes {Blank, Data, Info, Error };
        public enum nameParts{City, State};

        //variables
        public struct udSite
        {
            public string ID;
            public string City;
            public string State;
            public string lastNew;
            public string lastUpdated;
            public status siteStatus;
            public string newCount;
			public string updatedCount;
		}
		public static string logFolder;
		public static string dataFolder;
		public static string scriptFolder;
        public static string errMsg;
        public static string procName;
		public static string logFile;
		public static string rptFile;
        public static Dictionary<string, udSite> dicSites = new Dictionary<string, udSite>();
        public static SiteTable siteTbl;
        public static string siteInfoFile;
        public static string siteTableFile;

		static string fnLaunchVBScript(string scriptName, bool waitFinish)
		{
		//*****************************************************************************
		//Puprose:      Executes a VBScript file
		//Parameters:   scriptName	- VBScript file name with full path
		//				waitFinish	- Flag whether the app should wait for the script 
		//							to finish running
		//Returns:      Empty string on success, error message on failure
		//Revision History:
		//08/23/11  ZPG Original creation
		//*****************************************************************************

			string ret = "";

			try
			{
				Process scriptProc = new Process(); 
				scriptProc.StartInfo.FileName = @"wscript";
				if (File.Exists(scriptName))
				{
					scriptProc.StartInfo.Arguments = "//B " + scriptName; //Suppress error message pop-ups
					scriptProc.StartInfo.UseShellExecute = false;
					scriptProc.StartInfo.ErrorDialog = false;
					scriptProc.Start();
					if (waitFinish)
					{
						scriptProc.WaitForExit(); //Run synchronously
					}
					if (scriptProc != null)
					{
						scriptProc.Close();
					}
				}
				else
				{
					ret = "Script could not be found";
				}
			}
			catch (Exception ex)
			{
				ret = ex.Message;
			}

			return ret;

		}

		static void fnSaveSiteInfo()
		{
		//*****************************************************************************
		//Puprose:      Saves latest info about VistA sites
		//Parameters:   None
		//Returns:      None
		//Revision History:
		//08/23/11  ZPG Original creation
		//*****************************************************************************

			string siteText;
			StreamWriter SW;
			FileInfo FI;
			udSite udsCurrent;

			//Ensure that directory where site info file will be stored exists
			FI = new FileInfo(siteInfoFile);
			if (Directory.Exists(FI.DirectoryName) == false)
			{
				try
				{
					Directory.CreateDirectory(FI.DirectoryName);
				}
				catch (Exception ex)
				{
                    errMsg = "Unable to establish folder for site info file " + siteInfoFile + " (" + ex.Message + ")";
					fnLogEntry (logFile, errMsg, entryTypes.Error);
				}
			}

			//Create a new site info file
			if (Directory.Exists(FI.DirectoryName))
			{
				try
				{
					SW = File.CreateText(siteInfoFile);
					try
					{
						//Save header
                        siteText = String.Format("{0}\t{1}\t{2}\t{3}\t{4}", "Site_ID", "Last_New", "Last_Updated", "Site_City", "Site_State");
						SW.WriteLine(siteText);

						//Site info for each site
						foreach (KeyValuePair<string, udSite> record in dicSites)
						{
							udsCurrent = record.Value;
							if (udsCurrent.siteStatus != status.Removed)
							{
                                siteText = String.Format("{0}\t{1}\t{2}\t{3}\t{4}", udsCurrent.ID, udsCurrent.lastNew, udsCurrent.lastUpdated, udsCurrent.City, udsCurrent.State);
								SW.WriteLine(siteText);
							}
						}
					}
					catch (Exception ex)
					{
						errMsg = "Unable to save site info (" + ex.Message + ")";
						fnLogEntry(logFile, errMsg, entryTypes.Error);
					}
					finally
					{
						try
						{
							SW.Flush();
							SW.Close();
						}
						catch (Exception ex)
						{
							fnLogEntry(logFile, ex.Message, entryTypes.Error);
						}
					}
				}
				catch (Exception ex)
				{
                    errMsg = "Unable to create new site info file " + siteInfoFile + " (" + ex.Message + ")";
					fnLogEntry (logFile, errMsg, entryTypes.Error);
				}
			}
		}

		static int fnDayDiff(DateTime startDate, DateTime endDate)
        {
			//*****************************************************************************
			//Puprose:      Calculates interval between 2 dates in days
			//Parameters:   startDate	- Start date
			//				endDate		- End date
			//Returns:      Number of days between 2 specified dates
			//Revision History:
			//08/23/11  ZPG Original creation
			//*****************************************************************************

			System.TimeSpan diffResult = endDate.Subtract(startDate);

            return diffResult.Days;
        }

        static string fnGetSiteCityOrState(string siteName, nameParts namePart)
        {
            //*****************************************************************************
            //Puprose:      Extracts name of the city from a VistA site name
            //Parameters:   siteName    - Name of the site
            //              namePart    - City or State
            //Returns:      Extracted name of the city 
            //Revision History:
            //09/02/11  ZPG Original creation
            //*****************************************************************************

            string ret = "";
            string[] l_arrName;
            char[] arrReservedChars = {'<'};
            
            l_arrName = siteName.Split(',');
            if (namePart == nameParts.City)
            {
                ret = l_arrName[0].Trim();
            }
            else
            {
                if (l_arrName.Count() < 2)
                {
                    ret = "";
                }
                else
                {
                    ret = l_arrName[l_arrName.Count() - 1].Trim();
                }
            }

            //Replace all reserved characters with spaces
            if (ret != "")
            {
                for (int i = 0; i < arrReservedChars.Count(); i++)
                {
                    ret = ret.Replace(arrReservedChars[i], ' ');
                }
                ret = ret.Trim();
            }

            return ret;
        }

        static string fnGetAppConfig()
		{
			//*****************************************************************************
			//Puprose:      Retrieves application settings and establishes application defaults
			//Parameters:   None
			//Returns:      Empty string on success, error message on failure
			//Revision History:
			//08/18/11  ZPG Original creation
			//*****************************************************************************

			string ret = "";

			//Default values
			string logFolderDefault = ".\\Logs\\";
			string dataFolderDefault = ".\\OncoTraX_Data\\";
			string scriptFolderDefault = AppDomain.CurrentDomain.BaseDirectory;
			if (scriptFolderDefault.EndsWith("\\") == false)
			{
				scriptFolderDefault = scriptFolderDefault + "\\";
			}

			try
			{
				//Get configuration file
				System.Configuration.Configuration config =
				  ConfigurationManager.OpenExeConfiguration(
						ConfigurationUserLevel.None);
				//Get the AppSettings section.
				NameValueCollection appSettings =
				   ConfigurationManager.AppSettings;

				//Get log folder location
				logFolder = appSettings.Get("LogFolder");
				if (logFolder == null || logFolder.Trim() == "")
				{
					//Save default location
					if (logFolder != null)
					{
						config.AppSettings.Settings.Remove("LogFolder");
					}
					config.AppSettings.Settings.Add("LogFolder", logFolderDefault);
					config.Save(ConfigurationSaveMode.Modified);

					//Use default location
					logFolder = logFolderDefault;
				}
				else if (logFolder.EndsWith("\\") == false)
				{
					logFolder = logFolder + "\\";
				}

				// Get data folder location
				dataFolder = appSettings.Get("DataFolder");
				if (dataFolder == null || dataFolder.Trim() == "")
				{
					//Save default location
					if (dataFolder != null)
					{
						config.AppSettings.Settings.Remove("DataFolder");
					}
					config.AppSettings.Settings.Add("DataFolder", dataFolderDefault);
					config.Save(ConfigurationSaveMode.Modified);

					//Use default location
					dataFolder = dataFolderDefault;
				}
				else if (dataFolder.EndsWith("\\") == false)
				{
					dataFolder = dataFolder + "\\";
				}

				// Get script folder location
				scriptFolder = appSettings.Get("ScriptFolder");
				if (scriptFolder == null || scriptFolder.Trim() == "")
				{
					//Save default location
					if (scriptFolder != null)
					{
						config.AppSettings.Settings.Remove("ScriptFolder");
					}
					config.AppSettings.Settings.Add("ScriptFolder", scriptFolderDefault);
					config.Save(ConfigurationSaveMode.Modified);

					//Use default location
					scriptFolder = scriptFolderDefault;
				}
				else if (scriptFolder.EndsWith("\\") == false)
				{
					scriptFolder = scriptFolder + "\\";
				}

                // Get site info file pointer
                siteInfoFile = appSettings.Get("SiteInfo");
                if (siteInfoFile == null || siteInfoFile.Trim() == "")
                {
                    ret = "Unable to find SiteInfo setting in application configuration";
                }

                // Get site table pointer
                siteTableFile = appSettings.Get("SiteTable");
                if (siteTableFile == null || siteTableFile.Trim() == "")
                {
                    ret = "Unable to find SiteTable setting in application configuration";
                }

			}
			catch (ConfigurationErrorsException ex)
			{
				ret = ex.Message;
			}

			return ret;
		}
		
		static bool fnGetSites()
        {
            //*****************************************************************************
            //Puprose:      Fills global Dictionary object dicSites with data
            //              from last saved state of sites and current listof sites
            //Parameters:   None
            //Returns:      True on success, False on failure
            //Revision History:
            //08/18/11  ZPG Original creation
            //*****************************************************************************

            string[] l_arrSite;
            udSite udsCurrent;
            bool ret;

            //Assume success
            ret = true;
            
            //Get current list of sites from VistA
            try
            {
                siteTbl = new SiteTable(siteTableFile);

                IList listVistA = siteTbl.Sites.GetKeyList();
                if (listVistA.Count == 0)
                {
                    errMsg = "List of VistA sites is empty";
					fnLogEntry (logFile, errMsg, entryTypes.Error);
                    Debug.Print(errMsg);
                    ret = false;
                }
                else //VistA site list is not empty
                {
                    //Get list of sites from site info file 
                    if (System.IO.File.Exists(siteInfoFile))
                    {
                        try 
                        {
                            System.IO.StreamReader siteInfo = new System.IO.StreamReader(siteInfoFile);
                            while (siteInfo.EndOfStream == false)
                            {
                                string siteCurrent = siteInfo.ReadLine ();
                                l_arrSite = siteCurrent.Split('\t');
                                if (l_arrSite.Count() >= 3)
                                {
                                    if (l_arrSite[0] == "Site_ID")
                                    {
                                        //header
                                    }
                                    else
                                    {
                                       //Build a dictionary of sites
                                        if (dicSites.ContainsKey(l_arrSite[0]) == false)
                                        {
                                            udsCurrent.ID = l_arrSite[0];
                                            if (listVistA.Contains(l_arrSite[0]))
                                            {
                                                udsCurrent.siteStatus = status.Present;
                                                udsCurrent.City = fnGetSiteCityOrState(siteTbl.getSite(l_arrSite[0]).Name, nameParts.City);
                                                udsCurrent.State = fnGetSiteCityOrState(siteTbl.getSite(l_arrSite[0]).Name, nameParts.State);
                                            }
                                            else
                                            {
                                                udsCurrent.siteStatus = status.Removed;
                                                udsCurrent.City = "";
                                                udsCurrent.State = "";
                                            }
                                            udsCurrent.lastNew = l_arrSite[1];
                                            udsCurrent.lastUpdated = l_arrSite[2];
                                            udsCurrent.newCount = "";
											udsCurrent.updatedCount = "";
                                            dicSites.Add(udsCurrent.ID, udsCurrent);
                                        }

                                        //Add new sites
                                        for (int i = 0; i < listVistA.Count; i++)
                                        {
                                            if (dicSites.ContainsKey(listVistA[i].ToString()) == false)
                                            {
                                                udsCurrent.ID = listVistA[i].ToString();
                                                udsCurrent.City = fnGetSiteCityOrState(siteTbl.getSite(listVistA[i].ToString()).Name, nameParts.City);
                                                udsCurrent.State = fnGetSiteCityOrState(siteTbl.getSite(listVistA[i].ToString()).Name, nameParts.State);
                                                udsCurrent.siteStatus = status.Added;
                                                udsCurrent.lastNew = "";
                                                udsCurrent.lastUpdated = "";
                                                udsCurrent.newCount = "";
												udsCurrent.updatedCount = "";
                                                if (dicSites.ContainsKey(udsCurrent.ID) == false)
                                                {
                                                    dicSites.Add(udsCurrent.ID, udsCurrent);
                                                }
                                            }
                                        }
                                    }
                                }
                                else
                                {
                                    ret = false;
                                    errMsg = "Malformed site info file " + siteInfoFile;
									fnLogEntry(logFile, errMsg, entryTypes.Error);
									Debug.Print(errMsg);
                                }
                            }
                            siteInfo.Close();
                        }
                        catch (Exception e) 
                        {
                            ret=false;
                            errMsg = "Unable to read site info file " + siteInfoFile + e.ToString();
							fnLogEntry(logFile, errMsg, entryTypes.Error);
							Debug.Print(errMsg);
                        }   
                    }
                    else //site info file not found
                    {
                       //Build a dictionary of sites
                        for (int i = 0; i < listVistA.Count; i++)
                        {
                            udsCurrent.ID = listVistA[i].ToString();
                            udsCurrent.City = "";
                            udsCurrent.State = "";
                            udsCurrent.siteStatus = status.Added;
                            udsCurrent.lastNew = "";
                            udsCurrent.lastUpdated = "";
                            udsCurrent.newCount = "";
							udsCurrent.updatedCount = "";
                            if (dicSites.ContainsKey(udsCurrent.ID) == false)
                            {
                                dicSites.Add(udsCurrent.ID, udsCurrent);
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ret=false;
                errMsg = "Unable to read list of VistA sites (" + ex.ToString() + ")";
				fnLogEntry(logFile, errMsg, entryTypes.Error);
				Debug.Print(errMsg);
            }

            return ret;

        }

        static bool fnGetAbstracts(string siteID)
        {
            //*****************************************************************************
            //Puprose:      Extracts new and updated OncoTraX abstracts for specified site
            //              Function uses dicSites global object to determine date interval
            //Parameters:   siteID  - Site ID
            //Returns:      True on success, False on failure 
            //Revision History:
            //08/19/11  ZPG Original creation
            //*****************************************************************************

            bool ret;
            udSite udsCurr;
            String startDateNew;
            String startDateUpdated;
            string strRet;
            string newCount = "";
            string updatedCount = "";

            procName = "fnGetAbstracts";

            //Assume success
            ret = true;

            //Extract site info from dicSites global object
            if (dicSites.TryGetValue(siteID, out udsCurr))
            {
                if (udsCurr.siteStatus != status.Removed)
                {
                    //Set start and end dates
                    if (udsCurr.siteStatus == status.Added)
                    {
                        //Retrieve data for yesterday
                        startDateNew = DateTime.Today.AddDays(-1).ToShortDateString();
                        startDateUpdated = DateTime.Today.AddDays(-1).ToShortDateString();
                    }
                    else //Status - Present
                    {
                        startDateNew = udsCurr.lastNew;
                        startDateUpdated = udsCurr.lastUpdated;
                    }
                    
                    //Extract new abstracts
                    strRet = fnExtractData(siteID, udsCurr.City, udsCurr.State, startDateNew, type.New, out newCount);
                    if (strRet == "")
                    {
                        udsCurr.newCount = newCount;
                        udsCurr.lastNew = DateTime.Today.ToShortDateString();
                        strRet = fnExtractData(siteID, udsCurr.City, udsCurr.State, startDateUpdated, type.Updated, out updatedCount);
                        if (strRet == "")
                        {
                            udsCurr.updatedCount = updatedCount;
                            udsCurr.lastUpdated = DateTime.Today.ToShortDateString();
                        }
                        else
                        {
                            ret = false;
							fnLogEntry(logFile, strRet, entryTypes.Error);
							Debug.Print(strRet);
                        }
                        dicSites.Remove(siteID);
                        dicSites.Add(siteID, udsCurr);
                    }
                    else
                    {
                        ret = false;
						fnLogEntry(logFile, strRet, entryTypes.Error);
						Debug.Print(strRet);
                    }

                }
            }
            else
            {
                ret = false;
                errMsg = procName + ": Requested site ID " + siteID + " is not present among available sites";
				fnLogEntry(logFile, errMsg, entryTypes.Error);
				Debug.Print(errMsg);
            }

            return ret;

        }

		static string fnLogEntry(string logFile, string entryText)
		{
			return fnFileEntry(logFile, entryText, entryTypes.Blank, true);
		}

		static string fnLogEntry(string logFile, string entryText, entryTypes entryType)
		{
			return fnFileEntry(logFile, entryText, entryType, false);
		}

		static string fnFileEntry(string logFile, string entryText, entryTypes entryType, bool supressDateTime)
		{
			//********************************************************************
			// Purpose	:	Creates an entry in the specified log file
			// Parameters:	logFile     - The name of the log file. The log will be
			//                          created if it doesn't exist. However, this 
			//                          function will not create nonexistent folders
			//              entryText   - Text that should be logged
			//              entryType	- Type of log entry that will be used as a 
			//                          error log entry marker, such as ERROR or WARNING
			// Returns:		sResult     - Empty string on success, error message on failure.
			//                          
			// Revision History:
			// 08/23/11 ZPG Original creation 
			//'*********************************************************************

			string sResult = "";
			StreamWriter SW;
			string strText = "";

			SW = null;

			//Open the log file
			if (File.Exists(logFile) == false)
			{
				try
				{
					SW = File.CreateText(logFile);
				}
				catch (Exception ex)
				{
					sResult = ex.Message;
				}
			}
			else
			{
				try
				{
					SW = File.AppendText(logFile);
				}
				catch (Exception ex)
				{
					sResult = ex.Message;
				}
			}

			if (SW != null)
			{
				//Add date/time stamp
				if (supressDateTime == false)
				{
					strText = DateTime.Now.ToString();
				}

				//Add entry type indicator
				if (entryType == entryTypes.Blank)
				{
					if (strText != "")
					{
						strText = strText + ": ";
					}
				}
				else
				{
					strText = strText + " [" + entryType.ToString().ToUpper() + "]: ";
				}

				//Output log entry to file
				try
				{
					SW.WriteLine(strText + entryText);
				}
				catch (Exception ex)
				{
					sResult = ex.Message;
				}
				finally
				{
					try
					{
						SW.Flush();
						SW.Close();
					}
					catch (Exception ex)
					{
						sResult = ex.Message;
					}
				}
			}

			return sResult;

		}

        static string fnExtractData(string siteID, string siteCity, string siteState, string startDate, type extractType, out string recCount)
        {
        //*****************************************************************************
        //Puprose:      Extracts OncoTraX abstracts (new or updated) for specified site
        //              and date range to an output file. 
        //              Function uses siteTbl and dicSites global objects 
        //Parameters:   siteID      - Site ID
        //              siteCity    - City where the site is located  
        //              siteState   - State where the site is located  
        //              startDate   - Start date (end date is always yesterday)
        //              extractType - Type of abstracts (New vs. Updated)
        //              recCount    - Record count (output parameter)
        //Returns:      Empty string on success, error message on failure 
        //Revision History:
        //08/19/11  ZPG Original creation
        //*****************************************************************************

            string ret = "";
            string strType;
            int dayDiff;
            string endDate;
            int queryType;
            udSite udsCurrent;
            DateTime dtStart;
            DateTime dtEnd;
            String saveFile;

            //Initialize
            recCount = "";

            //Validate parameters
            if (extractType == type.New)
            {
                strType = "NEW";
            }
            else
            {
                strType = "UPDATED";
            }
                //Site ID
            if (dicSites.TryGetValue(siteID, out udsCurrent) == false)
            {
                ret = "Requested site ID " + siteID + " is not present among available sites";
				Debug.Print(ret);
            }

            if (ret == "")
            {
                //Start date
                if (DateTime.TryParse(startDate, out dtStart) == false)
                {
                    ret = "Invalid start date " + startDate + " specified for retrieval of " + strType +
                          " abstracts from site ID " + siteID;
                    Debug.Print(ret);
                }
                else
                {
                    dayDiff = fnDayDiff(dtStart, DateTime.Today);
                    if (dayDiff == 0)
                    {
                        recCount = "-1";
                    }
                    else if (dayDiff < 0)
                    {
                        ret = "Future date " + startDate + " cannot be used as start date for retrieval of "
                              + strType + " abstracts from site ID " + siteID;
                        Debug.Print(ret);
                    }
                }
            }

            if ((ret == "") && (DateTime.TryParse(startDate, out dtStart)) && (recCount != "-1"))
            {
                //End date is always yesterday
                endDate = DateTime.Today.AddDays(-1).ToShortDateString();
                dtEnd = DateTime.Today.AddDays(-1);

                try
                {
                    //Use requested site as data source
                    Site visitSite = (Site)siteTbl.Sites[siteID];
                    DataSource src = visitSite.getDataSourceByModality("HIS");

                    //Use generic user account to access VistA
                    User user;
                    user = new User();
                    //user.Name = new PersonName("DEPARTMENT OF DEFENSE,USER");
                    user.Name = new PersonName("ONCOLOGY,RPC USER");
                    //user.LogonSiteId = new KeyValuePair<string, string>("200", "DoD");
                    user.LogonSiteId = new KeyValuePair<string, string>(siteID, "OncoTrax");
                    //user.LogonSiteUid = "31066";
                    user.LogonSiteUid = "98765";
                    //user.SSN = new SocSecNum("123456789");
                    user.SSN = new SocSecNum("");
                    user.PermissionString = "ONC RPC BROKER";

                    //Establish VistA connection
                    ConnectionApi cxnApi = new ConnectionApi(src);
                    cxnApi.connect();
                    UserApi userApi = new UserApi();
                    //string duz = userApi.visit(cxnApi.MdoConnection, user, user.PermissionString, true);

                    VistaConnection cxn;
                    cxn = (VistaConnection)cxnApi.MdoConnection;
                    try
                    {
                        //                  ACCESS CODE  VERIFY CODE   "PERMISSION"
                        // Original before Verify Change Below               userApi.login(cxn, "ONCO/456", "ONCO,789", "ONC RPC BROKER");
                        userApi.login(cxn, "ONCO/456", "TRAX/123", "ONC RPC BROKER");
                    }
                    catch (Exception e1)
                    {
                        if (e1.Message == "VERIFY CODE must be changed before continued use.")
                            userApi.setNewPassword(cxn, "ONCO,789", "TRAX/123");
                        else if (e1.Message == "Security Error: The remote procedure ORWU USERINFO is not registered to the option ONC RPC BROKER. (This message has come directly from IP          ).")
                        { //Non Fatal Error 
                        }
                        else
                        { // fatal error of some sort}
                            ret = "Unable to extract data from site ID " + siteID + " (" + e1.Message + ")";
                            Debug.Print(ret);
                        }
                    }

                    if (cxn.IsConnected == true)
                    {
                        RegistriesApi api = new RegistriesApi();
                        queryType = (int)extractType;
                        OncologyExtract t = api.getOncologyExtract(cxn, startDate, endDate, queryType.ToString());
                        recCount = t.Text.Count().ToString();
                        if (t.Text.Count() != 0)
                        {
                            //Construct output file name
                            saveFile = dataFolder + siteID + "_";
                            if (siteCity.Trim() != "")
                            {
                                saveFile = saveFile + siteCity.Trim().Replace(' ', '_') + "_";
                            }
                            if (siteState.Trim() != "")
                            {
                                saveFile = saveFile + siteState.Trim().Replace(' ', '_') + "_";
                            }
                            saveFile = saveFile + strType.Substring(0, 1) + "_" + dtStart.ToString("yyMMdd") + "_" + dtEnd.ToString("yyMMdd");

                            //Write out retrieved records
                            StreamWriter writer = new StreamWriter(saveFile);
                            for (int cnt = 0; cnt < t.Text.Count(); cnt++)
                            {
                                writer.WriteLine(t.Text[cnt]);
                                Console.WriteLine(t.Text[cnt].Length);
                            }
                            writer.Flush();
                            writer.Close();
                        }
                        cxnApi.disconnect();
                    }
                    else
                    {
                        ret = "Unable to establish connection to site " + siteID;
						Debug.Print(ret);
                    }
                }
                catch (Exception ex)
                {
                    ret = "Unable to extract data from site ID " + siteID + " (" + ex.Message + ")";
					Debug.Print(ret);
                }
            }

            return ret;
        }

		static string fnReportSite(string siteID)
		{
		//*****************************************************************************
		//Puprose:      Generates report text for specified site 
		//              Function uses siteTbl and dicSites global objects 
		//Parameters:   siteID      - site ID
		//Returns:      Report text  
		//Revision History:
		//08/23/11  ZPG Original creation
		//*****************************************************************************

			string retReport;
            udSite udsCurr;

            //Extract site info from dicSites global object
            retReport = "\tSite " + siteID + ":" + "\t";
			if (dicSites.TryGetValue(siteID, out udsCurr))
			{
				if (udsCurr.siteStatus == status.Removed)
				{
					retReport = retReport + "No longer available";
				}
				else
				{
					//New abstract count
					retReport = retReport + "New - ";
					if (udsCurr.newCount == "")
					{
						retReport = retReport + "Unable to retrieve on " + DateTime.Today.ToShortDateString();
					}
                    else if (udsCurr.newCount == "-1")
                    {
                        retReport = retReport + "Already retrieved on " + DateTime.Today.ToShortDateString();
                    }
                    else
                    {
                        retReport = retReport + udsCurr.newCount + " extracted on ";
                        retReport = retReport + DateTime.Today.ToShortDateString();
                    }

					//Updated abstract count
					retReport = retReport + "\t" + "Updated - ";
					if (udsCurr.updatedCount == "")
					{
						retReport = retReport + "Unable to retrieve on " + DateTime.Today.ToShortDateString();
					}
                    else if (udsCurr.updatedCount == "-1")
                    {
                        retReport = retReport + "Already retrieved on " + DateTime.Today.ToShortDateString();
                    }
                    else
					{
                        retReport = retReport + udsCurr.updatedCount + " extracted on ";
						retReport = retReport + DateTime.Today.ToShortDateString();
					}

                    //Add site location info
                    if (udsCurr.City == "")
                    {
                        retReport = retReport + "\t(No city info, ";
                    }
                    else
                    {
                        retReport = retReport + "\t(" + udsCurr.City + ", ";
                    }
                    if (udsCurr.State == "")
                    {
                        retReport = retReport + "No state info)";
                    }
                    else
                    {
                        retReport = retReport + "\t(" + udsCurr.City + ", ";
                        retReport = retReport + udsCurr.State + ")";
                    }
                }
			}
			else //Should never happen
			{
				retReport = retReport + "No site information";
			}

			return retReport;

		}

        static void Main(string[] args)
        {
			String fileEntry;

			//Retrieve application settings
			errMsg = fnGetAppConfig();

			//Ensure that log folder exists
			if (Directory.Exists(logFolder) == false)
			{
				try 
				{
					Directory.CreateDirectory(logFolder);
				}
				catch
				{
					//If missing log folder couldn't be established, create logs in the folder containing current app
					logFolder = Path.GetDirectoryName(Assembly.GetAssembly(typeof(Program)).CodeBase);
					if (logFolder.EndsWith("\\") == false)
					{
						logFolder = logFolder + "\\";
					}
				}
			}

			//Construct error log file name
			logFile = logFolder + constLogPrefix + "_" + DateTime.Now.ToString("yyMMdd") + ".log";

			//Log configuration error if there was a problem with reading application settings
            if (errMsg != "")
            {
                fnLogEntry(logFile, errMsg, entryTypes.Error);
            }
            else
            {
			    //Construct execution report file name
			    rptFile = logFolder + constReportPrefix + "_" + DateTime.Now.ToString("yyMM") + ".log";

                if (fnGetSites())
                {
                    List<string> listSites = new List<string>(dicSites.Keys);

				    //Extract abstracts
                    foreach (string siteID in listSites)
                    {
					    fnGetAbstracts(siteID);
                    }

				    //Generate execution report
				    fnLogEntry(rptFile, "", entryTypes.Info);
				    foreach (string siteID in listSites)
				    {
					    fileEntry = fnReportSite(siteID);
					    fnLogEntry(rptFile, fileEntry);
				    }

				    //Update site info file
				    fnSaveSiteInfo();

				    //Execute data transformation script
				    errMsg = fnLaunchVBScript(scriptFolder + constScriptName, false);
				    if (errMsg != "")
				    {
					    errMsg = "Error executing VBScript " + scriptFolder + constScriptName + " (" + errMsg + ")";
					    fnLogEntry(logFile, errMsg, entryTypes.Error);
				    }
                }
            }
        }
    }
}
