/*******************************************************************************/
/*  Package:      ONC  Oncology                                               */
/*  Date Created: Jul 26,2004                                                  */
/*  Site Name:    Hines OIFO                                                   */
/*  Developers:   Sergey Gavrilov PII                     )                    */
/*  Description:  Main include file for Oncology Web-service                   */
/*******************************************************************************/

#ifndef _ONCOLOGYCGI_H
#define _ONCOLOGYCGI_H

#define ONCVERSION	"2.00P"
#define INI_FILE_NAME "oncsrv.ini"
#define OUTPUT_PATH_CONFIG_VALUE_START_TAG "<OUTPUT_PATH>"
#define OUTPUT_PATH_CONFIG_VALUE_END_TAG "</OUTPUT_PATH>"
#define DEBUG_CONFIG_VALUE_START_TAG "<DEBUG>"
#define DEBUG_CONFIG_VALUE_END_TAG "</DEBUG>"
#define VERSION_CONFIG_VALUE_START_TAG "<VERSION>"
#define VERSION_CONFIG_VALUE_END_TAG "</VERSION>"

/*******************************************************************************/
/*                            FORWARD DECLARATIONS                             */
/*******************************************************************************/

class TONCRequest;

/*******************************************************************************/
/*                                 PARAMETERS                                  */
/*******************************************************************************/

// Size of the XML parser buffer
const int BUFF_SIZE	= 32000;	

// Oncology namespace
const char ONCNAMESPACE[] = "http://URL       ";

/*******************************************************************************/
/*                       ERROR CODES (see ONC_Fault.cpp)                       */
/*-----------------------------------------------------------------------------*/
/*                     !!! DO NOT MODIFY THESE VALUES !!!                      */
/*******************************************************************************/

const int EC_UNKNOWN_METHOD		=  -2;
const int EC_UNSUP_METHOD		=  -3;
const int EC_UNKNOWN_CONTENT	=  -4;
const int EC_UNSUP_CONTENT		=  -5;
const int EC_XMLBUF_ALLOC		=  -6;
const int EC_BAD_XML			=  -7;
const int EC_INVALID_REQUEST	=  -8;
const int EC_INVALID_TABNUM		=  -9;
const int EC_SITE_HIST			= -10;
const int EC_CALC_WARNINGS		= -11;
const int EC_CALC_ERRORS		= -12;
const int EC_SERVER_EXCEPTION	= -13;
const int EC_NOT_ENOUGH_MEM		= -14;
const int EC_EDITS_ERROR		= -15;
const int EC_EDITS_ERROR_EX		= -16;
const int EC_INVALID_EDITSCFG	= -17;
const int EC_NEED_DISCRIMINATOR = -18;
const int EC_FOLDER_NOT_EXISTS  = -19;
const int EC_ERROR_WRITING_OUTPUT  = -20;
const int EC_ENVIRONMENT_VARIABLE_NOT_FOUND  = -21;

/*******************************************************************************/
/*                   REQUEST TAG CODES (see ONC_Parser.cpp)                    */
/*******************************************************************************/

enum TONCTagCode {
	//--- General tags
	UNKNOWN_TAG = 0, SOAP_ENVELOPE, SOAP_BODY, GET_VERSION,
	//--- CS_CALCULATE
	CS_CALCULATE, CSC_AGE, CSC_BEHAV, CSC_EXT, CSC_EXTEVAL, CSC_GRADE, CSC_HIST,
	CSC_LNEXAM,	CSC_LNPOS, CSC_METS, CSC_METSEVAL, CSC_NODES, CSC_NODESEVAL, 
	CSC_SITE, CSC_SIZE, CSC_SSF1, CSC_SSF2, CSC_SSF3, CSC_SSF4, CSC_SSF5, CSC_SSF6,
// JJB 100105 ->
	CSC_SSF7, CSC_SSF8, CSC_SSF9, CSC_SSF10, CSC_SSF11, CSC_SSF12,
	CSC_SSF13, CSC_SSF14, CSC_SSF15, CSC_SSF16, CSC_SSF17, CSC_SSF18,
	CSC_SSF19, CSC_SSF20, CSC_SSF21, CSC_SSF22, CSC_SSF23, CSC_SSF24, CSC_SSF25,
	CSC_DXYEAR, CSC_VERORIG, /*CSC_SEX,*/ CSC_LVI,
// JJB 100105 <-
	//--- CS_GET_SCHEMA
	CS_GET_SCHEMA, CSGS_HIST, CSGS_SITE, CSGS_DISCRIMINATOR,
	//--- CS_GET_TABLES
	CS_GET_TABLES, CSGT_SCHEMA, CSGT_TABLE,
	//--- ED_GET_EDITINFO
	ED_GET_EDITINFO, EDEI_EDIT, EDEI_EDITSET, EDEI_TEXTWIDTH,
	//--- ED_RUN_BATCH
	ED_RUN_BATCH, EDRB_NAACCR_RECORD
};

/*******************************************************************************/
/*                  GENERIC EXCEPTION/ERROR DESCRIPTOR OBJECT                  */
/*******************************************************************************/

class TONCFault {
	int		ErrorCode;				// Oncology web-service error code (see above)
	char	FaultCode[20];			// SOAP error code (see ONC_Fault.cpp)
	char	FaultString[200];		// Error message (see ONC_Fault.cpp)

protected:

	/*****	The outputDetail() function writes optional error details to the 
			stdout in XML format.

		The output is automatically enclosed in the <detail> and </detail> tags.
		Descendant classes can override this function to add more details in 
		addition to the error code, which is writted by the TONCFault.
	*/
	virtual void outputDetail();

public:
	TONCFault();
	TONCFault(const int anErrorCode, ...);
	virtual ~TONCFault() {};


	/*****	The output() function writes information about the error (the 
			<soap:Fault> ... </soap:Fault> section) to the stdout in XML format.
	*/
	void output();

};

/*******************************************************************************/
/*                                  TEXT BUFFER                                */
/*******************************************************************************/

class TONCTextBuffer {
	char*	Buffer;					// Pointer to the buffer
	size_t	BufferLength;			// Buffer length (maximum length of the text)
	size_t	TextLength;				// Current length of the text

public:
	TONCTextBuffer();
	virtual ~TONCTextBuffer();


	/*****	The append() function appends the string to the buffer.

		text		Appended text

		len			Append no more that this number of characters. If this
					parameter is omitted or less than 0, then the strlen function
					is used internally to get its value.

	*/
	void append(const XML_Char* text, int txtlen = -1);


	/*****	The clear() function clears the buffer.
	*/
	void clear();


	/*****	The copyText() function copies the text value of the current tag to 
			the provided buffer.

		dst		Pointer to the destination buffer

		maxlen	No more than this number of characters are copied.

		append0	If this parameter is true, then the function always appends '\0'
				after copied characters.  Otherwise, the function behaves exaclty
				as the strncpy.
	*/
	void copyText(char* dst, size_t maxlen, bool append0 = 0);


	/*****	The getText() function returns the pointer to the text buffer.
	*/
	const char* getAsText() { return Buffer; };


	/*****	The getAsInt() function converts the text value of the current tag
			into integer and returns the result.
	*/
	int getAsInt() { return atoi(Buffer); };


	/*****	The getAsChar() function returns the first character of the text value
			of the current tag.
	*/
	char getAsChar() { return Buffer[0]; };

};

/*******************************************************************************/
/*                                  XML PATH                                   */
/*******************************************************************************/

class TONCXMLPath {
	size_t	PathLength;				// Current length of the path
	size_t	BufferLength;			// Buffer length (maximum length of the path)
	char	*Buffer;				// Pointer to the XML path buffer

public:
	TONCXMLPath();
	virtual ~TONCXMLPath();


	/*****	The clear() function clears the path buffer.
	*/
	void clear();


	/*****	The getCode() function returns the code that correesponds to the current 
			path stored in the buffer.
	*/
	TONCTagCode getCode();


	/*****	The getLastTag() function returns name of the current tag (the last 
			one appended to the buffer).
	*/
	const XML_Char* getLastTag();


	/*****	The getPath() function returns the XML path stored in the path buffer.
	*/
	const XML_Char* getPath() { return Buffer; };


	/*****	The popTag() procedure removes the current tag (the last one appended)
			from the buffer.
	*/
	void popTag();


	/*****	The pushTag() procedure appends the tag name to the XML path stored in
			the path buffer.

		tag_name	Tag name (in Expat format)

	*/
	void pushTag(const XML_Char* tag_name);

};

/*******************************************************************************/
/*                              PARAMETER PARSER                               */
/*******************************************************************************/

class TONCParser {
	TONCRequest*	Request;		// Pointer to the current request

public:
	TONCXMLPath		Path;			// XML path
	TONCTextBuffer	TagText;		// Text of the current tag

	TONCParser();


	/*****  Get the Debug file name  *****/
	char* TONCParser::GetDubugFileName();


	/*****	Get the current date time to a string and format it as YYMMDDHHMMSS.
	*/
	char* TONCParser::GetCurrentDateTime();

	/*****	Get a string from a file
	*/
	char* TONCParser::GetStringFromFile(char* fileName);

	/*****	Get a string from an environment variable ****/
	char* TONCParser::GetStringEnvironmentVariable(char* variableName);

	/*****	Get a configuration value from the INI file ****/
	char* TONCParser::GetConfigValue(char* iniFileName, char* configKeyStartTag, char* configKeyEndTag);


	/*****	Get a file name with full path
	*/
	char* TONCParser::GetFileNameWithPath(char* fileName, bool debugFolder, char* outputSubFolderName);

	/*******************************************************************************/
	/* Write the medical data to an output file                                    */
	/*******************************************************************************/
	void TONCParser::WriteOutputFile(char* medicalData);

	/*****	The createRequest() function creates the request object that 
			corresponds to the provided code and assignes the pointer to the 
			Request property.

		TagCode		Code of the request tag (see TONCXMLPath::getTagCode for
					more details)

		atts		List of attributes and their values provided by the Expat
					XML parser

		If another request has been created before, then it is destroyed and the
		new one is created.
	*/
	TONCRequest* createRequest(const TONCTagCode TagCode, const XML_Char **atts);


	/*****	The getAttrVal() function returns text value of the attribute.

		atts		List of attributes and their values provided by the Expat
					XML parser

		name		Attribute name

		If the requested attribute is not present in the list, then an empty
		string is returned.
	*/
	static const XML_Char* getAttrVal(const XML_Char **atts, const char* name);


	/*****	The getRequest() function returns pointer to the current request
			object.
	*/
	TONCRequest* getRequest() { return Request; };


	/*****	The parseParameters() function reads the XML request from the stdin, 
			parses it, creates the corresponding request object, and returns the
			pointer.

		content_length	Length of the request body.  If this parameter is not 
						greater than 0, then the functions reads from standard 
						input stream until the 'end of file' is reached.
	*/
	TONCRequest* parseParameters(int content_length, char* debugFileName);

};

/*******************************************************************************/
/*                              GENERIC REQUEST                                */
/*******************************************************************************/

class TONCRequest {
	TONCTagCode Code;				// Code of the current request (tag code)
	TONCFault*	Fault;				// Pointer to the error descriptor
	double		Version;			// Version of the current request (optional)

protected:

	/*****	The execute() function processes the request and writes the XML
			response to the stdout.  Descendant classes must override this
			function.
	*/
	virtual void execute();


	/*****	The getVersionAttr() function returns the value of the optional ver
			attribute (version of the reques).  If it is not set by the client,
			then 0 is returned.
	*/
	static double getVersionAttr(const XML_Char **atts);


	/*****	The responseTag() function outputs opening and closing tags of the
			response to the stdout.

		open_tag		If value of this parameter is 'true', then the function
						writes the opening tag (e.g. <CS-RESPONSE>). Otherwise,
						the closing tag is written (e.g. </CS-RESPONSE>).

	*/
	virtual void responseTag(bool open_tag);

public:
	TONCRequest(const TONCTagCode aCode);
	virtual ~TONCRequest();


	/*****	The createRequest() function creates the request object that
			corresponds to the request tag code and returns the pointer.

		TagCode		Code of the request tag (see TONCXMLPath::getTagCode for
					more details)

		atts		List of attributes and their values provided by the Expat
					XML parser
	*/
	static TONCRequest* createRequest(TONCParser *Parser, const TONCTagCode TagCode, const XML_Char **atts);


	/*****	The executeRequest() function processes the request (see the execute()
			function for details) and encloses the output in the response tags
			(see the responseTag() function for details).
	*/
	void executeRequest();


	/*****	The getCode() function returns code of the request.
	*/
	int getCode() { return Code; };


	/*****	The getNamespace() function returns the oncology namespace.
	*/
	const char* getNamespace() { return ONCNAMESPACE; };


	/*****	The getVersion() function returns version of the request.
	*/
	double getVersion() { return Version; };


	/*****	The parseTagEnd() function is called by the XML parser (see the
			TONCParser for details) when a closing tag is parsed.

		Parser		Pointer to the parser object

		TagCode		Code of the tag (see TONCXMLPath::getTagCode for more details)
	*/
	virtual bool parseTagEnd(TONCParser* Parser, const TONCTagCode TagCode);


	/*****	The parseTagStart() function is called by the XML parser (see the
			TONCParser for details) when an opening tag is parsed.

		Parser		Pointer to the parser object

		TagCode		Code of the tag (see TONCXMLPath::getTagCode for more details)
	*/
	virtual bool parseTagStart(TONCParser* Parser, const TONCTagCode TagCode, const XML_Char **atts);


	/*****	The processErrors() function checks if an error descriptor has been
			created (see the Fault property) and outputs the Fault section of
			the response to the stdout.
	*/
	void processErrors();


	/*****	The setFault() function sets the value of the Fault property.

			If the property already has a value, then the old object is destroyed
			before the new pointer is assigned.

		Fault		Pointer to an error descriptor.  The request object becomes
					the owner of the descriptor and is responsible for its proper
					destruction.
	*/
	void setFault(TONCFault* aFault);

};

/*******************************************************************************/
/*                            FUNCTION DECLARATIONS                            */
/*******************************************************************************/

/*****	The getAppPath() function returns the drive and path (including the
		trailing separator) from where the Oncology application was started
		(e.g. "c:\iis\web\cs\cgi-bin\").
*/
const char* getAppPath();

/*****	The IntTag() function encloses the integer value in opening and closing
		tags and outputs it to the stdout (e.g. <RC>-12</RC>).

	name		Name of the tags

	val			Integer value
*/
void IntTag(const char *name, int val);


/*****	The Piece() function breaks the source string into pieces according to
		the provided delimiter and returns requested part(s).  It works in the
		same way as the $PIECE function in M (MUMPS).

	Value		Source string

	Delimiter	Delimiter string

	StartPiece	Number of the first extracted piece.  If value of this parameter
				is omitted, then the extraction starts from the beginning of the
				string.

	EndPiece	Number of the last extracted piece. If value of this parameter
				is omitted, then the single piece defined by the StartPiece
				parameter is extracted.

	If the StartPiece is not greater than 0 or the EndPiece is less than the
	StartPiece, then an empty string is returned.

*/
string Piece(string Value, string Delimiter, const int StartPiece = 1);
string Piece(string Value, string Delimiter, const int StartPiece, const int EndPiece);


/*****	The StrTag() function encloses the string in opening and closing tags and
		outputs it to the stdout (e.g. <EDIT-NAME>Age at Diagnosis</EDIT-NAME>).

	name		Name of the tag

	val			Pointer to a null-terminated string
*/
void StrTag(const char *name, const char *val);


/*****	The UnitTests() function performs unit tests of different parts of the
		Oncology CGI application and writes results to the stdout.

	If code passess all tests, this function returns EXIT_SUCCESS. Otherwise,
	EXIT_FAILURE is returned.
*/
int UnitTests();


/*****	The writeStructuredText() function encodes special characters ('<', '&',
		'\"', '\'', and '>') according to the XML rules, encloses the lines of
		text into the <P> and </P> tags, wraps the lines that are longer than
		240 characters, and writes them to the stdout.

	text		Pointer to a null-terminated string containing the source text.
*/
void writeStructuredText(const char* text);


/*****	The writeText() function encodes special characters ('<', '&', '\"',
		'\'', and '>') according to the XML rules, wraps the lines that are
		longer than 240 characters, and writes them to the stdout.  The original
		structure of the text is not preserved.

	text		Pointer to a null-terminated string containing the source text.

	crlf		If value of this parameter is 'true', then the function starts
				a new line after writing the text.  By default, this does not
				happen.
*/
void writeText(const char* text, bool crlf = false);


/*****	The xmlEncode() function encodes special characters ('<', '&', '\"',
		'\'', and '>') according to the XML rules and returns the result.

	str			Source string
*/
string xmlEncode(string str);

#endif