/*******************************************************************************/
/*  Package:      ONC  Oncology                                               */
/*  Date Created: Jul 26,2004                                                  */
/*  Site Name:    Hines OIFO                                                   */
/*  Developers:   Sergey Gavrilov PII                     )                    */
/*  Description:  This unit contains code that performs unit tests of          */
/*                different parts of the Oncology CGI application.             */
/*******************************************************************************/

#include "stdafx.h"
#include "OncologyCGI.h"
#include <memory>
#include <conio.h>

#include "ONC_Requests.h"

#ifdef _DEBUG

//***** PROMPTS THE USER AND WAITS FOR EITHER 'Y' OR 'N' ANSWER

bool askYesNo(const char* prompt = "<default>")
{
	char ch;
	if( strcmp(prompt, "<default>") != 0 )
		printf("\n%s", prompt);
	else
		printf("\nAsnwer \'Y\' or \'N\': ");

	do
	{
		ch = toupper(_getch());
	} 
	while( (ch != 'Y') && (ch != 'N') );
	printf("%c\n", ch);

	return (ch == 'Y');
}

//***** PRINTS NAME OF THE TEST AND THE RESULT

bool printTest(const char* header, const bool result)
{
	printf("\n%-70s  %s", header, result ? "Passed" : "Failed");
	return result;
}

//***** TESTS THE EXCEPTION OBJECT

bool testTONCFault()
{
	puts("\n\nTesting custom exception object...\n");

	try
	{
		throw TONCFault(EC_INVALID_EDITSCFG, "DEBUG");
	}
	catch(TONCFault Fault)
	{
		Fault.output();
	}

	puts("\nDoes the XML fragment above look like this:\n");

	puts("<soap:Fault>");
	puts("<faultcode>Client</faultcode>");
	puts("<faultstring>Invalid EDITS configuration name: \'DEBUG\'.</faultstring>");
	puts("<detail>");
	puts("<RC>-17</RC>");
	puts("</detail>");
	puts("</soap:Fault>");

	if( !askYesNo() )
		return false;

	return true;
}

//***** TESTS THE VERSION REQUEST OBJECT

bool testTONCReqGetVersion()
{
	puts("\n\nTesting version request object...\n");

	auto_ptr<TONCReqGetVersion> req(new TONCReqGetVersion(GET_VERSION));
	req->executeRequest();

	puts("\nDoes the XML fragment above look like this:\n");

	puts("<RESPONSE xmlns=\"http://URL       \">");
	puts("<CS-APIVER>010300</CS-APIVER>");
	puts("<ED-METAVER></ED-METAVER>");
	puts("<VERSION>2.00</VERSION>");
	puts("</RESPONSE>");

	if( !askYesNo() )
		return false;

	return true;
}

//***** TESTS THE BUFFER OBJECT THAT IS USED BY THE PARAMETER PARSER

bool testTONCTextBuffer()
{
	auto_ptr<TONCTextBuffer> buffer(new TONCTextBuffer);
	string ctrlbuf = "";
	int i;

	buffer->append("123");
	if( buffer->getAsInt() != 123 )
		return false;

	buffer->clear();

	//--- 8190 characters (buffer is almost full)
	for(i=0; i<819; i++) 
	{
		buffer->append("1234567890", 10);
		ctrlbuf.append("1234567890");
	}
	if( buffer->getAsText() != ctrlbuf )
		return false;

	//--- 32000 characters (initial buffer size)
	buffer->append("12");
	ctrlbuf.append("12");
	if( buffer->getAsText() != ctrlbuf )
		return false;

	//--- 8193 characters (buffer has been expanded)
	buffer->append("3");
	ctrlbuf.append("3");
	if( buffer->getAsText() != ctrlbuf)
		return false;

	//--- First character
	if( buffer->getAsChar() != ctrlbuf[0] )
		return false;

	return true;
}

//***** TESTS THE XML PATH OBJECT THAT IS USED BY THE PARAMETER PARSER

bool testTONCXMLPath()
{
	auto_ptr<TONCXMLPath> path(new TONCXMLPath);
	string ctrlbuf = "";
	int i;

	//--- 256 characters (initial buffer size)
	for(i=0; i<32; i++)
	{
		path->pushTag("^test-tag");
		ctrlbuf.append("/test-tag");
	}
	if( path->getPath() != ctrlbuf )
		return false;

	//--- Buffer has been expanded; empty tag name.
	path->pushTag("");
	ctrlbuf.append("/<unknown>");
	if( path->getPath() != ctrlbuf )
		return false;

	//--- Tag codes; push & pop
	path->clear();

	path->pushTag("^Envelope");
	path->pushTag("^Body");
	path->pushTag("^CS-GET-TABLES");
	if( path->getCode() != CS_GET_TABLES )
		return false;

	path->popTag();
	if( path->getCode() != SOAP_BODY )
		return false;

	path->popTag();
	if( path->getCode() != SOAP_ENVELOPE )
		return false;

	//--- Unknown tag; last tag name
	path->pushTag("^CS-GET-TABLES");
	if( strcmp(path->getLastTag(), "CS-GET-TABLES") != 0 )
		return false;
	if( path->getCode() != 0 )
		return false;

	return true;
}

//***** PERFORMS UNIT TESTS

int UnitTests()
{
	int rc = EXIT_FAILURE, errcnt = 0;

	try
	{
		//--- TONCFault
		if( !printTest("TONCFault", testTONCFault()) )
			errcnt++;

		//--- TONCReqGetVersion
		if( !printTest("TONCReqGetVersion", testTONCReqGetVersion()) )
			errcnt++;
		

		//--- TONCTextBuffer
		if( !printTest("TONCTextBuffer", testTONCTextBuffer()) )
			errcnt++;

		//--- TONCXMLPath
		if( !printTest("TONCXMLPath", testTONCXMLPath()) )
			errcnt++;

		//--- Success
		if( errcnt == 0 )
			rc = EXIT_SUCCESS;
	}
	//--- Process exceptions explicitly generated by the Oncology code
	catch(TONCFault Fault)
	{
		Fault.output();
	}

	puts("\n");
	//--- Check for memory leaks
	if( _CrtDumpMemoryLeaks() )
	{
		puts("Memory leak(s) detected!");
		rc = EXIT_FAILURE;
	}
	//--- Print the conclusion
	if( rc == EXIT_SUCCESS )
		puts("Code passed all unit tests.");
	else
		puts("Code did not pass unit test(s).");
	getchar();

	return rc;
}

#endif