/*******************************************************************************/
/*  Package:      ONC  Oncology                                               */
/*  Date Created: Jul 26,2004                                                  */
/*  Site Name:    Hines OIFO                                                   */
/*  Developers:   Sergey Gavrilov PII                     )                    */
/*  Description:  Declarations of request processing classes for Oncology      */
/*                Web-service                                                  */
/*******************************************************************************/

#ifndef _ONCREQUESTS_H
#define _ONCREQUESTS_H

/*******************************************************************************/
/*                            GENERAL REQUESTS                                 */
/*******************************************************************************/

class TONCReqGetVersion: public TONCRequest {
private:
	string EDITSVersion;			// Versiuon of the default EDITS metafile

protected:
	virtual void execute();

public:
	TONCReqGetVersion(const TONCTagCode aCode): TONCRequest(aCode) {};

	virtual bool parseTagStart(TONCParser* Parser, const TONCTagCode TagCode, const XML_Char **atts);

};

/*******************************************************************************/
/*                      COLLABORATIVE STAGING REQUESTS                         */
/*******************************************************************************/

class TONCReqCS: public TONCRequest {
protected:
	char	Site[6];				// Primary site
	char	Hist[5];				// Histology
	int		SchemaNumber;			// Schema number
	int		TableNumber;			// Table number
	char	Discriminator[4];		// Discriminator

	virtual void responseTag(bool open_tag);

public:
	TONCReqCS(const TONCTagCode aCode);

	virtual bool parseTagEnd(TONCParser* Parser, const TONCTagCode TagCode);

};

class TONCReqCSCalculate: public TONCReqCS {
protected:
	datacard dc;					// CS data card

	virtual void execute();

public:
	TONCReqCSCalculate(const TONCTagCode aCode);

	virtual bool parseTagEnd(TONCParser* Parser, const TONCTagCode TagCode);

};

class TONCReqCSGetSchema: public TONCReqCS {

protected:
	virtual void execute();

public:
	TONCReqCSGetSchema(const TONCTagCode aCode): TONCReqCS(aCode) {};

};

class TONCReqCSGetTables: public TONCReqCS {
public:
	TONCReqCSGetTables(const TONCTagCode aCode): TONCReqCS(aCode) {};

	virtual void execute();

};

/*******************************************************************************/
/*                              EDITS REQUESTS                                 */
/*******************************************************************************/

typedef struct {
	const char*	Name;				// Name of the edit
	int			Options;			// Edit options
} TONCEditCfg;

typedef struct {
	const char*	Name;				// Name of the edit set
	int			Option;				// Parameter for RunBatch
} TONCEditSetCfg;

typedef struct {
	const char*	Name;				// Standard name of the field
	int			Options;			// Field options
} TONCFieldCfg;

typedef struct  {
	const char*		Name;			// Name of the configuration
	char*			Metafile;		// Name of the metafile
	unsigned		RecordLength;	// Length of the NAACCR record
	TONCEditSetCfg	EditSets[10];	// List of edit set options
	TONCEditCfg		Edits[25];		// List of edit options
	TONCFieldCfg	Fields[10];		// List of field options
} TONCEDITSConfig;

class TONCReqED: public TONCRequest {
	string	Metafile;				// Full name of the EDITS metafile
	string	MetafileVersion;		// Version of the metafile

protected:
	const TONCEDITSConfig* Config;	// Pointer to the EDITS configuration descriptor (static)

	/*****	The getMetafile() function returns the pointer to the null-terminated
			string that stores the full name (drive, path, name, and extension) 
			of the EDITS metafile.
	*/
	const char* getMetafile() { return Metafile.c_str(); };


	virtual void responseTag(bool open_tag);

public:
	TONCReqED(const TONCTagCode aCode);
	virtual ~TONCReqED();


	/*****	The getMetafileVersion() function returns the pointer to the null-
			terminated string value of the MetafileVersion property (version of
			the EDITS metafile).
	*/
	const char* getMetafileVersion() { return MetafileVersion.c_str(); };

	virtual bool parseTagStart(TONCParser* Parser, const TONCTagCode TagCode, const XML_Char **atts);

	/*****	The GetVersion() function returns the version number oncsrv.ini file.
	*/
	char* GetMetaFileVersion();
};

class TONCReqEDGetEditInfo: public TONCReqED {
	int		EditIndex;				// Index of the requested edit in the edit set
	string	EditSetName;			// Name of the edit set
	int		TextWidth;				// Maximum width for formatting the description and help text

protected:
	virtual void execute();

public:
	TONCReqEDGetEditInfo(const TONCTagCode aCode);

	virtual bool parseTagEnd(TONCParser* Parser, const TONCTagCode TagCode);

};

class TONCReqEDRunBatch: public TONCReqED {
	int		RecordHandle;			// Handle of internal record buffer

protected:

	virtual void execute();


	/*****	The executeEditSet() function performs the validations defined by the
			edit set on the NAACCR record stored in the Record buffer.

		name		Name of the edit set (pointer to a null-terminated string).
	*/
	void executeEditSet(const char* name, const int option = 2);


	/*****	The outputMessages() function analyzes results of the EE_RunBatch()
			function, generates the corresponding XML response, and outputs it
			to the stdout.

		setHandle	Handle of the edit set (see the EDITS API documentation for
					details).

		setName		Name of the edit set (pointer to a null-terminated string).
	*/
	void outputMessages(EEt_mh setHandle, const char* setName);


	/*****	The setEditOptions() function passes edit options from the
			current configuration to the EDITS engine.
	*/
	void setEditOptions(EEt_mh EditSetHandle);


	/*****	The setFieldOptions() function passes field options from the
			current configuration to the EDITS engine.
	*/
	void setFieldOptions(void);

	/*****	The GetFileLayoutName() function returns the file layout name.  
			The version number is from oncsrv.ini file.
	*/
	char* GetMetaFileLayoutName();


public:
	TONCReqEDRunBatch(const TONCTagCode aCode);

	virtual bool parseTagEnd(TONCParser* Parser, const TONCTagCode TagCode);

};

#endif