/*******************************************************************************/
/*  Package:      ONC  Oncology                                               */
/*  Date Created: Jul 26,2004                                                  */
/*  Site Name:    Hines OIFO                                                   */
/*  Developers:   Sergey Gavrilov PII                     )                    */
/*  Description:  Implementation of EDITS request processing classes for       */
/*                Oncology Web-service                                         */
/*******************************************************************************/

#include "stdafx.h"
#include "OncologyCGI.h"
#include "ONC_Requests.h"

/*******************************************************************************/
/*                          CONFIGURATION MANAGEMENT                           */
/*-----------------------------------------------------------------------------*/
/*         !!! CONFIGURATIONS MUST BE LISTED IN ALPHABETICAL ORDER !!!         */
/*******************************************************************************/

static const TONCEDITSConfig Configs[] = {
	{//	Name			Metafile			Record Length
		"DEBUG",		"METAFILE.RMF",		6694,
		{//		Edit Set Name											Options
			{	"Veterans_Administration",								2				},
			{	NULL	}
		},
		{//		Edit Name												Options
			{	"Behavior Code ICDO3, Seq Num--Central (SEER IF114)",	EE_SUPPRESS		},
			{	"Behavior ICDO3, Site, Histology ICDO3 (NAACCR)",		EE_SUPPRESS		},
			{	"Cancer Status (COC)",									EE_SKIPEMPTY	},
			{	"Cause of Death (SEER COD)",							EE_SKIPEMPTY	},
			{	"Cervix In Situ ICDO2 (SEER IF88)",						EE_SUPPRESS		},
			{	"Cervix In Situ ICDO3 (SEER IF88)",						EE_SUPPRESS		},
			{	"Date of Last Contact, Date of Diag. (NAACCR IF19)",	EE_SKIPEMPTY	},
			{	"Date of Last Contact (NAACCR DATEEDIT)",				EE_SKIPEMPTY	},
			{	"Diagnostic Confirm, Seq Num--Central (SEER IF23)",		EE_SUPPRESS		},
			{	"Follow-Up Source, Date of Diagnosis (COC)",			EE_SKIPEMPTY	},
			{	"Follow-up source Central (NAACCR)",					EE_SUPPRESS		},
			{	"ICD Revision Number, Cause of Death (SEER IF37)",		EE_SKIPEMPTY	},
			{	"Next Follow-Up Source, Date of Diagnosis (COC)",		EE_SKIPEMPTY	},
			{	"PIN III ICDO3, Date of Diagnosis (SEER IF110)",		EE_SUPPRESS		},
			{	"RX Date--Rad Ended, Date Last Contact (COC)",			EE_SKIPEMPTY	},
			{	"RX Date--Surgical Disch, Date Last Contact (COC)",		EE_SKIPEMPTY	},
			{	"RX Date--Systemic, Date Last Contact (COC)",			EE_SKIPEMPTY	},
			{	"Seq Num--Central, Prim Site, Morph ICDO3(SEER IF22",	EE_SUPPRESS		},
			{	"Type of Rep Srce(DC),Seq Num--Cent,ICDO2(SEER IF04",	EE_SUPPRESS		},
			{	"Type of Rep Srce(DC),Seq Num--Cent,ICDO3(SEER IF04",	EE_SUPPRESS		},
			{	"Vital Status (COC)",									EE_SKIPEMPTY	},
			{	NULL	}
		},
		{//		Field Name												Options
			{	"Social Security Number",								EE_SUPPRESS		},
			{	"Follow-Up Source Central",								EE_SUPPRESS		},
			{	NULL	}
		}
	},
	{//	Name			Metafile			Record Length
		"DEFAULT",		"METAFILE.RMF",		6694,
		{//		Edit Set Name											Options
			{	"Veterans_Administration",								2				},
			{	NULL	}
		},
		{//		Edit Name												Options
			{	"Behavior Code ICDO3, Seq Num--Central (SEER IF114)",	EE_SUPPRESS		},
			{	"Behavior ICDO3, Site, Histology ICDO3 (NAACCR)",		EE_SUPPRESS		},
			{	"Cancer Status (COC)",									EE_SKIPEMPTY	},
			{	"Cause of Death (SEER COD)",							EE_SKIPEMPTY	},
			{	"Cervix In Situ ICDO2 (SEER IF88)",						EE_SUPPRESS		},
			{	"Cervix In Situ ICDO3 (SEER IF88)",						EE_SUPPRESS		},
			{	"Date of Last Contact, Date of Diag. (NAACCR IF19)",	EE_SKIPEMPTY	},
			{	"Date of Last Contact (NAACCR DATEEDIT)",				EE_SKIPEMPTY	},
			{	"Diagnostic Confirm, Seq Num--Central (SEER IF23)",		EE_SUPPRESS		},
			{	"Follow-Up Source, Date of Diagnosis (COC)",			EE_SKIPEMPTY	},
			{	"Follow-up source Central (NAACCR)",					EE_SUPPRESS		},
			{	"ICD Revision Number, Cause of Death (SEER IF37)",		EE_SKIPEMPTY	},
			{	"Next Follow-Up Source, Date of Diagnosis (COC)",		EE_SKIPEMPTY	},
			{	"PIN III ICDO3, Date of Diagnosis (SEER IF110)",		EE_SUPPRESS		},
			{	"RX Date--Rad Ended, Date Last Contact (COC)",			EE_SKIPEMPTY	},
			{	"RX Date--Surgical Disch, Date Last Contact (COC)",		EE_SKIPEMPTY	},
			{	"RX Date--Systemic, Date Last Contact (COC)",			EE_SKIPEMPTY	},
			{	"Seq Num--Central, Prim Site, Morph ICDO3(SEER IF22",	EE_SUPPRESS		},
			{	"Type of Rep Srce(DC),Seq Num--Cent,ICDO2(SEER IF04",	EE_SUPPRESS		},
			{	"Type of Rep Srce(DC),Seq Num--Cent,ICDO3(SEER IF04",	EE_SUPPRESS		},
			{	"Vital Status (COC)",									EE_SKIPEMPTY	},
			{	NULL	}
		},
		{//		Field Name												Options
			{	"Follow-Up Source Central",								EE_SUPPRESS		},
			{	NULL	}
		}
	},
	{//	Name			Metafile			Record Length
		"FULL",			"METAFILE.RMF",		6694,
		{//		Edit Set Name											Options
			{	"Veterans_Administration",								2				},
			{	NULL	}
		},
		{//		Edit Name												Options
			{	NULL	}
		},
		{//		Field Name												Options
			{	"Follow-Up Source Central",								EE_SUPPRESS		},
			{	NULL	}
		}
	},
	{//	Name			Metafile			Record Length
		"NCDB11B",		"NCD_110B.RMF",		6694,
		{//		Edit Set Name											Option
			{	"*",													2				},
			{	NULL	}
		},
		{//		Edit Name												Options
			{	NULL	}
		},
		{//		Field Name												Option
			{	NULL	}
		}
	}
};

static int compare_configs( const void *arg1, const void *arg2 )
{
	return _stricmp( ((TONCEDITSConfig*)arg1)->Name, ((TONCEDITSConfig*)arg2)->Name );
}

static TONCEDITSConfig* findEDITSConfig(const char* ConfigName)
{
	TONCEDITSConfig key = { ConfigName };
	return (TONCEDITSConfig*)bsearch(&key, Configs, sizeof(Configs)/sizeof(Configs[0]), 
		sizeof(Configs[0]), compare_configs);
}

/*******************************************************************************/
/*                           GENERIC EDITS REQUEST                             */
/*******************************************************************************/

TONCReqED::TONCReqED(const TONCTagCode aCode): TONCRequest(aCode)
{
	Config = NULL;
	Metafile = "";
	MetafileVersion = "";
}

TONCReqED::~TONCReqED()
{
	EE_Exit();
	TONCRequest::~TONCRequest();
}

//*****	PROCESSES THE OPENING TAG

bool TONCReqED::parseTagStart(TONCParser* Parser, const TONCTagCode TagCode, const XML_Char **atts)
{
	bool prc = TONCRequest::parseTagStart(Parser, TagCode, atts);

	if( TagCode == getCode() )
	{
		//--- Get the name of the configuration
		const XML_Char* cfgname = TONCParser::getAttrVal(atts, "edits-config");
		if( strlen(cfgname) <= 0 )
			cfgname = "DEFAULT";

		//--- Find the configuration descriptor
		Config = findEDITSConfig(cfgname);
		if( Config == NULL )
			throw TONCFault(EC_INVALID_EDITSCFG, cfgname);

		//--- Construct the name of the EDITS metafile
		Metafile = getAppPath();
		Metafile += Config->Metafile;

		//--- Load the metafile
		char version[80];
		int rc = EE_Init((char*)getMetafile(), version);
		if( rc != 0 )
			throw TONCFault(EC_EDITS_ERROR, rc, "EE_Init");

		//--- Get the version
		char* metaVersion = GetMetaFileVersion();
		if( (metaVersion != NULL) && (strlen(metaVersion) > 0) )
			MetafileVersion = metaVersion;
		else
			MetafileVersion = version;

		return true;
	}

	return prc;
}

//***** RENDERS THE ROOT TAG OF THE RESPONSE

void TONCReqED::responseTag(bool open_tag)
{
	if( open_tag )
		printf("<ED-RESPONSE xmlns=\"%s\">\n", getNamespace());
	else
		puts("</ED-RESPONSE>");
}

/*******************************************************************************/
/*                             EDITS INFO REQUEST                              */
/*******************************************************************************/

TONCReqEDGetEditInfo::TONCReqEDGetEditInfo(const TONCTagCode aCode): TONCReqED(aCode)
{
	EditIndex   = 0;
	EditSetName = "";
	TextWidth	= 75;
}

//*****	PROCESSES THE REQUEST

void TONCReqEDGetEditInfo::execute()
{
	const int BUFSIZE = 32768;
	EEt_mh esHandle = 0;
	int retLen, status;

	char* buf = new char[BUFSIZE];
	if( buf == NULL )
		throw TONCFault(EC_NOT_ENOUGH_MEM);
	buf[0] = '\0';

	try
	{
		//--- Load the requested edit set
		int rc = EE_LoadEditSet(&esHandle, (char*)EditSetName.c_str(), &status, 1);
		if( rc != 0 )
			throw TONCFault(EC_EDITS_ERROR_EX, rc, "EE_LoadEditSet", "Edit Set name", EditSetName.c_str());

		//--- Get the edit name
		rc = EE_GetEditName(esHandle, EditIndex, buf);
		if( rc != 0 )
			throw TONCFault(EC_EDITS_ERROR, rc, "EE_GetEditName");
		StrTag("NAME", buf);

		//--- Get the edit description text
		if( !EE_GetEditDescript(esHandle, EditIndex, TextWidth, BUFSIZE, &retLen, buf, 1) )
		{
			puts("<DESCRIPTION>");
			writeStructuredText(buf);
			puts("</DESCRIPTION>");
		}

		//--- Get the edit help text
		if( !EE_GetEditHelp(esHandle, EditIndex, TextWidth, BUFSIZE, &retLen, buf, 1) )
		{
			puts("<HELP>");
			writeStructuredText(buf);
			puts("</HELP>");
		}
	}
	catch(...)
	{
		if( esHandle != 0 )
			EE_FreeEditSet(esHandle);
		delete [] buf;
		throw;
	}

	//---- Free the edit set
	EE_FreeEditSet(esHandle);
	delete [] buf;
}

//*****	PROCESSES THE CLOSING TAG

bool TONCReqEDGetEditInfo::parseTagEnd(TONCParser* Parser, const TONCTagCode TagCode)
{
	int tmp;
	if( !TONCReqED::parseTagEnd(Parser, TagCode) )
		switch( TagCode )
		{
		case EDEI_EDIT:
			EditIndex = Parser->TagText.getAsInt();
			break;
		case EDEI_EDITSET:
			EditSetName = Parser->TagText.getAsText();
			break;
		case EDEI_TEXTWIDTH:
			tmp =  Parser->TagText.getAsInt();
			if( tmp > 0 )
				TextWidth = tmp;
			break;
		default:
			return false;
		}
	return true;
}

/*******************************************************************************/
/*                          EDITS 'RUN BATCH' REQUEST                          */
/*******************************************************************************/

TONCReqEDRunBatch::TONCReqEDRunBatch(const TONCTagCode aCode): TONCReqED(aCode)
{
	RecordHandle = -1;
}

//*****	PROCESSES THE REQUEST

void TONCReqEDRunBatch::execute()
{
	if( Config->EditSets[0].Name != NULL )
	{
		char buf[80];
		int esi, rc;

		//--- If the first name in the list is "*", execute all edit sets from the metafile
		if( !strcmp(Config->EditSets[0].Name, "*") )
		{
			int i, nes;
			rc = EE_GetEditSetCount(&nes);
			if( rc != 0 )
				throw TONCFault(EC_EDITS_ERROR, rc, "EE_GetEditSetCount");
			for( i=1; i<=nes; i++)
			{
				rc = EE_GetEditSetName(i, buf);
				if( rc != 0 )
					throw TONCFault(EC_EDITS_ERROR, rc, "EE_GetEditSetName");
				executeEditSet(buf);
			}
		}
		//--- Otherwise, execute the edits sets from the list
		else
			for(esi=0; Config->EditSets[esi].Name != NULL; esi++ )
				executeEditSet(Config->EditSets[esi].Name, Config->EditSets[esi].Option);

		StrTag("VERSION", ONCVERSION);
		StrTag("METAVER", getMetafileVersion());
	}
}

//*****	EXECUTES THE SINGLE EDIT SET FROM THE LIST

void TONCReqEDRunBatch::executeEditSet(const char* name, const int option)
{
	int rc, status;
	EEt_mh setHandle;

	//--- Load the edit set
	rc = EE_LoadEditSet(&setHandle, (char*)name, &status, 1);
	if( rc != 0 )
		throw TONCFault(EC_EDITS_ERROR_EX, rc, "EE_LoadEditSet", "Edit Set name", name);
	try
	{
		setFieldOptions();
		setEditOptions(setHandle);

		//--- Clear the internal EDITS flags
		rc = EE_ClearAllFlags();
		if( rc != 0 )
			throw TONCFault(EC_EDITS_ERROR, rc, "EE_ClearAllFlags");

		//--- Run all edits from the set
		rc = EE_RunBatch(setHandle, option, &status);
		if( rc != 0 )
			throw TONCFault(EC_EDITS_ERROR_EX, rc, "EE_RunBatch", "Edit Set name", name);

		//--- If errors are returned, add them to the response
		if( status > 0 )
			outputMessages(setHandle, name);
	}
	catch(...)
	{
		EE_FreeEditSet(setHandle);
		throw;
	}
	//--- Free the memory
	EE_FreeEditSet(setHandle);
}

//*****	FORMATS THE MESSAGES GENERATED BY THE EDIT SET AND WRITES THEM TO THE STDOUT

void TONCReqEDRunBatch::outputMessages(EEt_mh setHandle, const char* setName)
{
	int mc, prev_edit_index = -1, rc;
	extern int GlobalInputDataHasError;

	//--- Get the number of messages
	rc = EE_GetMessageCount(&mc);
	if( rc != 0 )
		throw TONCFault(EC_EDITS_ERROR, rc, "EE_GetMessageCount");

	//--- Write the messages to the stdout im XML format
	if( mc > 0 )
	{
		//--- Group the messages under the edit set
		printf("<EDIT-SET NAME=\"%s\" ECNT=\"%d\">\n", xmlEncode(setName).c_str(), mc);
		try
		{
			const int BUFSIZE = 160;
			int pos, edit_index, ifld, imsg, num_fields, tmp;
			char edit_name[60], msg_type[5], msg_text[BUFSIZE], field_name[BUFSIZE], buf[BUFSIZE];
			long err_code, num_err, num_warn, num_skip;

			for( imsg=1; imsg<=mc; imsg++ )
			{
				//--- Get the message data
				rc = EE_GetMessageTextEx(msg_type, imsg, msg_text, BUFSIZE, &err_code, &tmp, &edit_index, edit_name, &tmp);
				if( rc != 0 )
					throw TONCFault(EC_EDITS_ERROR, rc, "EE_GetMessageTextEx");

				//--- Group the messages by edits
				if( edit_index != prev_edit_index )
				{
					//--- Edit info
					rc = EE_GetEditCounts(setHandle, edit_index, &num_fields, &num_err, &num_warn, &num_skip);
					if( rc != 0 )
						throw TONCFault(EC_EDITS_ERROR, rc, "EE_GetEditCounts");

					if( prev_edit_index >= 0 )
						puts("</EDIT>");
					printf("<EDIT INDEX=\"%d\" NAME=\"%s\" ECNT=\"%ld\" WCNT=\"%ld\">\n", 
						edit_index, xmlEncode(edit_name).c_str(), num_err, num_warn);
					prev_edit_index = edit_index;

					//--- Fields validated by the edit
					for(ifld=1; ifld<=num_fields; ifld++)
					{
						//--- Start position of the field
//JJB 100105 the sprintf's became _s versions
						sprintf_s(buf, sizeof(buf), "%%C%d", ifld);
						EE_GetEditInfo(setHandle, edit_index, buf, buf, BUFSIZE);
						pos = atoi(buf+1);
						//--- Field name
//JJB 100105
						sprintf_s(buf, sizeof(buf), "%%F%d", ifld);
						EE_GetEditInfo(setHandle, edit_index, buf, field_name, BUFSIZE);
						//--- Value of the field
//JJB 100105
						sprintf_s(buf, sizeof(buf), "%%V%d", ifld);
						EE_GetEditInfo(setHandle, edit_index, buf, buf, BUFSIZE);
						//--- Write the tags
						printf("<FLD NAME=\"%s\" POS=\"%d\">%s</FLD>\n", 
							xmlEncode(field_name).c_str(), pos, xmlEncode(buf).c_str());

						GlobalInputDataHasError++;
					}
				}

				//--- Message
				printf("<MSG CODE=\"%d\" TYPE=\"%s\">", err_code, msg_type);
				writeText(msg_text);
				puts("</MSG>");
				GlobalInputDataHasError++;
			}
		}
		catch(...)
		{
			if( prev_edit_index >= 0 )
				puts("</EDIT>");
			puts("</EDIT-SET>");
			throw;
		}
		if( prev_edit_index >= 0 )
			puts("</EDIT>");
		puts("</EDIT-SET>");
	}
}

//*****	PROCESSES THE CLOSING TAG

bool TONCReqEDRunBatch::parseTagEnd(TONCParser* Parser, const TONCTagCode TagCode)
{
	int rc;
	char* layoutNameWithVersion;

	if( !TONCReqED::parseTagEnd(Parser, TagCode) )
		switch( TagCode )
		{
		case EDRB_NAACCR_RECORD:
			//--- Allocate internal buffer for the NAACCR record
			rc = EE_AllocRecord(Config->RecordLength, &RecordHandle);
			if( rc != 0 )
				throw TONCFault(EC_EDITS_ERROR, rc, "EE_AllocRecord");

			//--- Load and initialize the record layout
			layoutNameWithVersion = GetMetaFileLayoutName();

			rc = EE_InitRecordLayout(RecordHandle, layoutNameWithVersion);
			if( rc != 0 )
				throw TONCFault(EC_EDITS_ERROR_EX, rc, "EE_InitRecordLayout", "Layout name", layoutNameWithVersion);

			//--- Load the data into the internal buffer
			char* data;
			data = (char*) Parser->TagText.getAsText();
			rc = EE_UpdateRecord(RecordHandle, data);

			//--- Write the data into output file
			try
			{
				Parser->WriteOutputFile(data);
			}
			catch(TONCFault Fault)
			{
				Fault.output();
			}

			if( rc != 0 )
				throw TONCFault(EC_EDITS_ERROR, rc, "EE_UpdateRecord");

			break;
		default:
			return false;
		}
	return true;
}


//***** PASSES EDIT OPTIONS FROM THE CONFIGURATION TO THE EDITS ENGINE

void TONCReqEDRunBatch::setEditOptions(EEt_mh EditSetHandle)
{
	int eh, ei, eopts, rc;
	for(ei=0; Config->Edits[ei].Name != NULL; ei++)
	{
		//--- Get edit handle
		rc = EE_GetEditHandle(EditSetHandle, &eh, (char*)(Config->Edits[ei].Name));
		if( rc != 0 )
			throw TONCFault(EC_EDITS_ERROR_EX, rc, "EE_GetEditHandle", "Edit name", Config->Edits[ei].Name);

		//--- Set edit options
		if( eh >= 0 )
		{
			eopts = Config->Edits[ei].Options;
			rc = EE_EditOptions(EditSetHandle, eh, &eopts);
			if( rc != 0 )
				throw TONCFault(EC_EDITS_ERROR_EX, rc, "EE_EditOptions", "Edit name", Config->Edits[ei].Name);
		}
	}
}

//***** PASSES FIELD OPTIONS FROM THE CONFIGURATION TO THE EDITS ENGINE

void TONCReqEDRunBatch::setFieldOptions(void)
{
	int fh, fi, fopts, rc;
	char* layoutNameWithVersion;
			
	layoutNameWithVersion = GetMetaFileLayoutName();
	for(fi=0; Config->Fields[fi].Name != NULL; fi++)
	{
		//--- Get field handle
		fh = EE_FieldHandle((char*)(Config->Fields[fi].Name), "S", layoutNameWithVersion);
		if( fh < 0 )
			throw TONCFault(EC_EDITS_ERROR_EX, fh, "EE_FieldHandle", "Field name", Config->Fields[fi].Name);

		//--- Set field options
		fopts = Config->Fields[fi].Options;
		rc = EE_FieldOptions(fh, &fopts);
		if( rc != 0 )
			throw TONCFault(EC_EDITS_ERROR_EX, rc, "EE_FieldOptions", "Field name", Config->Fields[fi].Name);
	}
}

char* TONCReqEDRunBatch::GetMetaFileLayoutName()
{
	char* layoutName = (char*) malloc(30);
	TONCParser* Parser;

	strcpy(layoutName, "NAACCR DATA EXCHANGE VS ");
	strcat(layoutName, Parser->GetConfigValue(INI_FILE_NAME, VERSION_CONFIG_VALUE_START_TAG, VERSION_CONFIG_VALUE_END_TAG));
	layoutName[(int) strlen(layoutName)] = 0;
	
	return layoutName;
}


char* TONCReqED::GetMetaFileVersion()
{
	TONCParser* Parser;
	
	return Parser->GetConfigValue(INI_FILE_NAME, VERSION_CONFIG_VALUE_START_TAG, VERSION_CONFIG_VALUE_END_TAG);
}
