/*******************************************************************************/
/*  Package:      ONC  Oncology                                               */
/*  Date Created: Jul 26,2004                                                  */
/*  Site Name:    Hines OIFO                                                   */
/*  Developers:   Sergey Gavrilov PII                     )                    */
/*  Description:  Implementation of Collaborative Staging request processing   */
/*                classes for Oncology Web-service                             */
/*******************************************************************************/

#include "stdafx.h"
#include "OncologyCGI.h"
#include "ONC_Requests.h"

/*******************************************************************************/
/*               ERROR DESCRIPTOR FOR THE CS 'CALCULATE' REQUEST               */
/*******************************************************************************/

class TONCFaultCSCalculate: public TONCFault {
	unsigned long	CSError;
	char*			Msg;

protected:
	virtual void	outputDetail();

public:
	TONCFaultCSCalculate(const int anErrorCode, const unsigned long aCSError, const char* aMsg);
	virtual ~TONCFaultCSCalculate();

};

//*****	CONSTRUCTOR

TONCFaultCSCalculate::TONCFaultCSCalculate(const int anErrorCode, const unsigned long aCSError, const char* aMsg): 
	TONCFault(anErrorCode)
{
	CSError = aCSError;
	Msg = ( aMsg != NULL && strlen(aMsg) > 0 ) ? _strdup(aMsg) : NULL;
}

//*****	DESTRUCTOR

TONCFaultCSCalculate::~TONCFaultCSCalculate()
{
	free(Msg);
	TONCFault::~TONCFault();
}

//*****	OUTPUTS DETAILED ERROR INFO

void TONCFaultCSCalculate::outputDetail()
{
	TONCFault::outputDetail();
	if( CSError != 0 )
		printf("<ERROR>%lu</ERROR>\n", CSError);
	if( Msg != NULL && strlen(Msg) > 0 )
		printf("<MSG>\n%s</MSG>\n", Msg);
}

/*******************************************************************************/
/*                   GENERIC COLLABORATIVE STAGING REQUEST                     */
/*******************************************************************************/

TONCReqCS::TONCReqCS(const TONCTagCode aCode): TONCRequest(aCode)
{
	Hist[0] = '\0';
	Site[0] = '\0';
	SchemaNumber = 0;
	TableNumber = 0;
	Discriminator[0] = '\0';
}

//*****	PROCESSES THE CLOSING TAG

bool TONCReqCS::parseTagEnd(TONCParser* Parser, const TONCTagCode TagCode)
{
	if( !TONCRequest::parseTagEnd(Parser, TagCode) )
		switch( TagCode )
		{
		case CSC_SITE:
		case CSGS_SITE:
//JJB 100106 4 became 5
			Parser->TagText.copyText(Site, 5, 1);
			break;
		case CSC_HIST:
		case CSGS_HIST:
			Parser->TagText.copyText(Hist, 4, 1);
			break;
//JJB 100115
		case CSGS_DISCRIMINATOR:
			Parser->TagText.copyText(Discriminator, 3, 1);
			break;
		case CSGT_SCHEMA:
			SchemaNumber = Parser->TagText.getAsInt();
			break;
		case CSGT_TABLE:
			TableNumber = Parser->TagText.getAsInt();
			break;
		default:
			return false;
		}

	return true;
}

//***** RENDERS THE ROOT TAG OF THE RESPONSE

void TONCReqCS::responseTag(bool open_tag)
{
	if( open_tag )
		printf("<CS-RESPONSE xmlns=\"%s\">\n", getNamespace());

	else
		puts("</CS-RESPONSE>");
}

/*******************************************************************************/
/*                 COLLABORATIVE STAGING 'CALCULATE' REQUEST                   */
/*******************************************************************************/

TONCReqCSCalculate::TONCReqCSCalculate(const TONCTagCode aCode): TONCReqCS(aCode)
{
	memset((char *)&dc, 0, sizeof(dc));
	CStage_clear_datacard(&dc);
	dc.error = 0;
	dc.messages[0] = '\0';
}

//*****	PROCESSES THE REQUEST

void TONCReqCSCalculate::execute()
{
	int rc = CStage_calculate(&dc);
	if( strlen(dc.messages) > 0 || rc < 0 )
		setFault(new TONCFaultCSCalculate(rc < 0 ? EC_CALC_ERRORS : EC_CALC_WARNINGS, dc.error, dc.messages));
/*
	         ;--- SOAP RESPONSE FROM THE COLLABORATIVE STAGING WEB SERVICE
         ;
         ; <?xml version="1.0" encoding="utf-8"?>
         ; <soap:Envelope
         ;   xmlns:soap="http://www.w3.org/2001/12/soap-envelope"
         ;   soap:encodingStyle="http://www.w3.org/2001/12/soap-encoding">
         ;   <soap:Body>
         ;     <CS-RESPONSE xmlns="http://DNS/oncology">
         ;       <CS-STOR>
         ;         <T> ... </T>
         ;         <TDESCR> ... </TDESCR>
         ;         <N> ... </N>
         ;         <NDESCR> ... </NDESCR>
         ;         <M> ... </M>
         ;         <MDESCR> ... </MDESCR>
         ;         <AJCC> ... </AJCC>
         ;         <AJCC7-T> ... </AJCC7-T>
         ;         <AJCC7-TDESCR> ... </AJCC7-TDESCR>
         ;         <AJCC7-N> ... </AJCC7-N>
         ;         <AJCC7-NDESCR> ... </AJCC7-NDESCR>
         ;         <AJCC7-M> ... </AJCC7-M>
         ;         <AJCC7-MDESCR> ... </AJCC7-MDESCR>
         ;         <AJCC7-STAGE> ... </AJCC7-STAGE>
         ;         <SS1977> ... </SS1977>
         ;         <SS2000> ... </SS2000>
         ;       </CS-STOR>
         ;       <CS-DISP>
         ;         <T> ... </T>
         ;         <TDESCR> ... </TDESCR>
         ;         <N> ... </N>
         ;         <NDESCR> ... </NDESCR>
         ;         <M> ... </M>
         ;         <MDESCR> ... </MDESCR>
         ;         <AJCC> ... </AJCC>
         ;         <AJCC7-T> ... </AJCC7-T>
         ;         <AJCC7-TDESCR> ... </AJCC7-TDESCR>
         ;         <AJCC7-N> ... </AJCC7-N>
         ;         <AJCC7-NDESCR> ... </AJCC7-NDESCR>
         ;         <AJCC7-M> ... </AJCC7-M>
         ;         <AJCC7-MDESCR> ... </AJCC7-MDESCR>
         ;         <AJCC7-STAGE> ... </AJCC7-STAGE>
         ;         <SS1977> ... </SS1977>
         ;         <SS2000> ... </SS2000>
         ;       </CS-DISP>
         ;       <APIVER> ... </APIVER>
         ;       <VERSION> ... </VERSION>
         ;     </CS-RESPONSE>
         ;     <soap:Fault>
         ;       <faultcode> ... </faultcode>
         ;       <faultstring> ... </faultstring>
         ;       <detail>
         ;         <ERROR> ... </ERROR>
         ;         <MSG>
         ;         ...
         ;         </MSG>
         ;         <RC> ... </RC>
         ;       </detail>
         ;     </soap:Fault>
         ;   </soap:Body >
         ; </soap:Envelope>

*/
	//--- Store Values
	puts("<CS-STOR>");
	StrTag("T",			dc.stor_t);
	StrTag("TDESCR",	dc.stor_tdescr);
	StrTag("N",			dc.stor_n);
	StrTag("NDESCR",	dc.stor_ndescr);
	StrTag("M",			dc.stor_m);
	StrTag("MDESCR",	dc.stor_mdescr);
    StrTag("AJCC",			dc.stor_ajcc);
	StrTag("AJCC7-T",		dc.stor_ajcc7_t);
	StrTag("AJCC7-TDESCR",	dc.stor_ajcc7_tdescr);
	StrTag("AJCC7-N",		dc.stor_ajcc7_n);
	StrTag("AJCC7-NDESCR",	dc.stor_ajcc7_ndescr);
	StrTag("AJCC7-M",		dc.stor_ajcc7_m);
	StrTag("AJCC7-MDESCR",	dc.stor_ajcc7_mdescr);
	StrTag("AJCC7-STAGE",	dc.stor_ajcc7_stage);
	StrTag("SS1977",		dc.stor_ss77);
	StrTag("SS2000",		dc.stor_ss2000);
	puts("</CS-STOR>");

	//--- Display Values
	puts("<CS-DISP>");
	StrTag("T",			dc.t);
	StrTag("TDESCR",	dc.tdescr);
	StrTag("N",			dc.n);
	StrTag("NDESCR",	dc.ndescr);
	StrTag("M",			dc.m);
	StrTag("MDESCR",	dc.mdescr);
	StrTag("AJCC",			dc.ajcc);
	StrTag("AJCC7-T",		dc.ajcc7_t);
	StrTag("AJCC7-TDESCR",	dc.ajcc7_tdescr);
	StrTag("AJCC7-N",		dc.ajcc7_n);
	StrTag("AJCC7-NDESCR",	dc.ajcc7_ndescr);
	StrTag("AJCC7-M",		dc.ajcc7_m);
	StrTag("AJCC7-MDESCR",	dc.ajcc7_mdescr);
	StrTag("AJCC7-STAGE",	dc.ajcc7_stage);
	StrTag("SS1977",	dc.ss77);
	StrTag("SS2000",	dc.ss2000);
	puts("</CS-DISP>");

	//--- Versions
	StrTag("APIVER", CStage_get_version());
	StrTag("VERSION", ONCVERSION);
}

//*****	PROCESSES THE CLOSING TAG

bool TONCReqCSCalculate::parseTagEnd(TONCParser* Parser, const TONCTagCode TagCode)
{
 	bool rc = TONCReqCS::parseTagEnd(Parser, TagCode);

	switch( TagCode )
	{
//JJB 100106 4 became 5
	case CSC_SITE:
		Parser->TagText.copyText(dc.site, 5);
		break;
	case CSC_HIST:
		Parser->TagText.copyText(dc.hist, 4);
		break;
	case CSC_BEHAV:
		dc.behav = Parser->TagText.getAsChar();
		break;
	case CSC_GRADE:
		dc.grade = Parser->TagText.getAsChar();
		break;
	case CSC_AGE:
		Parser->TagText.copyText(dc.age, 3);
		break;
	case CSC_SIZE:
		Parser->TagText.copyText(dc.size, 3);
		break;
	case CSC_EXT:
//JJB 100105 2 became 3
		Parser->TagText.copyText(dc.ext, 3);
		break;
	case CSC_EXTEVAL:
		dc.exteval = Parser->TagText.getAsChar();
		break;
	case CSC_NODES:
//JJB 100105 2 became 3
		Parser->TagText.copyText(dc.nodes, 3);
		break;
	case CSC_NODESEVAL:
		dc.nodeseval = Parser->TagText.getAsChar();
		break;
	case CSC_LNPOS:
		Parser->TagText.copyText(dc.lnpos, 2);
		break;
	case CSC_LNEXAM:
		Parser->TagText.copyText(dc.lnexam, 2);
		break;
	case CSC_METS:
		Parser->TagText.copyText(dc.mets, 2);
		break;
	case CSC_METSEVAL:
		dc.metseval = Parser->TagText.getAsChar();
		break;
//JJB 100105 ->
	case CSC_DXYEAR:
		Parser->TagText.copyText(dc.diagnosis_year, 4);
		break;
	case CSC_VERORIG:
		Parser->TagText.copyText(dc.csver_original, 6);
		break;
	case CSC_LVI:
		dc.lvi = Parser->TagText.getAsChar();
		break;
//	case CSC_SEX:
//		dc.sex = Parser->TagText.getAsChar();
//		break;
//JJB 100105 <-
	case CSC_SSF1:
		Parser->TagText.copyText(dc.ssf1, 3);
		break;
	case CSC_SSF2:
		Parser->TagText.copyText(dc.ssf2, 3);
		break;
	case CSC_SSF3:
		Parser->TagText.copyText(dc.ssf3, 3);
		break;
	case CSC_SSF4:
		Parser->TagText.copyText(dc.ssf4, 3);
		break;
	case CSC_SSF5:
		Parser->TagText.copyText(dc.ssf5, 3);
		break;
	case CSC_SSF6:
		Parser->TagText.copyText(dc.ssf6, 3);
		break;
//JJB 100105 ->
	case CSC_SSF7:
		Parser->TagText.copyText(dc.ssf7, 3);
		break;
	case CSC_SSF8:
		Parser->TagText.copyText(dc.ssf8, 3);
		break;
	case CSC_SSF9:
		Parser->TagText.copyText(dc.ssf9, 3);
		break;
	case CSC_SSF10:
		Parser->TagText.copyText(dc.ssf10, 3);
		break;
	case CSC_SSF11:
		Parser->TagText.copyText(dc.ssf11, 3);
		break;
	case CSC_SSF12:
		Parser->TagText.copyText(dc.ssf12, 3);
		break;
	case CSC_SSF13:
		Parser->TagText.copyText(dc.ssf13, 3);
		break;
	case CSC_SSF14:
		Parser->TagText.copyText(dc.ssf14, 3);
		break;
	case CSC_SSF15:
		Parser->TagText.copyText(dc.ssf15, 3);
		break;
	case CSC_SSF16:
		Parser->TagText.copyText(dc.ssf16, 3);
		break;
	case CSC_SSF17:
		Parser->TagText.copyText(dc.ssf17, 3);
		break;
	case CSC_SSF18:
		Parser->TagText.copyText(dc.ssf18, 3);
		break;
	case CSC_SSF19:
		Parser->TagText.copyText(dc.ssf19, 3);
		break;
	case CSC_SSF20:
		Parser->TagText.copyText(dc.ssf20, 3);
		break;
	case CSC_SSF21:
		Parser->TagText.copyText(dc.ssf21, 3);
		break;
	case CSC_SSF22:
		Parser->TagText.copyText(dc.ssf22, 3);
		break;
	case CSC_SSF23:
		Parser->TagText.copyText(dc.ssf23, 3);
		break;
	case CSC_SSF24:
		Parser->TagText.copyText(dc.ssf24, 3);
		break;
	case CSC_SSF25:
		Parser->TagText.copyText(dc.ssf25, 3);
		break;
//JJB 100105 <-

	default:
		return rc;
	}

	return true;
}

/*******************************************************************************/
/*                COLLABORATIVE STAGING 'GET SCHEMA' REQUEST                   */
/*******************************************************************************/

void TONCReqCSGetSchema::execute()
{
/*
	Situations where a schema discriminator is required can be identified 
	by calling CStage_get_schema_number() and examining the return value.  
	If a schema discriminator is required but a valid discriminator has not 
	been supplied, the function will return a positive integer whose value 
	is greater than the value returned by CStage_get_number_of_schemas().   
	The integers  high 16 bits will contain a schema number and its low 
	16 bits will contain a table number.  This value is guaranteed to be 
	non-negative and outside the range of valid schema numbers 
	(as determined by CStage_get_number_of_schemas()); thus, any return 
	value greater than CStage_get_number_of_schemas() indicates a schema 
	discriminator is required.
    The schema number and table number taken together indicate the location 
	of a schema discriminator table that lists all valid discriminator values 
	for the site/histology combination.  Once a schema discriminator code has 
	been selected, a second call to CStage_get_schema_number() can be made.
    In situations where a discriminator is not required, the schema 
	discriminator will be 	ignored
	*/
	int ii;
//	char *discriminator = NULL;
	int num_of_schemas = CStage_get_number_of_schemas();
/*JJB NEEDS WORK */
	SchemaNumber = CStage_get_schema_number(Site, Hist, Discriminator);
	if( SchemaNumber > 0 ) {
		if(SchemaNumber <= num_of_schemas) {
			IntTag("SCHEMA", SchemaNumber);
			StrTag("SCHEMA-NAME", CStage_get_schema_name(SchemaNumber));
		}
		else {
			int dSchemaNumber = SchemaNumber / 65536;
			int dTableNumber = SchemaNumber % 65536;

/* Use CStage_get_number_of_rows() to get the table length, 
 CStage_get_table_pattern() to determine the table width, 
 and CStage_get_code_string() and 
 CStage_get_description_string() for every row to build a choice list.

 Example call to CStage_get_code_string(), CStage_get_description_string() 
    to populate second row of choice list
 CStage_get_code_string(19, 34, 2, 1) = 020
 CStage_get_description_string(19, 34, 2) = Adenoid\nPharyngealtonsil\nNasopharyngeal tonsil
*/
			int number_of_rows = CStage_get_number_of_rows(dSchemaNumber, dTableNumber);
/* For any table the pattern attribute is a string that shows the column structure 
of the table.  Pattern strings have the form J-K-L, in which 
J represents the number of code cells at the beginning of a row; 
K represents the number (0 or 1) of description cells; and 
L represents the number of code cells following a description cell.

1-0-0 is the minimal table pattern, indicating a row consisting of only one code cell.  
1-1-3, a common pattern, indicates a row consisting of a single code cell 
followed by a description cell followed by three code cells.
*/
			const char *table_pattern = CStage_get_table_pattern(dSchemaNumber, dTableNumber); 
			setFault(new TONCFault(EC_NEED_DISCRIMINATOR, Site, Hist));
			for(ii=1; ii<=number_of_rows; ii++) {
				const char *codeString = 
					CStage_get_code_string(dSchemaNumber, dTableNumber, ii, 1);
				const char *descString   = 
					CStage_get_description_string(dSchemaNumber, dTableNumber, ii);
				printf("<DISCRIMINATOR CODE=\"%s\">", codeString);
				writeText(descString);
				puts("</DISCRIMINATOR>");

			}
		}
	}
	else
		setFault(new TONCFault(EC_SITE_HIST, Site, Hist));
}

/*******************************************************************************/
/*                COLLABORATIVE STAGING 'GET TABLES' REQUEST                   */
/*******************************************************************************/

void TONCReqCSGetTables::execute()
{
		char *discriminator = NULL;

	//--- Schema number
	if( SchemaNumber <= 0 )
	{
/*JJB NEEDS WORK */
		SchemaNumber = CStage_get_schema_number(Site, Hist, discriminator);
		if( SchemaNumber <= 0 )
		{
			setFault(new TONCFault(EC_SITE_HIST, Site, Hist));
			return;
		}
	}

	//--- Table number(s)
	int NumberOfTables = CStage_get_number_of_tables(SchemaNumber);
	int FirstTable = 1, LastTable = NumberOfTables;
	if( TableNumber > 0 )
	{
		if( TableNumber > NumberOfTables )
		{
			setFault(new TONCFault(EC_INVALID_TABNUM));
			return;
		}
		FirstTable = TableNumber;
		LastTable = TableNumber;
	}

	IntTag("SCHEMA", SchemaNumber);

	//--- Tables
	for(int tblNum=FirstTable; tblNum<=LastTable; tblNum++)
	{
		//--- Get the table pattern and check for the presence of row description
//JJB 100105 Added const
		const char* TablePattern = CStage_get_table_pattern(SchemaNumber, tblNum);
		bool hasDescr = Piece(TablePattern, "-", 2) != "0";
		if( !hasDescr ) continue;

		//--- Determine the number of codes
		int NumberOfCodes = atoi(Piece(TablePattern, "-", 3).c_str()) + 1;
		if( NumberOfCodes == 1 )
			NumberOfCodes = atoi(Piece(TablePattern, "-", 1).c_str());

		//--- Table header
		puts("<TABLE>");
		IntTag("NUMBER", tblNum);
		StrTag("PATTERN", TablePattern);
/*JJB 091224 ->
Function Deprecated 		StrTag("ROLE", CStage_get_table_role(SchemaNumber, tblNum)); 
*/
		StrTag("SUBTITLE", CStage_get_table_subtitle(SchemaNumber, tblNum));
		StrTag("TITLE", CStage_get_table_title(SchemaNumber, tblNum));

		//--- Table rows
		puts("<ROWS>");
		int NumberOfRows = CStage_get_number_of_rows(SchemaNumber, tblNum);
		for(int Row=1; Row<=NumberOfRows; Row++)
		{
			puts("<ROW>");
			StrTag("CODE", CStage_get_code_string(SchemaNumber, tblNum, Row, 1));
			puts("<DESCR>");
//JJB 100105 Added const
			const char* Description = CStage_get_description_string(SchemaNumber, tblNum, Row);
			if( strlen(Description) > 0 ) 
				writeStructuredText(Description);
			puts("</DESCR>");
			for(int i=2; i<=NumberOfCodes; i++)
				StrTag("AC", CStage_get_code_string(SchemaNumber, tblNum, Row, i));
			puts("</ROW>");
		}
		puts("</ROWS>");

		//--- Notes
//JJB 100105 Added const
		const char *Note;
		int NumberOfNotes, NoteNumber;
		puts("<NOTES>");
		NumberOfNotes = CStage_get_number_table_notes(SchemaNumber, tblNum);
		for( NoteNumber=1; NoteNumber<=NumberOfNotes; NoteNumber++ )
		{
			Note = CStage_get_table_note(SchemaNumber, tblNum, NoteNumber );
			if( Note != NULL )
			{
				puts("<TN>");
				writeStructuredText(Note);
				puts("</TN>");
			}
		}
		NumberOfNotes = CStage_get_number_table_footnotes(SchemaNumber, tblNum);
		for( NoteNumber=1; NoteNumber<=NumberOfNotes; NoteNumber++ )
		{
			Note = CStage_get_table_footnote(SchemaNumber, tblNum, NoteNumber);
			if( Note != NULL )
			{
				puts("<FN>");
				writeStructuredText(Note);
				puts("</FN>");
			}
		}
		puts("</NOTES>");

		//--- Table trailer
		puts("</TABLE>");
	}
}
