/*******************************************************************************/
/*  Package:      ONC  Oncology                                               */
/*  Date Created: Jul 26,2004                                                  */
/*  Site Name:    Hines OIFO                                                   */
/*  Developers:   Sergey Gavrilov PII                     )                    */
/*  Description:  Implementation of the TONCParser class that encapsulates     */
/*                XML parsing functionality for requests processed by the      */
/*                Oncology Web-service                                         */
/*******************************************************************************/

#include "stdafx.h"
#include "OncologyCGI.h"
#include <fstream>
#include <ctime>
#include <string> 

using namespace std; 

#define XMLPARSER	((XML_Parser)userData)
#define DEBUG_FILE_NAME "oncsrvXML"
#define NAACCR_RECORD_LENGHTH 22824


/*******************************************************************************/
/*                       LOCAL DATA TYPES DECLARATIONS                         */
/*******************************************************************************/

typedef struct {
	const char* name;	// Tag Name
	TONCTagCode code;	// Tag Code
} TTagDef;

/*******************************************************************************/
/*                        LOCAL FUNCTION DECLARATIONS                          */
/*******************************************************************************/

static void XMLCALL TagEndHandler(void *userData, const XML_Char *name);
static void XMLCALL TagStartHandler(void *userData, const XML_Char *name, const XML_Char **atts);
static void XMLCALL TagTextHandler(void *userData, const XML_Char *text, int len);

/*******************************************************************************/
/*                        VARIABLES                                            */
/*******************************************************************************/
char* InputDataFileDateTime;

/*******************************************************************************/
/*                              PARAMETER PARSER                               */
/*******************************************************************************/

TONCParser::TONCParser()
{
	Request = NULL;
}

//*****	CREATES THE REQUEST OBJECT ACCORDING TO THE REQUEST TAG CODE

TONCRequest* TONCParser::createRequest(const TONCTagCode TagCode, const XML_Char **atts)
{
	//--- Destroy the previous (incomplete) request if it exists
	delete Request;
	//--- Create the request object
	Request = TONCRequest::createRequest(this, TagCode, atts);
	//---
	return Request;
}

//*****	RETURNS VALUE OF THE ATTRIBUTE

const XML_Char* TONCParser::getAttrVal(const XML_Char **atts, const char* name)
{
	static char empty[1] = "";
	const XML_Char* val = NULL;
	for( int i=0; atts[i]; i+=2 )
		if( !_stricmp(atts[i], name) )
		{
			val = atts[i+1];
			break;
		}
		return val != NULL ? val : empty;
}


/*******************************************************************************/
/* Extract a substring from a string in between a beginning tag/string and     */
/* an end tag/string                                                           */
/*******************************************************************************/
char* ExtractStringInBetween(char* sourceString, char* startTag, char* endTag, int fixedLength)
{
	char* result = NULL;
	char* tempString;
	char* tempString2;
	char* trailerString;
	int size = 0;

	if (strlen(sourceString) > 0)
	{
		tempString = strstr(sourceString, startTag);
		if (tempString)
		{			
			size = (int) strlen(tempString) - (int) strlen(startTag) + 1;
			tempString2 = (char*) malloc(size);
			strncpy(tempString2, tempString + strlen(startTag), strlen(tempString) - strlen(startTag));
			tempString2[size - 1] = 0;
			trailerString = strstr(tempString2, endTag);
			size = (int) strlen(tempString2) - (int) strlen(trailerString);
			if (fixedLength > 0)
			{
				result = (char*) malloc(fixedLength + 1);
				if (size < fixedLength)
				{
					strncpy(result, tempString2, size);

					for (int count = size; count < fixedLength; count++)
					{
						strcat(result, " ");
					}
				}
				else
				{
					strncpy(result, tempString2, fixedLength);
				}
				result[fixedLength] = 0;
			}
			else
			{
				result = (char*) malloc(size + 1);
				strncpy(result, tempString2, size);
				result[size] = 0;
			}
		}
	}
	return result;
}

/*****	Get a configuration value from the INI file ****/
char* TONCParser::GetConfigValue(char* iniFileName, char* configKeyStartTag, char* configKeyEndTag)
{
	char* configValue;

	char* allConfigInfo = GetStringFromFile(iniFileName);
	if (allConfigInfo)
	{
		configValue = ExtractStringInBetween(allConfigInfo, configKeyStartTag, configKeyEndTag, 0);
	}
	
	return configValue;
}

char* TONCParser::GetFileNameWithPath(char* fileName, bool debugFolder, char* outputSubFolderName)
{
	char* fileNameWithPath = (char*) malloc(100);

	char* path = GetConfigValue(INI_FILE_NAME, OUTPUT_PATH_CONFIG_VALUE_START_TAG, OUTPUT_PATH_CONFIG_VALUE_END_TAG);
	if (path)
	{
		if ((int) strlen(path) > 0)
		{
			strcpy(fileNameWithPath, path);
			strcat(fileNameWithPath, "\\");
			if (debugFolder)
			{
				strcat(fileNameWithPath, "DEBUG\\");
			}
			else
			{
				strcat(fileNameWithPath, "DATA\\");
				strcat(fileNameWithPath, outputSubFolderName);
				strcat(fileNameWithPath, "\\");
			}
		}
		else
		{
			throw TONCFault(EC_FOLDER_NOT_EXISTS, path);
		}
	}
	else
	{
		throw TONCFault(EC_FOLDER_NOT_EXISTS, path);
	}

	strcat(fileNameWithPath, fileName);
	return fileNameWithPath;

}

/*******************************************************************************/
/* Write the medical data to an output file                                    */
/*******************************************************************************/
void TONCParser::WriteOutputFile(char* medicalData)
{
	int size = 0;

	size = (int) strlen(medicalData);

	if (size > 0)
	{
		FILE *outputFile = NULL;
		char *outputFileName = (char*) malloc(100);

		//--- Parse request string and output the part of input string to a file
		if (outputFileName != NULL)   {
			char* dateTime = GetCurrentDateTime();
			if (size == NAACCR_RECORD_LENGHTH)
			{
				strcpy(outputFileName, "ONCSRV_"); 
			}
			else
			{
				strcpy(outputFileName, "LNCSRV_"); 
			}

			strcat(outputFileName, dateTime);

			// new way to determine new vs updates is inspecting the record type field - position 1
			// position #1 has "M" or "A". This is always populated, but keep the logic as it was before,
			// Check for a modification, and if it's anything else, mark as new.

			if (medicalData[0] == 'M')
			{
				outputFileName = GetFileNameWithPath(outputFileName, false, "UPDATE");
			}
			else
			{
				outputFileName = GetFileNameWithPath(outputFileName, false, "NEW");
			}

			//// position #2635 has "U" or "N" or BLANK
			//if (medicalData[2634] == 'U')
			//{
			//	outputFileName = GetFileNameWithPath(outputFileName, false, "UPDATE");
			//}
			//else
			//{
			//	outputFileName = GetFileNameWithPath(outputFileName, false, "NEW");
			//}

			outputFileName[(int) strlen(outputFileName)] = 0;
		} 

		if((errno = fopen_s(&outputFile, outputFileName, "wb")) == 0) {
			fwrite(medicalData, 1, size, outputFile);
			fflush(outputFile);
			fclose(outputFile);
			extern char* GlobalOutputFileName;
			GlobalOutputFileName = (char*) malloc(100);
			strcpy(GlobalOutputFileName, outputFileName);
		}
	}
}


/*****	Get the current date time to a string and format it as YYMMDDHHMMSS.  ****/
char* TONCParser::GetCurrentDateTime()
{
	time_t rawtime;   
	struct tm * timeinfo; 
	char buffer [80];    
	time ( &rawtime );   
	timeinfo = localtime ( &rawtime );    
	strftime (buffer,80,"%Y%m%d%H%M%S",timeinfo);   

	return buffer;
}

/*****	Get a string from an environment variable ****/
char* TONCParser::GetStringEnvironmentVariable(char* variableName)
{

	long double number;
	int i;

	char *value;
    size_t len;

    errno_t err = _dupenv_s( &value, &len, variableName );

	if (  err ) 
	{
		throw TONCFault(EC_ENVIRONMENT_VARIABLE_NOT_FOUND, "Environment variable not found.");
	}

	return value;
}

/*****  Get the Debug file name  *****/
char* TONCParser::GetDubugFileName()
{
	char *fileName = (char*) malloc(100);

	//--- Parse request string and output the part of input string to a file
	if (fileName != NULL)   {
		if (!InputDataFileDateTime)
		{
			InputDataFileDateTime = GetCurrentDateTime();
		}
		strcpy(fileName, DEBUG_FILE_NAME);  
		strcat(fileName, "_");   
		strcat(fileName, InputDataFileDateTime);   
		strcat(fileName, ".in");   
		fileName = GetFileNameWithPath(fileName, true, "");
		fileName[(int) strlen(fileName)] = 0;
	} 
	return fileName;
}

/*****	Get a string from a file ****/
char* TONCParser::GetStringFromFile(char* fileName)
{
	FILE *inputFile = NULL;
	long size;
	char * buffer = NULL;
	size_t result;

	if((errno = fopen_s(&inputFile, fileName, "r")) == 0) {
		// obtain file size:
		fseek (inputFile , 0 , SEEK_END);
		size = ftell (inputFile);
		rewind (inputFile);

		// allocate memory to contain the whole file:
		buffer = (char*) malloc (sizeof(char)*size);
		if (buffer != NULL) 
		{
			// copy the file into the buffer:
			result = fread (buffer, 1, size, inputFile);
			buffer[(int) size] = 0;
		}

		// terminate
		fclose (inputFile);
	}
	return buffer;
}
//*****	READS PARAMETERS OF THE REQUEST FROM THE STDIN AND PARSES THEM

TONCRequest* TONCParser::parseParameters(int content_length, char* debugFileName)
{
	XML_Parser Parser = XML_ParserCreateNS(NULL, '^');
	int bytes_read, rc = 0;

	XML_SetElementHandler(Parser, TagStartHandler, TagEndHandler);
	XML_SetCharacterDataHandler(Parser, TagTextHandler);
	XML_UseParserAsHandlerArg(Parser);
	XML_SetUserData(Parser, this);

	try
	{
		do {
			void* buf = XML_GetBuffer(Parser, BUFF_SIZE);
			if( buf == NULL )
				throw TONCFault(EC_XMLBUF_ALLOC);

			memset(buf, 0, BUFF_SIZE);
			bytes_read = (content_length > 0) && (content_length < BUFF_SIZE) ? content_length : BUFF_SIZE;
			bytes_read = (int)fread(buf, 1, bytes_read, stdin);

			//Write the input stream to a debug file
			FILE *dumpFile = NULL;
			if((errno = fopen_s(&dumpFile, debugFileName, "wb")) == 0) {
				fwrite(buf, 1, bytes_read, dumpFile);
				fflush(dumpFile);
				fclose(dumpFile);
			}

			if( XML_ParseBuffer(Parser, bytes_read, bytes_read < BUFF_SIZE) != XML_STATUS_OK )
				throw TONCFault(EC_BAD_XML);

			if( content_length > 0 )
			{
				content_length -= bytes_read;
				if( content_length <= 0 )
					bytes_read = 0;
			}
		} while( bytes_read == BUFF_SIZE );
	}
	catch(...)
	{
		int ierror = XML_GetErrorCode(Parser);
		XML_SetUserData(Parser, NULL);
		XML_ParserFree(Parser);
		delete Request;
		Request = NULL;
		throw;
	}

	XML_SetUserData(Parser, NULL);
	XML_ParserFree(Parser);

	return Request;
}

/*******************************************************************************/
/*                                  TEXT BUFFER                                */
/*******************************************************************************/

TONCTextBuffer::TONCTextBuffer()
{
	BufferLength = 32000;  // Initial size of the text buffer (in bytes)
	Buffer = new char[BufferLength+1];
	clear();
}

TONCTextBuffer::~TONCTextBuffer()
{
	delete [] Buffer;
}

//***** APPENDS THE TEXT TO THE BUFFER

void TONCTextBuffer::append(const XML_Char* text, int txtlen)
{
	//--- Calculate the actual text length
	if( txtlen < 0 )
		txtlen = (int)strlen(text);

	if( txtlen > 0 )
	{
		//--- Expand the buffer if necessary
		if( (TextLength + txtlen) > BufferLength )
		{
			BufferLength += __max(txtlen, 4096);  // Increment of the text buffer size (in bytes)
			XML_Char* tmp = new char[BufferLength+1];
			if( tmp == NULL )
				throw TONCFault(EC_NOT_ENOUGH_MEM);
//JJB 100105
			strcpy_s(tmp, sizeof(tmp), Buffer);
			delete [] Buffer;
			Buffer = tmp;
		}

		//--- Append the text to the buffer
//JJB 100105 Added const
		strncat_s(Buffer, BufferLength, text, txtlen);
		TextLength += txtlen;
		Buffer[TextLength] = '\0';
	}
}

//***** CLEARS THE BUFFER

void TONCTextBuffer::clear()
{
	Buffer[0] = '\0';
	TextLength = 0;
}

//***** COPIES THE TEXT FROM THE BUFFER TO THE PROVIDED VARIABLE

void TONCTextBuffer::copyText(char* dst, size_t maxlen, bool append0)
{
/*JJB 100105 ->
This is only slightly better than strncpy; we don't know the size of dst
*/
	memcpy(dst, Buffer, maxlen);
	if( append0 )
		dst[maxlen] = '\0';
}

/*******************************************************************************/
/*                                  XML PATH                                   */
/*******************************************************************************/

TONCXMLPath::TONCXMLPath()
{
	BufferLength = 256;  // Initial size of the path buffer (in bytes)
	Buffer = new char[BufferLength+1];
	clear();
}

TONCXMLPath::~TONCXMLPath()
{
	delete [] Buffer;
}

//***** CLEARS THE BUFFER

void TONCXMLPath::clear()
{
	Buffer[0] = '\0';
	PathLength = 0;
}

//***** RETURNS THE CODE THAT CORRESPONDS TO THE CURRENT PATH

static int __cdecl tag_compare(const void *td1, const void *td2)
{
	return _stricmp(((TTagDef*)td1)->name, ((TTagDef*)td2)->name);
}

TONCTagCode TONCXMLPath::getCode()
{
	//   !!! TAG DEFINITIONS MUST BE LISTED IN ALPHABETICAL ORDER (BY PATH) !!!
	static TTagDef tag_list[] = {
		{ "/Envelope",										SOAP_ENVELOPE		},
		{ "/Envelope/Body",									SOAP_BODY			},

		{ "/Envelope/Body/CS-CALCULATE",					CS_CALCULATE		},
		{ "/Envelope/Body/CS-CALCULATE/AGE",				CSC_AGE				},
		{ "/Envelope/Body/CS-CALCULATE/BEHAV",				CSC_BEHAV			},
//JJB 100105
		{ "/Envelope/Body/CS-CALCULATE/CSVER_ORIGINAL",		CSC_VERORIG			},
		{ "/Envelope/Body/CS-CALCULATE/DIAGNOSIS_YEAR",		CSC_DXYEAR			},
		{ "/Envelope/Body/CS-CALCULATE/EXT",				CSC_EXT				},
		{ "/Envelope/Body/CS-CALCULATE/EXTEVAL",			CSC_EXTEVAL			},
		{ "/Envelope/Body/CS-CALCULATE/GRADE",				CSC_GRADE			},
		{ "/Envelope/Body/CS-CALCULATE/HIST",				CSC_HIST			},
		{ "/Envelope/Body/CS-CALCULATE/LNEXAM",				CSC_LNEXAM			},
		{ "/Envelope/Body/CS-CALCULATE/LNPOS",				CSC_LNPOS			},
//JJB 100105
		{ "/Envelope/Body/CS-CALCULATE/LVI",				CSC_LVI				},
		{ "/Envelope/Body/CS-CALCULATE/METS",				CSC_METS			},
		{ "/Envelope/Body/CS-CALCULATE/METSEVAL",			CSC_METSEVAL		},
		{ "/Envelope/Body/CS-CALCULATE/NODES",				CSC_NODES			},
		{ "/Envelope/Body/CS-CALCULATE/NODESEVAL",			CSC_NODESEVAL		},
//JJB 100105
//		{ "/Envelope/Body/CS-CALCULATE/SEX",				CSC_SEX				},
		{ "/Envelope/Body/CS-CALCULATE/SITE",				CSC_SITE			},
		{ "/Envelope/Body/CS-CALCULATE/SIZE",				CSC_SIZE			},
		{ "/Envelope/Body/CS-CALCULATE/SSF1",				CSC_SSF1			},
//JJB 100105 -> This is all rearranged to keep sorted
		{ "/Envelope/Body/CS-CALCULATE/SSF10",				CSC_SSF10			},
		{ "/Envelope/Body/CS-CALCULATE/SSF11",				CSC_SSF11			},
		{ "/Envelope/Body/CS-CALCULATE/SSF12",				CSC_SSF12			},
		{ "/Envelope/Body/CS-CALCULATE/SSF13",				CSC_SSF13			},
		{ "/Envelope/Body/CS-CALCULATE/SSF14",				CSC_SSF14			},
		{ "/Envelope/Body/CS-CALCULATE/SSF15",				CSC_SSF15			},
		{ "/Envelope/Body/CS-CALCULATE/SSF16",				CSC_SSF16			},
		{ "/Envelope/Body/CS-CALCULATE/SSF17",				CSC_SSF17			},
		{ "/Envelope/Body/CS-CALCULATE/SSF18",				CSC_SSF18			},
		{ "/Envelope/Body/CS-CALCULATE/SSF19",				CSC_SSF19			},
		{ "/Envelope/Body/CS-CALCULATE/SSF2",				CSC_SSF2			},
		{ "/Envelope/Body/CS-CALCULATE/SSF20",				CSC_SSF20			},
		{ "/Envelope/Body/CS-CALCULATE/SSF21",				CSC_SSF21			},
		{ "/Envelope/Body/CS-CALCULATE/SSF22",				CSC_SSF22			},
		{ "/Envelope/Body/CS-CALCULATE/SSF23",				CSC_SSF23		    },
		{ "/Envelope/Body/CS-CALCULATE/SSF24",				CSC_SSF24			},
		{ "/Envelope/Body/CS-CALCULATE/SSF25",				CSC_SSF25			},
		{ "/Envelope/Body/CS-CALCULATE/SSF3",				CSC_SSF3			},
		{ "/Envelope/Body/CS-CALCULATE/SSF4",				CSC_SSF4			},
		{ "/Envelope/Body/CS-CALCULATE/SSF5",				CSC_SSF5			},
		{ "/Envelope/Body/CS-CALCULATE/SSF6",				CSC_SSF6			},
		{ "/Envelope/Body/CS-CALCULATE/SSF7",				CSC_SSF7			},
		{ "/Envelope/Body/CS-CALCULATE/SSF8",				CSC_SSF8			},
		{ "/Envelope/Body/CS-CALCULATE/SSF9",				CSC_SSF9			},
//JJB 100105 <-
		{ "/Envelope/Body/CS-GET-SCHEMA",					CS_GET_SCHEMA		},
		{ "/Envelope/Body/CS-GET-SCHEMA/DISCRIMINATOR",		CSGS_DISCRIMINATOR	},
		{ "/Envelope/Body/CS-GET-SCHEMA/HIST",				CSGS_HIST			},
		{ "/Envelope/Body/CS-GET-SCHEMA/SITE",				CSGS_SITE			},

		{ "/Envelope/Body/CS-GET-TABLES",					CS_GET_TABLES		},
		{ "/Envelope/Body/CS-GET-TABLES/SCHEMA",			CSGT_SCHEMA			},
		{ "/Envelope/Body/CS-GET-TABLES/TABLE",				CSGT_TABLE			},

		{ "/Envelope/Body/ED-GET-EDITINFO",					ED_GET_EDITINFO		},
		{ "/Envelope/Body/ED-GET-EDITINFO/EDIT",			EDEI_EDIT			},
		{ "/Envelope/Body/ED-GET-EDITINFO/EDIT-SET",		EDEI_EDITSET		},
		{ "/Envelope/Body/ED-GET-EDITINFO/TEXT-WIDTH",		EDEI_TEXTWIDTH		},

		{ "/Envelope/Body/ED-RUN-BATCH",					ED_RUN_BATCH		},
		{ "/Envelope/Body/ED-RUN-BATCH/NAACCR-RECORD",		EDRB_NAACCR_RECORD	},

		{ "/Envelope/Body/GET-VERSION",						GET_VERSION			}
	};

	TTagDef *ptr;
	TTagDef tag = {Buffer, UNKNOWN_TAG};
	try
	{
		ptr = (TTagDef*)bsearch(&tag, tag_list,
			sizeof(tag_list)/sizeof(TTagDef), sizeof(TTagDef), tag_compare);
	}
	catch(...)
	{
		ptr = NULL;
	}
	return ptr != NULL ? ptr->code : UNKNOWN_TAG;
}

//***** RETURNS THE NAME OF THE CURRENT (LAST ADDED) TAG

const XML_Char* TONCXMLPath::getLastTag()
{
	const XML_Char* ptr = strrchr(Buffer, '/');
	return (ptr != NULL ? ptr+1 : Buffer);
}

//***** ADDS THE TAG NAME TO THE XML PATH

void TONCXMLPath::pushTag(const XML_Char* tag_name)
{
	string name = Piece(tag_name, "^", 2);
	if( name == "" )
		name = "<unknown>";
	size_t dl = name.length() + 1;

	//--- Expand the path buffer if necessary
	if( (PathLength + dl) > BufferLength )
	{
		BufferLength += __max(dl, 256);  // Incremenmt of the path buffer size (in bytes)
		XML_Char* tmp = new char[BufferLength+1];
		if( tmp == NULL )
			throw TONCFault(EC_NOT_ENOUGH_MEM);
//JJB 100105
		strcpy_s(tmp, sizeof(tmp), Buffer);
		delete [] Buffer;
		Buffer = tmp;
	}

	//--- Append the tag name to the path
//JJB 100105 the str's became _s versions
	strcat_s(Buffer, BufferLength, "/");
	strcat_s(Buffer, BufferLength, name.c_str());
	PathLength += dl;
}

//***** REMOVES THE CURRENT (LAST ADDED) TAG NAME FROM THE XML PATH

void TONCXMLPath::popTag()
{
	XML_Char* ptr = strrchr(Buffer, '/');
	PathLength = (ptr != NULL ? ptr - Buffer : 0);
	Buffer[PathLength] = '\0';
}

/*******************************************************************************/
/*                           TAG HANDLERS FOR EXPAT                            */
/*******************************************************************************/

void XMLCALL TagEndHandler(void *userData, const XML_Char *name)
{
	TONCParser *Parser = (TONCParser*)XML_GetUserData(XMLPARSER);

	//--- Pass the tag to to the request object for processing
	TONCRequest* Request = Parser->getRequest();
	if( Request != NULL )
	{
		TONCTagCode tcode = Parser->Path.getCode();
		Request->parseTagEnd(Parser, tcode);
	}

	//--- Remove the tag from the path (stack)
	Parser->Path.popTag();
}

void TagStartHandler(void *userData, const XML_Char *name, const XML_Char **atts)
{
	TONCParser *Parser = (TONCParser*)XML_GetUserData(XMLPARSER);

	//--- Check if the current tag defines a request
	bool crf = !_stricmp(Parser->Path.getPath(), "/Envelope/Body");

	//--- Append the tag name to the XML path (stack)
	Parser->Path.pushTag(name);

	//--- Get the code of the current XML path
	TONCTagCode tcode = Parser->Path.getCode();

	//--- Clear the buffer for tag value
	Parser->TagText.clear();

	//--- Get the request object or create a new one
	TONCRequest* Request = crf ? Parser->createRequest(tcode, atts) : Parser->getRequest();

	//--- Pass the tag to to the request object for processing
	if( Request != NULL )
		Request->parseTagStart(Parser, tcode, atts);
}

void TagTextHandler(void *userData, const XML_Char *text, int len)
{
	TONCParser *Parser = (TONCParser*)XML_GetUserData(XMLPARSER);

	//--- Accumulate the text value of the current tag
	Parser->TagText.append(text, len);
}