/*******************************************************************************/
/*  Package:      ONC  Oncology                                               */
/*  Date Created: Jul 26,2004                                                  */
/*  Site Name:    Hines OIFO                                                   */
/*  Developers:   Sergey Gavrilov PII                     )                    */
/*  Description:  Implemetation of the generic exception (error description)   */
/*                class for Oncology Web-service                               */
/*******************************************************************************/

#include "stdafx.h"
#include "OncologyCGI.h"

/*******************************************************************************/
/*                               SOAP FAULT CODES                              */
/*******************************************************************************/

const char FAULT_VERSION[]			= "VersionMismatch";
const char FAULT_MUST_UNDERSTAND[]	= "MustUnderstand";
const char FAULT_CLIENT[]			= "Client";
const char FAULT_SERVER[]			= "Server";

/*******************************************************************************/
/*                                FAULT TEMPLATES                              */
/*******************************************************************************/

typedef struct {
	const char* FaultCode;		// SOAP error code
	const char* FaultString;	// Error message
} TFaultTemplate;

/*****	The getFaultTemplate() function returns the Fault template for the
		provided error code.

	anErrorCOde	Error code (negative integer)
*/
static TFaultTemplate* getFaultTemplate(const int anErrorCode)
{
	static TFaultTemplate Templates[] =		// !!! Do not change the order of records !!!
	{
									{ FAULT_SERVER,	"Unknown error." },
		/* EC_UNKNOWN_METHOD	*/	{ FAULT_CLIENT,	"Unknown HTTP request method." },
		/* EC_UNSUP_METHOD		*/	{ FAULT_CLIENT,	"Unsupported HTTP request method: \'%s\'." },
		/* EC_UNKNOWN_CONTENT	*/	{ FAULT_CLIENT,	"Unknown content type." },
		/* EC_UNSUP_CONTENT		*/	{ FAULT_CLIENT,	"Unsupported content type: \'%s\'." },
		/* EC_XMLBUF_ALLOC		*/	{ FAULT_SERVER,	"Cannot allocate a buffer for XML parsing." },
		/* EC_BAD_XML			*/	{ FAULT_CLIENT,	"Incorrectly formatted request." },
		/* EC_INVALID_REQUEST	*/	{ FAULT_CLIENT,	"Invalid/unsupported request." },
		/* EC_INVALID_TABNUM	*/	{ FAULT_CLIENT,	"Invalid table number." },
		/* EC_SITE_HIST			*/	{ FAULT_CLIENT,	"Invalid site (%s) and/or histology (%s)." },
		/* EC_CALC_WARNINGS		*/	{ FAULT_CLIENT,	"CS calculation API returned warning(s)." },
		/* EC_CALC_ERRORS		*/	{ FAULT_CLIENT,	"CS calculation API returned error(s)." },
		/* EC_SERVER_EXCEPTION	*/	{ FAULT_SERVER,	"Server exception." },
		/* EC_NOT_ENOUGH_MEM	*/	{ FAULT_SERVER, "No enough memory for dynamic variable(s)." },
		/* EC_EDITS_ERROR		*/	{ FAULT_SERVER, "Error code \'%d\' was returned by the %s." },
		/* EC_EDITS_ERROR_EX	*/	{ FAULT_SERVER, "Error code \'%d\' was returned by the %s (%s: \'%s\')." },
		/* EC_INVALID_EDITSCFG	*/	{ FAULT_CLIENT,	"Invalid EDITS configuration name: \'%s\'." },
		/* EC_NEED_DISCRIMINATOR */	{ FAULT_CLIENT,	"Discriminator needed for site (%s) and histology (%s)." },
		/* EC_FOLDER_NOT_EXISTS */	{ FAULT_SERVER, "Output path not found: \'%s\'." }
	};
	int i = -anErrorCode;
	i = (i > 0) && (i <= sizeof(Templates)/sizeof(TFaultTemplate)) ? i-1 : 0;
	return Templates + i;
}

/*******************************************************************************/
/*                            GENERIC ERROR DESCRIPTOR                         */
/*******************************************************************************/

TONCFault::TONCFault()
{
	ErrorCode = 0;
	FaultCode[0] = '\0';
	FaultString[0] = '\0';
}

//*****	CONSTRUCTOR

TONCFault::TONCFault(const int anErrorCode,	...)
{
	TONCFault();
	va_list args;
	va_start(args, anErrorCode);

	TFaultTemplate* ft = getFaultTemplate(anErrorCode);
//JJB 100105 ->
	strcpy_s(FaultCode, 20, ft->FaultCode);
	vsprintf_s(FaultString, 200, ft->FaultString, args);
//JJB 100105 <-
	ErrorCode = anErrorCode;
}

//*****	WRITES DETAILS ABOUT THE ERROR (the <detail> SECTION)

void TONCFault::outputDetail()
{
	if( ErrorCode != 0 )
		printf("<RC>%d</RC>\n", ErrorCode);
}

//*****	WRITES THE <soap:Fault> SECTION CORRESPONDING TO THE ERROR TO THE STDOUT

void TONCFault::output()
{
	if( strlen(FaultCode) > 0 )
	{
		puts("<soap:Fault>");
		printf("<faultcode>%s</faultcode>\n", FaultCode);
		printf("<faultstring>%s</faultstring>\n", FaultString);
		puts("<detail>");
		outputDetail();
		puts("</detail>");
		puts("</soap:Fault>");
	}
}
