﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.SessionState;
using VeteransAffairs.Registries.BusinessManager.MDWS;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using System.Globalization;
using System.Data.Linq;
using VeteransAffairs.Registries.BusinessManager.TBIInstrumentVIAsrv;
using System.Web;

namespace VeteransAffairs.Registries.BusinessManager.MSSR
{

    [Serializable()]
    [System.ComponentModel.DataObject]
    public class MSSRInstrumentsManager : BaseBO
    {
        public static readonly string LookupStationCode = "442";
        private CprsManager _cprsManager = new CprsManager();
        public MSSRServiceTO MssrServiceTo { get; set; } // 01-Nov-2010 jps  replace [internal set] with [set] per JH;

        #region MDWS Variable Declarations

        private EmrSvc _mssrService;
        private FaultTO _faultTO;

        private PersonsTO _personsTO;
        public PersonsTO GetPersonTO()
        {
            return this._personsTO;
        }

        #endregion

        #region VIA Variable Declarations

        private TbiServiceInterfaceClient _VIAService;
        private personsTO _personsTOVIA;

        MDWSLog mdwsLog = new MDWSLog();
        MDWS_LOG _mdwsLog = new MDWS_LOG();

        protected string _loginSiteCode = null;
        private string _userID = String.Empty;
        private string _tokenizedDUZ = String.Empty;
        private string _localPid = String.Empty;
        //private string _savedItemID = String.Empty;
        private string _providerName = String.Empty;

        #endregion

        public MSSRInstrumentsManager()
        {
            _defaultSortField = "SURVEYS_ID";

            if (_loginSiteCode == null)
            {
                _loginSiteCode = LookupStationCode;
            }
        }

        ~MSSRInstrumentsManager()
        {
            if (_mssrService != null)
            {
                TaggedTextArray _result = _mssrService.disconnect();
            }
        }

        #region MDWS Service Methods

        /// <summary>
        /// Loads MSSR Service (MDWS) and calls addDataSourceMSSR
        /// </summary>
        /// <param name="mssrServiceTo"></param>
        /// <returns></returns>
        public bool LoadMssrService(MSSRServiceTO mssrServiceTo)
        {
            _mssrService = new EmrSvc();
            _mssrService.CookieContainer = new System.Net.CookieContainer();

            this.MssrServiceTo = mssrServiceTo;

            _faultTO = addDataSourceMSSR(mssrServiceTo);
            if (_faultTO != null)
            {
                this.MssrServiceTo = null;
            }

            _personsTO = CPRSLaunch(mssrServiceTo);
            if (_personsTO == null || _personsTO.fault != null)
            {
                this.MssrServiceTo = null;
            }

            if (this.MssrServiceTo != null && _personsTO != null)
            {
                this.MssrServiceTo.PatientName = _personsTO.patient.patientName;
                this.MssrServiceTo.PatientDOB = _personsTO.patient.dob;
                if (!String.IsNullOrEmpty(_personsTO.patient.deceasedDate))
                    this.MssrServiceTo.PatientDOD = _personsTO.patient.deceasedDate;
                this.MssrServiceTo.PatientGender = _personsTO.patient.gender;
                this.MssrServiceTo.PatientSoN = _personsTO.patient.ssn;
                this.MssrServiceTo.PatientICN = _personsTO.patient.mpiPid;
                this.MssrServiceTo.PatientDFN = _personsTO.patient.localPid;
                this.MssrServiceTo.STA6N = _personsTO.user.siteId;
                this.MssrServiceTo.PatientAge = _personsTO.patient.age.ToString();
                this.MssrServiceTo.PatientEthnicity = _personsTO.patient.ethnicity;

                // look for demographic data
                if (_personsTO.patient.demographics != null)
                {
                    if (_personsTO.patient.demographics.Length > 0)
                    {
                        DemographicSetTO demo = _personsTO.patient.demographics[0];

                        if (demo.addresses != null)
                        {
                            if (demo.addresses.Length > 0)
                            {
                                AddressTO addy = demo.addresses[0];
                                if (addy != null)
                                {
                                    if (!String.IsNullOrEmpty(addy.city))
                                    {
                                        this.MssrServiceTo.PatientCity = addy.city.ToUpper();
                                    }

                                    if (!String.IsNullOrEmpty(addy.state))
                                    {
                                        this.MssrServiceTo.PatientState = addy.state.ToUpper();
                                    }
                                }
                            }
                        }
                    }
                }

                this.MssrServiceTo.UserName = _personsTO.user.name;

                this.MssrServiceTo.PatientAdmintTimeStamp = _personsTO.patient.admitTimestamp;

                if (_personsTO.patient.location != null)
                {
                    this.MssrServiceTo.PatientLocationId = _personsTO.patient.location.id;

                    if (_personsTO.patient.location.facility != null)
                    {
                        this.MssrServiceTo.InstitutionName = _personsTO.patient.location.facility.name;
                    }

                }

                // TODO: PCP - would need to store in MSSR_PATIENT_DEMS
                //TaggedText patientPCP = GetPatientPCP(this.MssrServiceTo);
                //if (patientPCP != null) {

                //}
            }


            if (this.MssrServiceTo == null)
            {
                return false;
            }
            else
            {
                return true;
            }

        }

        /// <summary>
        /// MDWS: calls addDataSourceMSSR to tell MDWS what server to connect to.
        /// </summary>
        /// <param name="mssrServiceTo"></param>
        /// <returns></returns>
        private FaultTO addDataSourceMSSR(MSSRServiceTO mssrServiceTo)
        {
            FaultTO ret = null;

            if (mssrServiceTo.siteCode == "931")
            {
                SiteTO result = null;

                MDWSLog mdwsLog = new MDWSLog();
                MDWS_LOG _mdwsLog = new MDWS_LOG();

                _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                _mdwsLog.SERVICE_NAME = "addDataSource";
                _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { mssrServiceTo.siteCode, "TESTSITE", mssrServiceTo.ServerIP, mssrServiceTo.ServerPort, "HIS", "VISTA", "11" });

                try
                {
                    result = _mssrService.addDataSource(mssrServiceTo.siteCode, "TESTSITE", mssrServiceTo.ServerIP, mssrServiceTo.ServerPort, "HIS", "VISTA", "11");
                    if (result != null)
                    {
                        _mdwsLog.RETURNED_DATA = result.Serialize();

                        if (result.fault != null)
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                        }
                    }
                    else
                    {
                        _mdwsLog.ERROR_LEVEL = 2;
                    }
                }
                catch (Exception exc)
                {
                    Logging.WriteLogToFile("Failure in addDataSourceMSSR: " + exc.Message);

                    _mdwsLog.ERROR_LEVEL = 2;
                }

                mdwsLog.LogMDWSCall(_mdwsLog);

                ret = result.fault;
            }

            return ret;
        }

        /// <summary>
        /// MDWS: CPRSLauch Method
        /// </summary>
        /// <param name="mssrServiceTo"></param>
        /// <returns></returns>
        private PersonsTO CPRSLaunch(MSSRServiceTO mssrServiceTo)
        {
            PersonsTO result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "cprsLaunch";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { "TBI-PWD", mssrServiceTo.siteCode, mssrServiceTo.DUZ, mssrServiceTo.DFN });
            //*** DO NOT OVERRIDE THE siteCode IN THIS METHOD
            try
            {
                result = _mssrService.cprsLaunch("TBI-PWD", mssrServiceTo.siteCode, mssrServiceTo.DUZ, mssrServiceTo.DFN);
                if (result != null)
                {
                    _mdwsLog.RETURNED_DATA = result.Serialize();

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                    else
                    {
                        StationDivision = mssrServiceTo.siteCode;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in CPRSLaunch: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: GetPatientPCP Service Method.
        /// </summary>
        /// <param name="mssrServiceTo"></param>
        /// <returns></returns>
        //private TaggedText GetPatientPCP(MSSRServiceTO mssrServiceTo)
        //{
        //    TaggedText result = null;

        //    MDWSLog mdwsLog = new MDWSLog();
        //    MDWS_LOG _mdwsLog = new MDWS_LOG();

        //    _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
        //    _mdwsLog.SERVICE_NAME = "getPcpForPatient";
        //    _mdwsLog.PARAMETERS = mssrServiceTo.DFN;

        //    try
        //    {
        //        result = _mssrService.getPcpForPatient(mssrServiceTo.DFN);

        //        if (result != null)
        //        {
        //            _mdwsLog.RETURNED_DATA = result.Serialize();

        //            if (result.fault != null)
        //            {
        //                _mdwsLog.ERROR_LEVEL = 1;
        //            }
        //        }
        //        else
        //        {
        //            _mdwsLog.ERROR_LEVEL = 2;
        //        }
        //    }
        //    catch (Exception exc)
        //    {
        //        Logging.WriteLogToFile("Failure in GetPatientPCP: " + exc.Message);

        //        _mdwsLog.ERROR_LEVEL = 2;
        //    }

        //    mdwsLog.LogMDWSCall(_mdwsLog);

        //    return result;
        //}

        /// <summary>
        /// MDWS: Disconnect site service call.
        /// </summary>
        /// <param name="mssrServiceTo"></param>
        /// <returns></returns>
        public TaggedTextArray DisconnectSite(MSSRServiceTO mssrServiceTo)
        {
            TaggedTextArray result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "disconnect";
            _mdwsLog.PARAMETERS = "";

            try
            {
                result = _mssrService.disconnect();
                if (result != null)
                {
                    _mdwsLog.RETURNED_DATA = result.Serialize();

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in DisconnectSite: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: Retrieve consults
        /// </summary>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getConsults(out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            TaggedConsultArrays ret = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getConsultsForPatient";
            _mdwsLog.PARAMETERS = "";

            try
            {
                ret = _mssrService.getConsultsForPatient();
                if (ret != null)
                {
                    _mdwsLog.RETURNED_DATA = ret.Serialize();

                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        error = null;
                        if (!(ret.count == 0 || ret.arrays[0].consults == null))
                        {
                            var consultsArray = ret.arrays.SelectMany(s => s.consults).ToList();
                            result = (from e in consultsArray orderby e.timestamp descending select new KeyValuePair<string, string>(e.id, DateTime.ParseExact(e.timestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("MMM dd,yy") + " (" + e.status + ") " + e.title + " " + "Consult #: " + e.id)).ToList();
                        }
                    }
                }
                else
                {
                    error = "Service returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in getConsults: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: getHospitalAdmissions (getAdmissions) service call
        /// </summary>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getHospitalAdmissions(out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            TaggedInpatientStayArray ret = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getAdmissions";
            _mdwsLog.PARAMETERS = "";

            try
            {
                ret = _mssrService.getAdmissions();
                if (ret != null)
                {
                    _mdwsLog.RETURNED_DATA = ret.Serialize();

                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        error = null;
                        if (!(ret.count == 0 || ret.stays == null))
                        {
                            result = (from e in ret.stays orderby e.admitTimestamp descending select new KeyValuePair<string, string>(e.location.id + ";" + e.admitTimestamp + ";H", e.location.name + " (" + DateTime.ParseExact(e.admitTimestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("MMM dd,yy") + ")")).ToList();
                        }
                    }
                }
                else
                {
                    error = "Service returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in getHospitalAdmissions: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: getVisits service call
        /// </summary>
        /// <param name="fromDate"></param>
        /// <param name="toDate"></param>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getVisits(DateTime fromDate, DateTime toDate, out string error)
        {
            TaggedVisitArray ret = null;
            List<KeyValuePair<string, string>> result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getVisits";

            string formattedFromDate = FormatDateForMDWS(fromDate);
            string formattedToDate = FormatDateForMDWS(toDate);
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { formattedFromDate, formattedToDate });

            try
            {
                ret = _mssrService.getVisits(formattedFromDate, formattedToDate);
                if (ret != null)
                {
                    _mdwsLog.RETURNED_DATA = ret.Serialize();

                    if (ret.fault != null)
                    {
                        error = ret.fault.message;
                        result = null;

                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                    else
                    {
                        error = null;
                        result = (from e in ret.visits orderby e.timestamp descending select new KeyValuePair<string, string>(e.location.id + ";" + e.timestamp + ";A", DateTime.ParseExact(e.timestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("g") + " " + e.location.name + " " + e.status)).ToList();
                    }
                }
                else
                {
                    error = "Service returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in getVisits: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: Utility method for formatting date.
        /// </summary>
        /// <param name="dateValue"></param>
        /// <returns></returns>
        private string FormatDateForMDWS(DateTime dateValue)
        {
            // CPRS is expecting date in the following format "20000101.000000";
            return String.Format("{0:yyyyMMdd.HHmmss}", dateValue);
        }

        /// <summary>
        /// MDWS: This will get the list of CPRS users for the correct MDWS connection
        /// </summary>
        /// <param name="target">Beginning of first user name in the list</param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> cprsUserLookup(string target, out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            UserArray ret = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();


            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "cprsUserLookup";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { target });

            try
            {
                ret = _mssrService.cprsUserLookup(target);

                if (ret != null)
                {
                    _mdwsLog.RETURNED_DATA = ret.Serialize();

                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        error = null;
                        if (!(ret.count == 0 || ret.users == null))
                        {
                            result = (from e in ret.users orderby e.name select new KeyValuePair<string, string>(e.DUZ, e.name)).ToList();
                        }
                    }
                }
                else
                {
                    error = "Service returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in cprsUserLookup: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: This will get the locations for the correct MDWS connection
        /// </summary>
        /// <param name="filter">Use this to filter the results returned from MDWS</param>
        /// <param name="target">Purpose and expected values Unknown </param>
        /// <param name="direction">MDWS expects a value of 1 or -1</param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getLocations(string filter, string target, string direction, out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            TaggedHospitalLocationArray ret = null;
            string searchParameter;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            if (filter.Trim() == String.Empty)
            {
                searchParameter = target;
            }
            else
            {
                searchParameter = filter.Trim().ToUpper();
            }

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getLocations";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { searchParameter, direction });

            try
            {
                ret = _mssrService.getLocations(searchParameter, direction);

                if (ret != null)
                {
                    _mdwsLog.RETURNED_DATA = ret.Serialize();

                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        error = null;
                        if (!(ret.count == 0 || ret.locations == null))
                        {
                            result = (from e in ret.locations where e.name.ToLower().StartsWith(filter.ToLower()) orderby e.name select new KeyValuePair<string, string>(e.id, e.name)).ToList();
                        }
                    }
                }
                else
                {
                    error = "Service returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in getLocations: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: Find STD_INSTITUTION_ID by STA3N and data source provider
        /// </summary>
        /// <param name="sta3n"></param>
        /// <param name="dataSourceProvider"></param>
        /// <param name="error"></param>
        /// <returns></returns>
        public System.Nullable<int> GetStdInstitutionId(string sta3n, string dataSourceProvider)
        {
            System.Nullable<int> result;
            int sta3nInteger = VeteransAffairs.Registries.Business.Utilities.Helpers.ExtractSta3n(sta3n);

            if (sta3nInteger != VeteransAffairs.Registries.Business.Utilities.Helpers.InvalidSta3n)
            {
                if (sta3nInteger > 900)   // test site sta3n
                {
                    MDWSLog mdwsLog = new MDWSLog();
                    MDWS_LOG _mdwsLog = new MDWS_LOG();
                    RegionArray ret = null;

                    _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                    _mdwsLog.SERVICE_NAME = "getVHA";
                    _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { dataSourceProvider });

                    try
                    {
                        if (_mssrService == null)
                        {
                            _mssrService = new EmrSvc();
                            _mssrService.CookieContainer = new System.Net.CookieContainer();
                        }
                        ret = _mssrService.getVHA();
                        if (ret != null)
                        {
                            _mdwsLog.RETURNED_DATA = ret.Serialize();

                            if (ret.fault != null)
                            {
                                _mdwsLog.ERROR_LEVEL = 1;
                            }
                            else
                            {
                                bool sta3nFound = false;
                                for (int i = 0; i < ret.regions.Length; i++)
                                {
                                    RegionTO region = ret.regions[i];

                                    for (int j = 0; j < region.sites.count; j++)
                                    {
                                        SiteTO site = region.sites.sites[j];
                                        for (int k = 0; k < site.dataSources.count; k++)
                                        {
                                            DataSourceTO dataSource = site.dataSources.items[k];
                                            if (dataSource.provider.ToLower() == dataSourceProvider.ToLower())
                                            {
                                                sta3n = dataSource.siteId.tag;
                                                sta3nFound = true;
                                                break;
                                            }
                                        }

                                        if (sta3nFound)
                                        {
                                            break;
                                        }
                                    }

                                    if (sta3nFound)
                                    {
                                        break;
                                    }
                                }
                            }
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 2;
                        }
                    }
                    catch (Exception exc)
                    {
                        Logging.WriteLogToFile("Failure in GetStdInstitutionId: " + exc.Message);

                        _mdwsLog.ERROR_LEVEL = 2;
                        throw;
                    }

                    mdwsLog.LogMDWSCall(_mdwsLog);
                }


            }
            MssrServiceTo.StationNumber = sta3n;
            InstitutionManager institutionManager = new InstitutionManager();
            result = institutionManager.getInstitutionIDByStationNumber(sta3n);

            return result;
        }

        /// <summary>
        /// MDWS: getNoteTitles service call
        /// </summary>
        /// <param name="target"></param>
        /// <param name="direction"></param>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getNoteTitles(string target, string direction, out string error)
        {
            TaggedTextArray ret = null;
            List<KeyValuePair<string, string>> result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getNoteTitles";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { target, direction });

            try
            {
                ret = _mssrService.getNoteTitles(target, direction);
                if (ret != null)
                {
                    _mdwsLog.RETURNED_DATA = ret.Serialize();

                    if (ret.fault != null)
                    {
                        error = ret.fault.message;
                        result = null;

                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                    else
                    {
                        error = null;
                        result = (from e in ret.results.FirstOrDefault().taggedResults orderby e.text select new KeyValuePair<string, string>(e.tag, e.textArray[0])).ToList();
                    }
                }
                else
                {
                    error = "Service returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in getNoteTitles: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: writeNote service call.
        /// </summary>
        /// <param name="mssrServiceTO"></param>
        /// <returns></returns>
        public NoteResultTO writeNote(MSSRServiceTO mssrServiceTO)
        {
            NoteResultTO _writeNoteResult = null;
            TextTO _closeNoteResult;
            string titleIEN = "";
            string encounterString = "";
            string noteText = "";
            string authorDUZ = "";
            string cosignerDUZ = "";
            string consultIEN = "";
            string prfIEN = "";

            //set the values for the writeNote web service based on the noteType
            switch (mssrServiceTO.SelectedNoteType.Id)
            {
                case (int)NoteTypeEnum.ScheduledClinicAppointment:  //Scheduled Clinic Appointment

                    //if the note is for current visit or previous visit, call getVisits and have the user pick one . 
                    //Then encounterString = visit.location.id + ‘;’ + visit.timestamp + ‘;A’ 

                    //already formated from the original call to MDWS
                    encounterString = mssrServiceTO.SelectedVisitId;

                    break;
                case (int)NoteTypeEnum.HospitalAdmission:  //Hospital Admission

                    //if the note is for previous stay, call getAdmissions and have the user pick one. 
                    //Then encounterString = stay.location.id + ‘;’ + stay.admitTimestamp + ‘;H’ 

                    //already formated from the original call to MDWS
                    encounterString = mssrServiceTO.SelectedHospitalAdmissionId;


                    break;
                case (int)NoteTypeEnum.CurrentStay:  //Current Stay

                    //if the note is for current stay, encounterString = patient.locationId + ‘;’ + patient.admitTimestamp + ‘;H’ 

                    encounterString = mssrServiceTO.PatientLocationId + ";" + mssrServiceTO.PatientAdmintTimeStamp + ";H";

                    break;
                case (int)NoteTypeEnum.UnscheduledOrNewVisit: //Unscheduled or New Visit

                    if (!mssrServiceTO.isHistorical)
                    {
                        //if the note is for a new visit, but not hisorical, call getLocations and have the user pick one. 
                        //Then encounterString = location.id + ‘;’ + 20080515.1500 (today's date) + ‘;A’ 

                        encounterString = mssrServiceTO.SelectedLocationID + ";" + FormatDateForMDWS(DateTime.Now) + ";A";
                    }
                    else
                    {
                        //if the note is for a new visit, and is historical, 
                        //then previous step but end with ‘;E’ instead of ‘;A’. 

                        encounterString = mssrServiceTO.SelectedLocationID + ";" + FormatDateForMDWS(DateTime.Now) + ";E";

                    }

                    break;
            }

            //For preserving line breaks within a text block that is being written to a note, replace the \r\n or \n characters with a pipe (|).
            noteText = mssrServiceTO.NoteText.Replace("<br>", "|").Replace(Environment.NewLine, "|");

            authorDUZ = mssrServiceTO.DUZ;

            if (!String.IsNullOrEmpty(mssrServiceTO.SelectedConsultId))
            {
                consultIEN = mssrServiceTO.SelectedConsultId;
            }

            if (mssrServiceTO.SelectedNoteTitleId != "" && mssrServiceTO.SelectedNoteTitleId != null)
            {
                titleIEN = mssrServiceTO.SelectedNoteTitleId;
            }

            if (!String.IsNullOrEmpty(mssrServiceTO.SelectedCosignerID))
            {
                cosignerDUZ = mssrServiceTO.SelectedCosignerID;
            }

            prfIEN = "";

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            bool savedFlag = false;

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "writeNote";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { titleIEN, encounterString, noteText, authorDUZ, cosignerDUZ, consultIEN, prfIEN });

            try
            {
                _writeNoteResult = _mssrService.writeNote(titleIEN, encounterString, noteText, authorDUZ, cosignerDUZ, consultIEN, prfIEN);
                if (_writeNoteResult != null)
                {
                    _mdwsLog.RETURNED_DATA = _writeNoteResult.Serialize();

                    if (!String.IsNullOrEmpty(_writeNoteResult.id))
                    {
                        mdwsLog.LogMDWSCall(_mdwsLog);
                        savedFlag = true;

                        CloseNote(_writeNoteResult.id, consultIEN);

                        DisconnectSite(mssrServiceTO);

                    }

                    if (_writeNoteResult.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in writeNote: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            if (!savedFlag)
                mdwsLog.LogMDWSCall(_mdwsLog);

            return _writeNoteResult;
        }

        /// <summary>
        /// MDWS: CloseNote service call
        /// </summary>
        /// <param name="noteIEN"></param>
        /// <param name="consultIEN"></param>
        /// <returns></returns>
        private TextTO CloseNote(string noteIEN, string consultIEN)
        {
            TextTO result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "closeNote";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { noteIEN, consultIEN });

            try
            {
                result = _mssrService.closeNote(noteIEN, consultIEN);
                if (result != null)
                {
                    _mdwsLog.RETURNED_DATA = result.Serialize();

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in CloseNote: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// MDWS: isCosignerRequired service call
        /// </summary>
        /// <param name="mssrServiceTO"></param>
        /// <param name="errMsg"></param>
        /// <returns></returns>
        public bool isCosignerRequired(MSSRServiceTO mssrServiceTO, out string errMsg)
        {
            bool ret = false;
            TextTO result = null;

            MDWSLog mdwsLog = new MDWSLog();
            MDWS_LOG _mdwsLog = new MDWS_LOG();

            string noteDefinitionIEN = "";
            string authorDUZ = "";

            errMsg = "Service isCosignerRequired returned NULL value"; //Default error message

            noteDefinitionIEN = mssrServiceTO.SelectedNoteTitleId;
            authorDUZ = mssrServiceTO.DUZ;

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "isCosignerRequired";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { noteDefinitionIEN, authorDUZ });

            try
            {
                result = _mssrService.isCosignerRequired(noteDefinitionIEN, authorDUZ);
                if (result != null)
                {
                    _mdwsLog.RETURNED_DATA = result.Serialize();

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        errMsg = result.fault.message;
                    }
                    else
                    {
                        errMsg = String.Empty;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in isCosignerRequired: " + exc.Message);

                Console.Write(exc.Message);
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            if (!String.IsNullOrEmpty(result.text))
            {
                switch (result.text.ToUpper())
                {
                    case "Y":
                        ret = true;
                        break;
                    case "N":
                        break;
                    default:
                        errMsg = "Service isCosignerRequired returned uexpected value " + result.text;
                        break;
                }
            }

            return ret;
            //return true; //Test data
        }

        #endregion

        #region VIA Service Methods

        /// <summary>
        /// VIA: LoadMSSR_VIAService - Load VIA Service Method
        /// </summary>
        /// <param name="mssrServiceTo"></param>
        /// <returns></returns>
        public bool LoadMSSR_VIAService(MSSRServiceTO mssrServiceTo)
        {
            _VIAService = new TbiServiceInterfaceClient("TbiServiceImplPort");

            this.MssrServiceTo = mssrServiceTo;

            if (!string.IsNullOrEmpty(StationDivision))
            {
                _loginSiteCode = StationDivision;
            }
            else if (!String.IsNullOrEmpty(mssrServiceTo.siteCode))
            {
                _loginSiteCode = mssrServiceTo.siteCode;
            }

            _personsTOVIA = CPRSLaunchVIA(mssrServiceTo);

            if (_personsTOVIA == null)
            {
                this.MssrServiceTo = null;
            }

            if (this.MssrServiceTo != null)
            {
                if (_personsTOVIA != null)
                {
                    this.MssrServiceTo.StationNumber = _loginSiteCode;
                    this.MssrServiceTo.PatientName = _personsTOVIA.patient.patientName;
                    this.MssrServiceTo.PatientDOB = _personsTOVIA.patient.dob;

                    if (!String.IsNullOrEmpty(_personsTOVIA.patient.deceasedDate))
                        this.MssrServiceTo.PatientDOD = _personsTOVIA.patient.deceasedDate;

                    this.MssrServiceTo.PatientGender = _personsTOVIA.patient.gender;
                    this.MssrServiceTo.PatientSoN = _personsTOVIA.patient.ssn;
                    this.MssrServiceTo.PatientICN = _personsTOVIA.patient.mpiPid;
                    this.MssrServiceTo.PatientDFN = _personsTOVIA.patient.localPid;
                    this.MssrServiceTo.PatientAdmintTimeStamp = _personsTOVIA.patient.admitTimestamp;

                    if (_personsTOVIA.patient.age != null)
                        this.MssrServiceTo.PatientAge = _personsTOVIA.patient.age.ToString();
                }

                if (_personsTOVIA.user != null)
                {
                    this.MssrServiceTo.STA6N = _personsTOVIA.user.siteId;
                    this.MssrServiceTo.UserName = _personsTOVIA.user.name;
                    _providerName = _personsTOVIA.user.name;
                    _tokenizedDUZ = _personsTOVIA.user.DUZ;

                    if (Helpers.DeSerializSessionItems(HttpContext.Current.Session["TokenizedDUZ"]) == null)
                    {
                        HttpContext.Current.Session["TokenizedDUZ"] = Helpers.SerializSessionItemsObj(_tokenizedDUZ);
                    }
                    if (Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderName"]) == null)
                    {
                        HttpContext.Current.Session["ProviderName"] = Helpers.SerializSessionItemsObj(_providerName);
                    }
                }

                if (_personsTOVIA.patient.location != null)
                {
                    this.MssrServiceTo.PatientLocationId = _personsTOVIA.patient.location.id;

                    if (_personsTOVIA.patient.location.facility != null)
                    {
                        this.MssrServiceTo.InstitutionName = _personsTOVIA.patient.location.facility.name;
                    }
                }
            }

            if (this.MssrServiceTo == null)
            {
                return false;
            }
            else
            {
                return true;
            }
        }

        /// <summary>
        /// VIA: CPRSLaunchVIA service method
        /// </summary>
        /// <param name="tbiServiceTo"></param>
        /// <returns></returns>
        private personsTO CPRSLaunchVIA(MSSRServiceTO mssrServiceTo)
        {
            try
            {
                personsTO result = CPRSLaunchVIAWithError(mssrServiceTo);
                return result;
            }
            catch (Exception x)
            {
                Logging.WriteLogToFile(x.ToString());
            }
            return null;
        }

        private personsTO CPRSLaunchVIAWithError(MSSRServiceTO mssrServiceTo)
        {
            Exception x = null;
            personsTO result = null;

            try
            {
                provider _providerbean = new provider();
                patient _patientbean = new patient();
                queryBean _bean = new queryBean();

                _userID = mssrServiceTo.DUZ;
                _localPid = mssrServiceTo.DFN;
                HttpContext.Current.Session["ProviderDFN"] = Helpers.SerializSessionItemsObj(_localPid);

                _bean.criteria = "TBI_PWD";
                _bean.requestingApp = Helpers.GetDBConfigEntry("MSSRServiceRequestingApp"); //_bean.requestingApp = "TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("MSSRServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("MSSRServiceCred2");

                //*** DO NOT OVERRIDE THE _loginSiteCode IN THIS METHOD

                //patient attributes
                _patientbean.localSiteId = _loginSiteCode;
                _patientbean.localPid = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderDFN"]).ToString();
                _bean.patient = _patientbean;
                //provider attributes
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = _userID;
                _bean.provider = _providerbean;

                // log so we know what we are doing ....
                 Logging.WriteLogToFile("LaunchCprs: MSSR InstrumentManager: CPRSLaunchVIA Start:");

                _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                _mdwsLog.SERVICE_NAME = "LaunchCprs";
                _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { "TBI-PWD", mssrServiceTo.siteCode, mssrServiceTo.DUZ, mssrServiceTo.DFN });

                result = _VIAService.cprsLaunch(_bean);

                if (result != null && result.fault == null)
                {
                    _mdwsLog.ERROR_LEVEL = 0;//success
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 1;//failed service launch

                    if (result.fault != null)

                        Logging.WriteLogToFile("MSSR InstrumentManager: CPRSLaunchVIA failure: " + result.fault.message);
                }
            }
            catch (Exception ex)
            {
                x = ex;
                Logging.WriteLogToFile("Failure in CPRSLaunchVIA: " + ex.Message);

                string testerror = ex.Message.ToString();
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            if (null != x)
            {
                Logging.WriteLogToFile(x.ToString());
                throw x;
            }

            return result;

        }

        /// <summary>
        /// VIA: getNoteTitlesFromVIA service method
        /// </summary>
        /// <param name="target"></param>
        /// <param name="direction"></param>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getNoteTitlesFromVIA(string target, string direction, out string error)
        {
            taggedTextArray ret = null;
            List<KeyValuePair<string, string>> result = null;

            string searchParameter = String.Empty;

            if (target.Trim() == String.Empty)
            {
                searchParameter = "T";//defaults to tbi note get
            }
            else
            {
                searchParameter = target.Trim().ToUpper();
            }

            try
            {
                if (_VIAService != null)
                {
                    queryBean _bean = new queryBean();
                    patient _patientbean = new patient();
                    provider _providerbean = new provider();

                    if (!string.IsNullOrEmpty(StationDivision))
                        this.MssrServiceTo.StationNumber = StationDivision;
                    _loginSiteCode = this.MssrServiceTo.StationNumber;

                    _bean.target = searchParameter;
                    _bean.direction = "1";

                    _bean.requestingApp = Helpers.GetDBConfigEntry("MSSRServiceRequestingApp");
                    _bean.consumingAppToken = Helpers.GetDBConfigEntry("MSSRServiceCred1");
                    _bean.consumingAppPassword = Helpers.GetDBConfigEntry("MSSRServiceCred2");

                    _patientbean.localSiteId = _loginSiteCode;
                    _patientbean.localPid = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderDFN"]).ToString();
                    _providerbean.name = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderName"]).ToString();
                    _providerbean.userId = Helpers.DeSerializSessionItems(HttpContext.Current.Session["TokenizedDUZ"]).ToString();
                    _providerbean.loginSiteCode = _loginSiteCode;
                    _bean.patient = _patientbean;
                    _bean.provider = _providerbean;

                    ret = _VIAService.getNoteTitles(_bean);

                    if (ret != null)
                    {
                        if (ret.fault != null)
                        {
                            error = ret.fault.message;
                            result = null;

                            _mdwsLog.ERROR_LEVEL = 1;
                        }
                        else
                        {
                            if (ret.count > 0)
                            {
                                _mdwsLog.ERROR_LEVEL = 0;
                                error = "SUCCESS";
                                result = (from e in ret.results.FirstOrDefault().taggedResults orderby e.text select new KeyValuePair<string, string>(e.tag, e.textArray[0])).ToList();
                            }
                            else
                            {
                                _mdwsLog.ERROR_LEVEL = 1;
                                error = null;
                            }
                        }
                    }
                    else
                    {
                        error = error = "VIA Service call get Note Titles returned null";
                        _mdwsLog.ERROR_LEVEL = 2;
                    }
                }
                else
                {
                    error = null;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in getNoteTitlesFromVia: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;

        }

        /// <summary>
        /// VIA: getConsultsFromVIA service method
        /// </summary>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getConsultsFromVIA(out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            taggedConsultArrays ret = null;

            try
            {
                if (_VIAService != null)
                {
                    queryBean _bean = new queryBean();
                    patient _patientbean = new patient();
                    provider _providerbean = new provider();

                    if (!string.IsNullOrEmpty(StationDivision))
                        this.MssrServiceTo.StationNumber = StationDivision;
                    _loginSiteCode = this.MssrServiceTo.StationNumber;

                    _bean.requestingApp = Helpers.GetDBConfigEntry("MSSRServiceRequestingApp");
                    _bean.consumingAppToken = Helpers.GetDBConfigEntry("MSSRServiceCred1");
                    _bean.consumingAppPassword = Helpers.GetDBConfigEntry("MSSRServiceCred2");

                    _patientbean.localPid = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderDFN"]).ToString();
                    _patientbean.localSiteId = _loginSiteCode;
                    _providerbean.name = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderName"]).ToString();
                    _providerbean.userId = Helpers.DeSerializSessionItems(HttpContext.Current.Session["TokenizedDUZ"]).ToString();
                    _providerbean.loginSiteCode = _loginSiteCode;
                    _bean.patient = _patientbean;
                    _bean.provider = _providerbean;

                    ret = _VIAService.getConsultsForPatient(_bean);

                    if (ret != null)
                    {
                        if (ret.fault != null)
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                            error = ret.fault.message;
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            if (!(ret.count == 0 || ret.arrays[0].consults == null))
                            {
                                var consultsArray = ret.arrays.SelectMany(s => s.consults).ToList();
                                result = (from e in consultsArray orderby e.timestamp descending select new KeyValuePair<string, string>(e.id, DateTime.ParseExact(e.timestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("MMM dd,yy") + " (" + e.status + ") " + e.title + " " + "Consult #: " + e.id)).ToList();
                            }
                            else
                            {
                                _mdwsLog.ERROR_LEVEL = 1;
                                error = null;
                            }
                        }
                    }
                    else
                    {
                        error = "VIA Service call get Consults returned null";
                        _mdwsLog.ERROR_LEVEL = 2;
                    }
                }
                else
                {
                    error = null;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in getConsultsFromVIA: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// VIA: getHospitalAdmissionsFromVIA service call
        /// </summary>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getHospitalAdmissionsFromVIA(out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            taggedInpatientStayArray ret = null;

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                if (!string.IsNullOrEmpty(StationDivision))
                    this.MssrServiceTo.StationNumber = StationDivision;
                _loginSiteCode = this.MssrServiceTo.StationNumber;

                _patientbean.localPid = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderDFN"]).ToString();
                _patientbean.localSiteId = _loginSiteCode;
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.name = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderName"]).ToString();
                _providerbean.userId = Helpers.DeSerializSessionItems(HttpContext.Current.Session["TokenizedDUZ"]).ToString();
                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                _bean.requestingApp = Helpers.GetDBConfigEntry("MSSRServiceRequestingApp");
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("MSSRServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("MSSRServiceCred2");

                ret = _VIAService.getAdmissions(_bean);
                if (ret != null)
                {
                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        if (!(ret.count == 0 || ret.stays == null))
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            result = (from e in ret.stays orderby e.admitTimestamp descending select new KeyValuePair<string, string>(e.location.id + ";" + e.admitTimestamp + ";H", e.location.name + " (" + DateTime.ParseExact(e.admitTimestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("MMM dd,yy") + ")")).ToList();
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                            error = null;
                        }
                    }
                }
                else
                {
                    error = "VIA Service call get Hospital Admissions returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in getHospitalAdmissionsFromVIA: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// VIA: getVisitsFromVIA service call
        /// </summary>
        /// <param name="fromDate"></param>
        /// <param name="toDate"></param>
        /// <param name="error"></param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getVisitsFromVIA(DateTime fromDate, DateTime toDate, out string error)
        {
            taggedVisitArray ret = null;
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();

            string formattedFromDate = FormatDateForVIA(fromDate, true);
            string formattedToDate = FormatDateForVIA(toDate, true);

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                if (!string.IsNullOrEmpty(StationDivision))
                    this.MssrServiceTo.StationNumber = StationDivision;
                _loginSiteCode = this.MssrServiceTo.StationNumber;

                _bean.recordSiteCode = _loginSiteCode;

                _bean.requestingApp = Helpers.GetDBConfigEntry("MSSRServiceRequestingApp");
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("MSSRServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("MSSRServiceCred2");

                _bean.startDate = formattedFromDate;
                _bean.endDate = formattedToDate;
                _patientbean.localPid = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderDFN"]).ToString();
                _patientbean.localSiteId = _loginSiteCode;
                _providerbean.name = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderName"]).ToString();
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = Helpers.DeSerializSessionItems(HttpContext.Current.Session["TokenizedDUZ"]).ToString();

                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                ret = _VIAService.getVisits(_bean);

                if (ret != null)
                {
                    if (ret.fault != null)
                    {
                        error = ret.fault.message;
                        result = null;

                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                    else
                    {
                        if (ret.count > 0)
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            result = (from e in ret.visits orderby e.timestamp descending select new KeyValuePair<string, string>(e.location.id + ";" + e.timestamp + ";A", DateTime.ParseExact(e.timestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("g") + " " + e.location.name + " " + e.status)).ToList();
                        }
                        else
                        {
                            error = null;
                            _mdwsLog.ERROR_LEVEL = 1;
                        }
                    }
                }
                else
                {
                    error = "VIA Service call get Visits returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in getVistsFromVIA: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// This will get the list of CPRS users for the correct VIA connection
        /// </summary>
        /// <param name="target">Beginning of first user name in the list</param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> cprsUserLookupFromVIA(string target, out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            userArray ret = null;

            try
            {
                queryBean _bean = new queryBean();
                provider _providerbean = new provider();

                target = MssrServiceTo.PatientName;
                if (!string.IsNullOrEmpty(StationDivision))
                    this.MssrServiceTo.StationNumber = StationDivision;
                _loginSiteCode = MssrServiceTo.StationNumber;

                _bean.requestingApp = Helpers.GetDBConfigEntry("MSSRServiceRequestingApp");
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("MSSRServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("MSSRServiceCred2");

                _bean.target = target;
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = Helpers.DeSerializSessionItems(HttpContext.Current.Session["TokenizedDUZ"]).ToString();
                _bean.provider = _providerbean;

                ret = _VIAService.cprsUserLookup(_bean);

                if (ret != null)
                {
                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        if (!(ret.count == 0 || ret.users == null))
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            result = (from e in ret.users orderby e.name select new KeyValuePair<string, string>(e.DUZ, e.name)).ToList();
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                            error = null;
                        }
                    }
                }
                else
                {
                    error = error = "VIA Service call CPRS User Lookup returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in cprsUserLookupFromVIA: " + exc.Message);
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// This will get the locations for the correct VIA connection
        /// </summary>
        /// <param name="filter">Use this to filter the results returned from MDWS</param>
        /// <param name="target">Purpose and expected values Unknown </param>
        /// <param name="direction">MDWS expects a value of 1 or -1</param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getLocationsFromVIA(string filter, string target, string direction, out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            taggedHospitalLocationArray ret = null;
            string searchParameter = String.Empty;

            if (filter.Trim() == String.Empty)
            {
                searchParameter = target;
            }
            else
            {
                searchParameter = filter.Trim().ToUpper();
            }

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                if (!string.IsNullOrEmpty(StationDivision))
                    this.MssrServiceTo.StationNumber = StationDivision;
                _loginSiteCode = MssrServiceTo.StationNumber;
                _bean.target = searchParameter;

                _bean.requestingApp = Helpers.GetDBConfigEntry("MSSRServiceRequestingApp");
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("MSSRServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("MSSRServiceCred2");

                _bean.direction = direction;
                _patientbean.localSiteId = _loginSiteCode;
                _providerbean.loginSiteCode = _loginSiteCode;
                _patientbean.localPid = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderDFN"]).ToString();
                _providerbean.name = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderName"]).ToString();
                _providerbean.userId = Helpers.DeSerializSessionItems(HttpContext.Current.Session["TokenizedDUZ"]).ToString();
                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                ret = _VIAService.getLocations(_bean);

                if (ret != null)
                {
                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        if (!(ret.count == 0 || ret.locations == null))
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            result = (from e in ret.locations where e.name.ToLower().StartsWith(filter.ToLower()) orderby e.name select new KeyValuePair<string, string>(e.id, e.name)).ToList();
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                            error = null;
                        }
                    }
                }
                else
                {
                    error = error = "VIA Service call get Locations returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in getLocationsFromVIA: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// VIA: writeNoteWithVIA write note method
        /// </summary>
        /// <param name="tbiServiceTO"></param>
        /// <returns></returns>
        public noteResultTO writeNoteWithVIA(MSSRServiceTO mssrServiceTO)
        {
            noteResultTO _writeNoteResult = new noteResultTO();
            textTO _closeNoteResult = null;
            string titleIEN = "";
            string encounterString = "";
            string noteText = "";
            string authorDUZ = "";
            string cosignerDUZ = "";
            string consultIEN = "";
            string prfIEN = "";

            //set the values for the writeNote web service based on the noteType
            switch (mssrServiceTO.SelectedNoteType.Id)
            {
                case (int)NoteTypeEnum.ScheduledClinicAppointment:  //Scheduled Clinic Appointment
                    Logging.WriteLogToFile("Encounter Type: ScheduledClinicAppointment");
                    //if the note is for current visit or previous visit, call getVisits and have the user pick one . 
                    //Then encounterString = visit.location.id + ‘;’ + visit.timestamp + ‘;A’ 

                    //already formated from the original call to MDWS

                    // VIA does not like the timestamp in the date:
                    //encounterString = "1691;3150211;A"
                    //message = "Invalid vista timestamp: The 2nd part of the 3 part, semicolon-separated encounter string 
                    // must have a format of [yyyy -1700]MMdd.HHmmss.  For example, 3130715.163242 represents the date and 
                    // time of JUL 15, 2013 at 16:32:42"

                    //string[] s = mssrServiceTO.SelectedVisitId.Split(';');

                    //if (s[1].Contains("."))
                    //{
                    //    string[] d = s[1].Split('.');
                    //    int year = Convert.ToInt32(d[0].Substring(0,4)); // Year is the first four chars
                    //    year = year - 1700;

                    //    d[0] = year.ToString() + d[0].Substring(4, d[0].Length - 4);// d[0].Replace(d[0].Substring(0, 4), year.ToString()); // Replace Year with Year - 1700

                    //    s[1] = string.Join(".", d);
                    //}

                    //encounterString = string.Join(";", s);

                    encounterString = FormatDateForVIA(mssrServiceTO.SelectedVisitId);

                    break;
                case (int)NoteTypeEnum.HospitalAdmission:  //Hospital Admission
                    Logging.WriteLogToFile("Encounter Type: HospitalAdmission");
                    //if the note is for previous stay, call getAdmissions and have the user pick one. 
                    //Then encounterString = stay.location.id + ‘;’ + stay.admitTimestamp + ‘;H’ 

                    //already formated from the original call to MDWS
                    encounterString = FormatDateForVIA(mssrServiceTO.SelectedHospitalAdmissionId);
                    break;
                case (int)NoteTypeEnum.CurrentStay:  //Current Stay
                    Logging.WriteLogToFile("Encounter Type: CurrentStay");
                    //if the note is for current stay, encounterString = patient.locationId + ‘;’ + patient.admitTimestamp + ‘;H’ 
                    //encounterString = mssrServiceTO.PatientLocationId + ";" + mssrServiceTO.PatientAdmintTimeStamp + ";H";
                    encounterString = mssrServiceTO.PatientLocationId + ";" + mssrServiceTO.PatientAdmintTimeStamp + ";H";
                    encounterString = FormatDateForVIA(encounterString);
                    break;
                case (int)NoteTypeEnum.UnscheduledOrNewVisit: //Unscheduled or New Visit
                    Logging.WriteLogToFile("Encounter Type: UnscheduledOrNewVisit");

                    if (!mssrServiceTO.isHistorical)
                    {
                        //if the note is for a new visit, but not hisorical, call getLocations and have the user pick one. 
                        //Then encounterString = location.id + ‘;’ + 20080515.1500 (today's date) + ‘;A’ 
                        encounterString = mssrServiceTO.SelectedLocationID + ";" + FormatDateForVIA(DateTime.Now, false) + ";A";
                    }
                    else
                    {
                        //if the note is for a new visit, and is historical, 
                        //then previous step but end with ‘;E’ instead of ‘;A’. 
                        encounterString = mssrServiceTO.SelectedLocationID + ";" + FormatDateForVIA(DateTime.Now, false) + ";E";
                    }
                    break;
            }

            //For preserving line breaks within a text block that is being written to a note, replace the \r\n or \n characters with a pipe (|).
            noteText = mssrServiceTO.NoteText.Replace("<br>", "|").Replace(Environment.NewLine, "|");

            authorDUZ = mssrServiceTO.DUZ;

            if (!String.IsNullOrEmpty(mssrServiceTO.SelectedConsultId))
            {
                consultIEN = mssrServiceTO.SelectedConsultId;
            }

            if (mssrServiceTO.SelectedNoteTitleId != "" && mssrServiceTO.SelectedNoteTitleId != null)
            {
                titleIEN = mssrServiceTO.SelectedNoteTitleId;
            }

            if (!String.IsNullOrEmpty(mssrServiceTO.SelectedCosignerID))
            {
                cosignerDUZ = mssrServiceTO.SelectedCosignerID;
            }

            prfIEN = "";

            bool savedFlag = false;

            //_cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            //_mdwsLog.SERVICE_NAME = "writeNote-VIA";
            //_mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { titleIEN, encounterString, noteText, authorDUZ, cosignerDUZ, consultIEN, prfIEN });

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                if (!string.IsNullOrEmpty(StationDivision))
                    this.MssrServiceTo.StationNumber = StationDivision;
                _loginSiteCode = this.MssrServiceTo.StationNumber;

                _bean.requestingApp = Helpers.GetDBConfigEntry("MSSRServiceRequestingApp");
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("MSSRServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("MSSRServiceCred2");

                _patientbean.localPid = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderDFN"]).ToString();
                //_writenote1.queryBean.patient.localSiteId = _loginSiteCode;
                //_writenote1.queryBean.patient.vistaLocations = _loginSiteCode;
                _providerbean.loginSiteCode = _loginSiteCode;
                //_writenote1.queryBean.provider.name = HttpContext.Current.Session["ProviderName"].ToString();
                _providerbean.userId = Helpers.DeSerializSessionItems(HttpContext.Current.Session["TokenizedDUZ"]).ToString();

                //_bean = _writenote1.queryBean;
                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                _writeNoteResult = _VIAService.writeNote(titleIEN, encounterString, noteText, authorDUZ, cosignerDUZ, consultIEN, prfIEN, _bean);

                if (_writeNoteResult != null)
                {
                    if (!String.IsNullOrEmpty(_writeNoteResult.id))
                    {
                        savedFlag = true;
                        CloseNoteWithVIA(titleIEN, consultIEN);
                    }

                    if (_writeNoteResult.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in writeNoteWithVIA: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            if (!savedFlag)
                mdwsLog.LogMDWSCall(_mdwsLog);

            return _writeNoteResult;
        }

        /// <summary>
        /// VIA: CloseNote service method call
        /// </summary>
        /// <param name="noteIEN"></param>
        /// <param name="consultIEN"></param>
        /// <returns></returns>
        private textTO CloseNoteWithVIA(string noteIEN, string consultIEN)
        {
            textTO result = null;

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();
                if (!string.IsNullOrEmpty(StationDivision))
                    this.MssrServiceTo.StationNumber = StationDivision;
                _loginSiteCode = this.MssrServiceTo.StationNumber;

                _bean.requestingApp = Helpers.GetDBConfigEntry("MSSRServiceRequestingApp");
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("MSSRServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("MSSRServiceCred2");

                _bean.recordSiteCode = _loginSiteCode;
                _providerbean.userId = Helpers.DeSerializSessionItems(HttpContext.Current.Session["TokenizedDUZ"]).ToString();
                _bean.provider = _providerbean;

                result = _VIAService.closeNote(noteIEN, consultIEN, _bean);

                if (result != null)
                {
                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 0;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 1;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in CloseNoteWithVIA: " + exc.Message);

                _mdwsLog.ERROR_LEVEL = 2;
            }

            return result;
        }

        /// <summary>
        /// VIA: isCosignerRequired service method call
        /// </summary>
        /// <param name="mssrServiceTO"></param>
        /// <param name="errMsg"></param>
        /// <returns></returns>
        public bool isCosignerRequiredVIA(MSSRServiceTO mssrServiceTO, out string errMsg)
        {
            bool ret = false;
            textTO result = null;
            string noteDefinitionIEN = "";
            string authorDUZ = "";

            errMsg = "Service isCosignerRequiredVIA returned NULL value"; //Default error message

            noteDefinitionIEN = mssrServiceTO.SelectedNoteTitleId;
            authorDUZ = mssrServiceTO.DUZ;
            if (!string.IsNullOrEmpty(StationDivision))
                mssrServiceTO.StationNumber = StationDivision;
            _loginSiteCode = mssrServiceTO.StationNumber;

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                _bean.itemId = noteDefinitionIEN;
                _bean.recordSiteCode = _loginSiteCode;

                _bean.requestingApp = Helpers.GetDBConfigEntry("MSSRServiceRequestingApp");
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("MSSRServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("MSSRServiceCred2");

                _providerbean.name = Helpers.DeSerializSessionItems(HttpContext.Current.Session["ProviderName"]).ToString();
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = Helpers.DeSerializSessionItems(HttpContext.Current.Session["TokenizedDUZ"]).ToString();
                _bean.criteria = authorDUZ;
                _patientbean.mpiPid = mssrServiceTO.PatientICN;

                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                result = _VIAService.isCosignerRequired(_bean);

                if (result != null)
                {
                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        errMsg = result.fault.message;
                    }
                    else
                    {
                        _mdwsLog.ERROR_LEVEL = 0;
                        errMsg = String.Empty;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Logging.WriteLogToFile("Failure in isCosignerRequiredVIA: " + exc.Message);

                Console.Write(exc.Message);
                _mdwsLog.ERROR_LEVEL = 2;
            }

            if (!String.IsNullOrEmpty(result.text))
            {
                switch (result.text.ToUpper())
                {
                    case "Y":
                        ret = true;
                        break;
                    case "N":
                        break;
                    default:
                        errMsg = "VIA Service isCosignerRequiredVIA returned uexpected value " + result.text;
                        break;
                }
            }

            return ret;
        }

        /// <summary>
        /// VIA: FormatDateForVIA date formatting
        /// </summary>
        /// <param name="dateValue"></param>
        /// <returns></returns>
        private string FormatDateForVIA(DateTime dateValue, bool numeric)
        {
            // CPRS is expecting date in the following format "20000101.000000";
           
            if (dateValue != null)
            {
                if (numeric)
                {
                    return String.Format("{0:yyyyMMdd}", dateValue);
                }
                else
                {
                    return (dateValue.Year - 1700) + "" + dateValue.ToString("MM") + dateValue.ToString("dd"); // +"." + dateValue.ToString("HH") + dateValue.ToString("mm") + dateValue.ToString("ss");
                }
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// VIA: FormatDateForVIA date formatting
        /// </summary>
        /// <param name="dateValue"></param>
        /// <returns></returns>
        private string FormatDateForVIA(string sDate)
        {
            string[] s = sDate.Split(';');

            if (s[1].Contains("."))
            {
                string[] d = s[1].Split('.');
                int year = Convert.ToInt32(d[0].Substring(0, 4)); // Year is the first four chars
                year = year - 1700;

                d[0] = year.ToString() + d[0].Substring(4, d[0].Length - 4);// d[0].Replace(d[0].Substring(0, 4), year.ToString()); // Replace Year with Year - 1700

                s[1] = string.Join(".", d);
            }

            return string.Join(";", s); 
        }

        #endregion

        #region LINQ / Utility Methods

        /// <summary>
        /// Method used to retrieve all "active" instruments from the database.  Replaces legacy systems usage of
        /// TbiServiceTo.Instruments (static) list.
        /// </summary>
        /// <returns></returns>
        public List<InstrumentType> GetActiveInstrumentList()
        {
            using (_db = GetDataContext())
            {
                // default to TBI if nothing passed.
                int tbiRegistryId = _db.STD_REGISTRies.FirstOrDefault(R => R.CODE == "MSSR").ID;
                return convertStdSurveyTypeListToInstrumentType(_db.STD_SURVEY_TYPEs.Where(S => S.REGISTRY_ID == tbiRegistryId && S.IS_ACTIVE == true).ToList());
            }
        }

        /// <summary>
        /// Method used to retrieve all instruments from the database.  Replaces legacy systems usage of
        /// TbiServiceTo.Instruments (static) list.
        /// </summary>
        /// <returns></returns>
        public List<InstrumentType> GetAllInstrumentList()
        {
            using (_db = GetDataContext())
            {
                int tbiRegistryId = _db.STD_REGISTRies.FirstOrDefault(R => R.CODE == "MSSR").ID;
                return convertStdSurveyTypeListToInstrumentType(_db.STD_SURVEY_TYPEs.Where(S => S.REGISTRY_ID == tbiRegistryId).ToList());
            }
        }

        /// <summary>
        /// Set Instrument Active Status by ID
        /// </summary>
        /// <param name="surveyTypeId"></param>
        /// <param name="isActive"></param>
        public void SetInstrumentActiviationStatus(int surveyTypeId, bool isActive)
        {
            using (_db = GetDataContext())
            {
                STD_SURVEY_TYPE surveyType = _db.STD_SURVEY_TYPEs.FirstOrDefault(S => S.ID == surveyTypeId);
                if (surveyType != null)
                {
                    surveyType.IS_ACTIVE = isActive;

                    _db.SubmitChanges();
                }
            }
        }

        /// <summary>
        /// Returns the Instrument Status - is active.
        /// </summary>
        /// <param name="surveyTypeId"></param>
        /// <param name="isActive"></param>
        public bool IsInstrumentActive(int surveyTypeId)
        {
            using (_db = GetDataContext())
            {
                STD_SURVEY_TYPE surveyType = _db.STD_SURVEY_TYPEs.FirstOrDefault(S => S.ID == surveyTypeId);
                if (surveyType != null)
                {
                    return surveyType.IS_ACTIVE;
                }
            }

            return false;
        }

        /// <summary>
        /// SetLoadWith method - private
        /// </summary>
        /// <param name="db"></param>
        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<SURVEY>(e => e.SURVEY_RESULTs);
            lo.LoadWith<SURVEY_RESULT>(e => e.STD_QUESTION_CHOICE);
            lo.LoadWith<STD_QUESTION_CHOICE>(e => e.STD_QUESTION);
            lo.LoadWith<SURVEY>(e => e.STD_SURVEY_TYPE);
            lo.LoadWith<SURVEY>(e => e.SURVEY_NOTEs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        /// <summary>
        /// SetSurveyLoadWith private method
        /// </summary>
        /// <param name="db"></param>
        private void SetSurveyLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<SURVEY>(e => e.SURVEY_RESULTs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        /// <summary>
        /// SetQuestionnaireLoadWith private method
        /// </summary>
        /// <param name="db"></param>
        private void SetQuestionnaireLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<SURVEY_RESULT>(e => e.SURVEY);
            lo.LoadWith<SURVEY_RESULT>(e => e.STD_QUESTION_CHOICE);
            lo.LoadWith<STD_QUESTION>(e => e.STD_QUESTION_CHOICEs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        /// <summary>
        /// SetQuestionsLoadWith private method
        /// </summary>
        /// <param name="db"></param>
        private void SetQuestionsLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<STD_QUESTION>(e => e.STD_QUESTION_CHOICEs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        /// <summary>
        /// LinqAll
        /// </summary>
        /// <returns></returns>
        private IQueryable<SURVEY> LinqAll()
        {
            //populate LinqAll
            IQueryable<SURVEY> tempLinq = (from e in _db.SURVEYs
                                           select e);

            //TODO - add all business filtering rules 

            return tempLinq;
        }

        /// <summary>
        /// SelectByPatientICNFilter
        /// </summary>
        /// <param name="ICN"></param>
        /// <returns></returns>
        private IQueryable<SURVEY> SelectByPatientICNFilter(string ICN)
        {
            IQueryable<SURVEY> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.patient_ICN == ICN select t;

            return linqFilter;
        }

        /// <summary>
        /// Get a survey by a surveyID.
        /// Return a survey object with all the survey results
        /// </summary>
        /// <param name="surveyId"></param>
        /// <returns></returns>
        public SURVEY GetSurveyWithAllAnswers(int surveyId)
        {
            using (_db = GetDataContext())
            {
                SetSurveyLoadWith(_db);

                var survey = (from t in _db.SURVEYs
                              where t.SURVEYS_ID == surveyId
                              select t).FirstOrDefault();
                return survey;
            }
        }

        /// <summary>
        /// GetChoicesByQuestionNumber - returns a list of survey results for question number
        /// </summary>
        /// <param name="questionNumber"></param>
        /// <param name="surveyId"></param>
        /// <returns></returns>
        public List<SURVEY_RESULT> GetChoicesByQuestionNumber(string questionNumber, int surveyId)
        {
            using (_db = GetDataContext())
            {
                SetQuestionnaireLoadWith(_db);

                var choices = (from t in _db.SURVEY_RESULTs
                               where t.SURVEYS_ID == surveyId
                               where t.STD_QUESTION_CHOICE.STD_QUESTION.QUESTION_NUMBER == questionNumber
                               select t).ToList();

                return choices;
            }

        }

        /// <summary>
        /// GetQuestionbyNumberAndSurveyType
        /// </summary>
        /// <param name="questionNumber"></param>
        /// <param name="surveyTypeId"></param>
        /// <returns></returns>
        public string GetQuestionbyNumberAndSurveyType(string questionNumber, int surveyTypeId)
        {
            using (_db = GetDataContext())
            {
                string ret = "";

                SetQuestionnaireLoadWith(_db);

                var question = (from t in _db.STD_QUESTIONs
                                where t.STD_SURVEY_TYPE_ID == surveyTypeId
                                where t.QUESTION_NUMBER == questionNumber
                                select t).ToList();

                if (question.Count() > 0)
                {
                    ret = question.First().QUESTION_TEXT;
                }

                return ret;
            }

        }

        /// <summary>
        /// IsInitialNoteType
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="surveyType"></param>
        /// <returns></returns>
        public bool IsInitialNoteType(string patientICN, int surveyType)
        {
            bool ret = false;

            //Get note type of latest completed survey
            string lastNoteType = GetLastCompletedNoteTypeByPatientICNAndSurveyType(patientICN, surveyType);

            if (String.IsNullOrEmpty(lastNoteType) ||
                String.Equals(lastNoteType, "Discharge", StringComparison.OrdinalIgnoreCase))
            {
                ret = true;
            }

            return ret;
        }

        /// <summary>
        /// GetMapiNoteTypeChoices
        /// </summary>
        /// <returns></returns>
        public List<STD_QUESTION_CHOICE> GetMapiNoteTypeChoices()
        {
            return GetChoicesByQuestionNumberandSurveyType("-1", "MPAI");
        }

        /// <summary>
        /// GetMapiPersonReportingChoices
        /// </summary>
        /// <returns></returns>
        public List<STD_QUESTION_CHOICE> GetMapiPersonReportingChoices()
        {
            return GetChoicesByQuestionNumberandSurveyType("0", "MPAI");
        }

        /// <summary>
        /// Returns the Instrument with the passed ID - updated for new database approach.
        /// </summary>
        /// <param name="instrumentID"></param>
        /// <returns></returns>
        public InstrumentType GetInstrumentByID(int instrumentID)
        {
            return this.GetAllInstrumentList().FirstOrDefault(I => I.Id == instrumentID);
        }

        /// <summary>
        /// GetChoicesByQuestionNumberandSurveyType
        /// </summary>
        /// <param name="questionNumber"></param>
        /// <param name="surveyTypeCode"></param>
        /// <returns></returns>
        public List<STD_QUESTION_CHOICE> GetChoicesByQuestionNumberandSurveyType(string questionNumber, string surveyTypeCode)
        {
            using (_db = GetDataContext())
            {
                SetQuestionsLoadWith(_db);

                var question = (from t in _db.STD_QUESTIONs
                                where t.STD_SURVEY_TYPE.CODE == surveyTypeCode
                                where t.QUESTION_NUMBER == questionNumber
                                select t).ToList();

                var choices = (from t in _db.STD_QUESTION_CHOICEs
                               where t.STD_QUESTION_ID == question.FirstOrDefault().ID
                               where t.INACTIVE_FLAG == false
                               select t).OrderBy("CHOICE_SORT_ORDER").ToList();

                return choices;
            }
        }

        /// <summary>
        /// SelectByPatientICNAndSurveyType
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <param name="followUp"></param>
        /// <returns></returns>
        public IEnumerable<SURVEY> SelectByPatientICNAndSurveyType(string patientICN, int typeId, bool followUp)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<SURVEY> surveys;

                surveys = from t in SelectByPatientICNFilter(patientICN)
                          where t.STD_SURVEY_TYPE_ID == typeId
                          select t;

                if (followUp)

                    surveys = from s in surveys where s.SURVEY_STATUS == "inprocess" select s;

                return surveys.ToList();
            }

        }

        /// <summary>
        /// SelectInProcessByPatientICNAndSurveyType: only queries for the inprocess surveys for a patient
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <param name="followUp"></param>
        /// <returns></returns>
        public IEnumerable<SURVEY> SelectInProcessByPatientICNAndSurveyType(string patientICN, int typeId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<SURVEY> surveys;

                surveys = from t in SelectByPatientICNFilter(patientICN)
                          where t.STD_SURVEY_TYPE_ID == typeId && t.SURVEY_STATUS == "inprocess"
                          select t;

                return surveys.ToList();
            }
        }

        public SURVEY SelectSurveyByPatientIcnAndSurveyType(string patientIcn, int surveyId, int typeId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                SURVEY survey;

                survey = (from t in SelectByPatientICNFilter(patientIcn)
                          where t.STD_SURVEY_TYPE_ID == typeId && t.SURVEYS_ID == surveyId && t.SURVEY_STATUS.ToLower() == "completed"
                    select t).FirstOrDefault();

                return survey;
            }


        }



        /// <summary>
        /// SelectLatestByPatientICNAndSurveyType: Returns the latest (most recent) survey for the given patient and survey type.  If null, no survey was found.
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <returns></returns>
        public SURVEY SelectLatestByPatientICNAndSurveyType(string patientICN, int typeId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                SURVEY survey = SelectByPatientICNFilter(patientICN).Where(T => T.STD_SURVEY_TYPE_ID == typeId).OrderByDescending(T => T.SURVEY_DATE).OrderByDescending(T => T.CREATED).FirstOrDefault();

                return survey;

                //SURVEY survey;

                //survey = (from t in SelectByPatientICNFilter(patientICN)
                //          where t.STD_SURVEY_TYPE_ID == typeId
                //          select t).Take(1).to;

                //return survey;
            }
        }


        /// <summary>
        /// SelectLatestCompletedSurvey: Returns the latest (most recent) complete survey for the given patient and survey type.  If null, no survey was found.
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <returns></returns>
        public SURVEY SelectLatestCompletedSurvey(string patientICN, int typeId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                SURVEY survey = SelectByPatientICNFilter(patientICN).Where(T => T.STD_SURVEY_TYPE_ID == typeId && T.SURVEY_STATUS.ToLower() == "completed").OrderByDescending(T => T.SURVEY_DATE).FirstOrDefault();

                return survey;
            }
        }

        /// <summary>
        /// SelectCountByPatientICNAndSurveyType: returns count of surveys for patient
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <param name="followUp"></param>
        /// <returns></returns>
        public int SelectCountByPatientICNAndSurveyType(string patientICN, int typeId, bool followUp)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<SURVEY> surveys;

                surveys = from t in SelectByPatientICNFilter(patientICN)
                          where t.STD_SURVEY_TYPE_ID == typeId
                          select t;

                if (followUp)
                    surveys = from s in surveys where s.SURVEY_STATUS == "inprocess" select s;

                return surveys.Count();
            }

        }

        /// <summary>
        /// SelectLatestCompletedByPatientICNAndSurveyType: Returns the latest (most recent) complete survey for the given patient and survey type.  If null, no survey was found.
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <returns></returns>
        public SURVEY SelectLatestCompletedByPatientICNAndSurveyType(string patientICN, int typeId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                SURVEY survey = SelectByPatientICNFilter(patientICN).Where(T => T.STD_SURVEY_TYPE_ID == typeId && T.SURVEY_STATUS.ToLower() == "completed").OrderByDescending(T => T.SURVEY_DATE).OrderByDescending(T => T.UPDATED).FirstOrDefault();

                return survey;
            }
        }

        
        /// <summary>
        /// SelectLastCompletedSurveyByPatientICNAndSurveyType
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <returns></returns>
        public IEnumerable<SURVEY> SelectLastCompletedSurveyByPatientICNAndSurveyType(string patientICN, int typeId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<SURVEY> surveys;

                surveys = SelectByPatientICNAndSurveyType(patientICN, typeId, false);
                surveys = (from s in surveys
                           where
                               String.Equals(s.SURVEY_STATUS, "completed", StringComparison.OrdinalIgnoreCase)
                           orderby s.UPDATED descending
                           select s).Take(1).ToList();


                return surveys.ToList();
            }
        }

        /// <summary>
        /// SelectByStatus
        /// </summary>
        /// <param name="statusId"></param>
        /// <param name="patientICN"></param>
        /// <returns></returns>
        public IEnumerable<SURVEY> SelectByStatus(string statusId, string patientICN)
        {

            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<SURVEY> surveys;

                surveys = from t in SelectByPatientICNFilter(patientICN)
                          where t.SURVEY_STATUS == statusId
                          select t;


                return surveys;
            }
        }

        /// <summary>
        /// GetLastCompletedNoteTypeByPatientICNAndSurveyType
        /// </summary>
        /// <param name="patientICN"></param>
        /// <param name="typeId"></param>
        /// <returns></returns>
        public string GetLastCompletedNoteTypeByPatientICNAndSurveyType(string patientICN, int typeId)
        {
            string noteType = String.Empty;

            IEnumerable<SURVEY> survey;

            survey = SelectLastCompletedSurveyByPatientICNAndSurveyType(patientICN, typeId);

            if (survey.Count() == 1)
            {
                int surveyID = survey.First().SURVEYS_ID;

                try
                {
                    List<SURVEY_RESULT> lastCompletedSurveyResults = GetChoicesByQuestionNumber("-1", surveyID);

                    if (lastCompletedSurveyResults.Count() > 0)
                    {
                        noteType = lastCompletedSurveyResults.First().STD_QUESTION_CHOICE.CHOICE_NAME;
                    }
                }
                catch
                {
                    //Ignore error
                }
            }

            return noteType;
        }

        /// <summary>
        /// SelectByStatusCount
        /// </summary>
        /// <param name="statusId"></param>
        /// <param name="patientICN"></param>
        /// <returns></returns>
        public int SelectByStatusCount(string statusId, string patientICN)
        {
            using (_db = GetDataContext())
            {
                return SelectByPatientICNFilter(patientICN).Where(c => c.SURVEY_STATUS == statusId).Count();
            }
        }

        /// <summary>
        /// SelectAllQuestions
        /// </summary>
        /// <param name="typeId"></param>
        /// <returns></returns>
        public IEnumerable<STD_QUESTION> SelectAllQuestions(int typeId)
        {
            using (_db = GetDataContext())
            {
                SetQuestionnaireLoadWith(_db);

                var questions = from t in _db.STD_QUESTIONs
                                where t.STD_SURVEY_TYPE_ID == typeId && t.INACTIVE_FLAG == false
                                select t;

                return questions.ToList();
            }
        }

        /// <summary>
        /// Update
        /// </summary>
        /// <param name="survey"></param>
        /// <returns></returns>
        public string Update(SURVEY survey, bool writeToLog)
        {
            string errorMessage = "";

            survey.AUTHOR_DUZ = MssrServiceTo.DUZ;
            survey.PROVIDER_DUZ = MssrServiceTo.DUZ;
            if (!string.IsNullOrEmpty(StationDivision))
                this.MssrServiceTo.siteCode = StationDivision;
            if (!string.IsNullOrEmpty(MssrServiceTo.siteCode) && !string.IsNullOrEmpty(MssrServiceTo.ServerIP))
            {
                survey.STD_INSTITUTION_ID = GetStdInstitutionId(MssrServiceTo.siteCode, MssrServiceTo.ServerIP);
                if (!string.IsNullOrEmpty(MssrServiceTo.StationNumber))
                {
                    UsersManager userManager = new UsersManager();
                    if (!string.IsNullOrEmpty(MssrServiceTo.SelectedCosignerID))
                    {
                        survey.PROVIDER_DUZ = MssrServiceTo.SelectedCosignerID;
                    }
                    int sta3n = VeteransAffairs.Registries.Business.Utilities.Helpers.ExtractSta3n(MssrServiceTo.StationNumber);
                    survey.AUTHOR_ID = userManager.GetProviderIdByStaffIenAndSta3n(survey.AUTHOR_DUZ, sta3n);
                    survey.PROVIDER_ID = survey.AUTHOR_ID;
                }
                else
                {
                    Logging.WriteLogToFile("WARNING: MSSR.InstrumentsManager: MssrServiceTo.StationNumber is null-empty");
                }
            }
            else
            {
                Logging.WriteLogToFile("WARNING: MSSR.InstrumentsManager: MssrServiceTo.siteCode and ServerIP are null-empty");
            }

            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;

                survey.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                    //create an instance of the custom eventArgs in order to populate the id selected
                    BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                    eventArgs.SavedItemId = survey.SURVEYS_ID;
                    eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;

                    RaiseSaveEvent(this, eventArgs);
                }
                catch (Exception ex)
                {
                    Logging.WriteLogToFile("MSSRInstrumentManager: Failure in Update(survey): " + ex.Message);

                    Console.Write(ex.Message);
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                    errorMessage = ex.Message;
                }
                
                if (writeToLog)
                    Logging.WriteLogToFile("Update(Survey) completed: \r\n");

                return errorMessage;
            }
        }

        public string SurveyEditUpdateSurvey(SURVEY survey, bool writeToLog)
        {
            
            string errorMessage = "";
            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                survey.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                try
                {
                    _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                    //create an instance of the custom eventArgs in order to populate the id selected
                    BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                    eventArgs.SavedItemId = survey.SURVEYS_ID;
                    eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;

                    RaiseSaveEvent(this, eventArgs);
                }
                catch (Exception ex)
                {
                    Logging.WriteLogToFile("MSSRInstrumentManager: Failure in Update(survey): " + ex.Message);
                    Console.Write(ex.Message);
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                    errorMessage = ex.Message;
                }

                if (writeToLog)
                    Logging.WriteLogToFile("Update(Survey) completed: \r\n");

                return errorMessage;
            }
        }
        /// <summary>
        /// DeleteAllSurveyResults
        /// </summary>
        /// <param name="survey"></param>
        public void DeleteAllSurveyResults(SURVEY survey)
        {
            if (survey != null)
            {
                if (survey.SURVEY_RESULTs.Count > 0)
                {
                    survey.SetAsChangeTrackingRoot();

                    foreach (var result in survey.SURVEY_RESULTs)
                    {
                        result.SetAsDeleteOnSubmit();
                    }

                    using (_db = GetDataContext())
                    {
                        _db.DeferredLoadingEnabled = false;

                        survey.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                        try
                        {
                            _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                            //create an instance of the custom eventArgs in order to populate the id selected
                            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                            eventArgs.SavedItemId = survey.SURVEYS_ID;
                            eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        }
                        catch (Exception ex)
                        {
                            Logging.WriteLogToFile("MSSRInstrumentManager: Failure in Update(survey): " + ex.Message);
                            _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        }
                    }
                }
            }
        }

        /// <summary>
        /// Accepts a List<STD_SURVEY_TYPE> objects, converts and returns the equivalent List<InstrumentType>
        /// </summary>
        /// <returns></returns>
        private List<InstrumentType> convertStdSurveyTypeListToInstrumentType(List<STD_SURVEY_TYPE> surveyTypes)
        {
            List<InstrumentType> returnList = new List<InstrumentType>();

            foreach (STD_SURVEY_TYPE survey in surveyTypes)
            {
                // for Mayo-Portland Adaptability Inventory-4 Participation Index and TBI/Polytrauma Individualized Rehabilitation/Reintegration Plan of Care - custom titles required

                InstrumentType instrumentType = new InstrumentType();
                instrumentType.Id = survey.ID;
                instrumentType.Name = survey.NAME;
                instrumentType.IsActive = survey.IS_ACTIVE;

                if (survey.CODE == "MPAI")
                {
                    List<string> titleMpai = new List<string>();
                    titleMpai.Add("Mayo-Portland Adaptability Inventory-4");
                    titleMpai.Add("Participation Index (M2PI)");
                    titleMpai.Add("Muriel D. Lezak, PhD, ABPP & James F. Malec, PhD, ABPP");
                    titleMpai.Add("");
                    titleMpai.Add("Used as VA Interdisciplinary Team Assessment of Community Functioning");

                    instrumentType.Title = titleMpai;

                }
                else if (survey.CODE == "CAREPLAN")
                {
                    List<string> titleCarePlan = new List<string>();
                    titleCarePlan.Add("TBI/Polytrauma Individualized Rehabilitation/Reintegration Plan of Care");

                    instrumentType.Title = titleCarePlan;
                }
                else
                {
                    instrumentType.Title = new List<string>();
                    instrumentType.Title.Add(survey.NAME);
                }

                returnList.Add(instrumentType);
            }

            return returnList;
        }

        #endregion

        public string LookupSiteCodeByServer(string vistaServer, out string error)
        {
            if (_VIAService == null)
                _VIAService = new TbiServiceInterfaceClient("TbiServiceImplPort");
            textTO temp = getSiteIdVIA(vistaServer, out error);
            if (null != temp)
                return temp.text;
            return null;
        }

        public textTO getSiteIdVIA(string vistaServer, out string error)
        {
            textTO result = null;

            _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
            _mdwsLog.SERVICE_NAME = "getSiteIdVIA";
            _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { vistaServer });

            try
            {
                result = _VIAService.getSiteId(vistaServer);

                if (result != null)
                {
                    _mdwsLog.RETURNED_DATA = result.Serialize();

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = result.fault.message;
                    }
                    else
                    {
                        _mdwsLog.ERROR_LEVEL = 0;
                        error = "SUCCESS";
                    }
                }
                else
                {
                    error = "VIA Service call getSiteId returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception ex)
            {
                Logging.WriteLogToFile(ex.ToString());
                error = ex.Message;
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }


        public bool TestVIALogin(MSSRServiceTO mssrServiceTo, out string error)
        {
            try
            {
                if (_VIAService == null)
                    _VIAService = new TbiServiceInterfaceClient("TbiServiceImplPort");
                _loginSiteCode = mssrServiceTo.siteCode;
                var temp = CPRSLaunchVIAWithError(mssrServiceTo);
                TestResultFault(temp);
                error = null;
                return true;
            }
            catch (Exception x)
            {
                Logging.WriteLogToFile(x.ToString());
                error = x.Message;
                return false;
            }
        }

        private void TestResultFault(abstractTO to)
        {
            if (null == to)
            {
                throw new NullReferenceException("No result returned from call to VIA service.");
            }
            if (to.fault != null && !string.IsNullOrEmpty(to.fault.message))
            {
                throw new Exception(to.fault.message);
            }
        }

        private string StationDivision
        {
            get
            {
                return Helpers.DeSerializSessionItems(System.Web.HttpContext.Current.Session["StationDivision"]) as string;
            }
            set
            {
                System.Web.HttpContext.Current.Session["StationDivision"] = Helpers.SerializSessionItemsObj(value);
            }
        }


        #region Enhancement

        public string GetStationIdByInstitutionId(int? instId)
        {
            string stationNumber = string.Empty;
            using (_db = GetDataContext())
            {
                stationNumber = (from st in _db.STD_INSTITUTIONs where st.ID == instId select st.STATIONNUMBER).FirstOrDefault();
            }

            return stationNumber;
        }

        public int GetStationIdByStationNumber(string stationNumber)
        {
            int stationid;
            using (_db = GetDataContext())
            {
                stationid =
                    (from st in _db.STD_INSTITUTIONs where st.STATIONNUMBER == stationNumber select st.ID)
                        .FirstOrDefault();
            }
            return stationid;

        }

        public SURVEY SelectSurveyCompletedByPatientICNAndSurveyType(string patientIcn, int typeId, int surveyId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                SURVEY survey = SelectByPatientICNFilter(patientIcn).FirstOrDefault(T => T.STD_SURVEY_TYPE_ID == typeId && T.SURVEY_STATUS.ToLower() == "completed" && T.SURVEYS_ID == surveyId);

                return survey;
            }
        }
        public SURVEY GetInprocessSurveyByPatientIcnAndSurveyType(string patientIcn, int surveyId, int typeId)
        {    
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                SURVEY survey = SelectByPatientICNFilter(patientIcn).FirstOrDefault(T => T.STD_SURVEY_TYPE_ID == typeId && T.SURVEY_STATUS.ToLower() == "inprocess" && T.SURVEYS_ID == surveyId);

                return survey;
            }
        }

        public void UpdateSurveyAndNewPatientStatus(string patientIcn, int surveyId)
        {
            using (_db = GetDataContext())
            {
                SURVEY survey = _db.SURVEYs.FirstOrDefault(s => s.patient_ICN == patientIcn && s.SURVEYS_ID == surveyId);

                if (survey != null)
                {
                    survey.SURVEY_STATUS = "completed";
                }
                _db.SubmitChanges();
            }
        }

        public IEnumerable<SURVEY> SelectLatestCompletedSurveyByPatientIcnAndSurveyType(string patientIcn, int typeId, DateTime created)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<SURVEY> surveys;

                surveys = SelectByPatientICNAndSurveyType(patientIcn, typeId, false);
                surveys = (from s in surveys
                           where
                               String.Equals(s.SURVEY_STATUS, "completed", StringComparison.OrdinalIgnoreCase) && s.CREATED > created

                           select s).ToList();


                return surveys.ToList();
            }
        }

        public IEnumerable<SURVEY> GetLatestSurveys(string patientIcn, int surveyType, DateTime surveyDate)
        {
            IEnumerable<SURVEY> surveys = SelectLatestCompletedSurveyByPatientIcnAndSurveyType(patientIcn, surveyType, surveyDate);

            return surveys;
            
        }

        public MSSR_PATIENT_MEDICATION GetPatientMedicationByMedicationId(Guid medId)
        {
            MSSR_PATIENT_MEDICATION patMed = new MSSR_PATIENT_MEDICATION();
            using (MSSRDataClassesDataContext ctx = new MSSRDataClassesDataContext())
            {
                patMed = ctx.MSSR_PATIENT_MEDICATIONs.FirstOrDefault(M => M.ID == medId);
                return patMed;
            }
        }

        public MSSR_MEDICATION GetMedicationByMedicationId(int id)
        {
            MSSR_MEDICATION med;
            using (MSSRDataClassesDataContext ctx = new MSSRDataClassesDataContext())
            {
                med = ctx.MSSR_MEDICATIONs.FirstOrDefault(m => m.ID == id);
                return med;
            }
        }

        public List<STD_STATE> GetStates()
        {
            using (_db = GetDataContext())
            {
                List<STD_STATE> states;
                states = (from s in _db.STD_STATEs
                           select s).ToList();
                return states;}
        }
        #endregion
    }

    [Serializable()]
    public class MSSRServiceTO
    {
        #region Properties

        public string siteCode { get; set; }
        public string DUZ { get; set; }
        public string DFN { get; set; }

        public string ServerIP { get; set; }
        public string ServerPort { get; set; }

        public string UserName { get; set; }
        public string PatientName { get; set; }
        public string PatientDOB { get; set; }
        public string PatientDOD { get; set; } // Deceased Date
        public string PatientGender { get; set; }
        public string PatientSoN { get; set; }
        public string PatientLocationId { get; set; }
        public string PatientICN { get; set; }
        public string PatientDFN { get; set; }
        public string STA6N { get; set; }
        public string PatientAdmintTimeStamp { get; set; }
        public string InstitutionName { get; set; }

        // Demographics
        public string PatientAge { get; set; }
        public string PatientCity { get; set; }
        public string PatientState { get; set; }
        public string PatientMaritalStatus { get; set; }
        public string PatientReligion { get; set; }
        public int PatientRegion { get; set; }
        public string PatientRace { get; set; }
        public string PatientEthnicity { get; set; }

        //public string PatientPCP { get; set; }

        public int CprsSentMSSREvaluationNoteId { get; set; }
        public int CprsSentMSSRFollowUpNoteId { get; set; }

        public NoteType SelectedNoteType { get; set; }
        public string NoteText { get; set; }

        public InstrumentType SelectedInstrumentType { get; set; }

        public string SelectedNoteTitleId { get; set; }
        public string SelectedConsultId { get; set; }
        public string SelectedVisitId { get; set; }
        public string SelectedHospitalAdmissionId { get; set; }
        public string SelectedLocationID { get; set; }
        public string SelectedNewVisitId { get; set; }

        public string SelectedCosignerID { get; set; }

        public bool isHistorical { get; set; }

        public List<InstrumentType> Instruments;

        public string StationNumber { get; set; }

        #endregion

        public MSSRServiceTO() { }
    }

    public enum NoteTypeEnum : int
    {
        ScheduledClinicAppointment = 0,
        HospitalAdmission = 1,
        CurrentStay = 2,
        UnscheduledOrNewVisit = 3
    }

    [Serializable()]
    public class NoteType
    {
        public int Id { get; set; }
        public string Name { get; set; }
    }

    [Serializable()]
    public class InstrumentType
    {
        public int Id { get; set; }
        public string Name { get; set; }
        public string Status { get; set; }
        public List<string> Title { get; set; }
        public Boolean IsActive { get; set; }
    }
}