﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;

using System.Configuration;
using System.Data;
using System.Diagnostics;

using DataAccess;

namespace RevampPhilipsService
{
    public class CPhilipsXferItem
    {
        public string PatID { get; set; }
        public string PatIsActive { get; set; }
        public string PatFacilityID { get; set; }
        public string PatFirstName { get; set; }
        public string PatLastName { get; set; }
        public string PatMiddleName { get; set; }

        public string PrescriptionsXML { get; set; }
        public string DaysXML { get; set; }
        public string SessionsXML { get; set; }
        public string FileName { get; set; }

        /// <summary>
        /// Save the patient days and sessions
        /// </summary>
        /// <param name="conn"></param>
        /// <returns></returns>
        public bool Save(ref CDataConnection conn)
        {
            long lXferID = 0;
            string strPatientID = String.Empty;

            lXferID = SavePatient(ref conn, ref strPatientID);
            if (lXferID > 0)
            {
                //save prescription
                bool bPrescriptionStatus = SavePrescription(ref conn, lXferID);

                //save the days
                bool bDaysStatus = false;
                if (!String.IsNullOrEmpty(DaysXML))
                {
                    bDaysStatus = SaveDays(ref conn, lXferID);
                }
                else
                {
                    CLogEvent evt = new CLogEvent();
                    evt.LogEvent(this.PatID + " - Patient data without Days records.", EventLogEntryType.Warning);
                }

                //save the sessions
                bool bSessionStatus = false;
                if (!String.IsNullOrEmpty(SessionsXML))
                {
                    bSessionStatus = SaveSessions(ref conn, lXferID);
                }
                else
                {
                    CLogEvent evt = new CLogEvent();
                    evt.LogEvent(this.PatID + " - Patient data without Sessions records.", EventLogEntryType.Warning);
                }

                if (!String.IsNullOrEmpty(strPatientID))
                {
                    SignalXfer(ref conn, lXferID, strPatientID);
                }
            }

            return true;
        }

        /// <summary>
        /// Save patient data and get PatientID and XferID
        /// </summary>
        /// <param name="conn"></param>
        /// <param name="strPatientID"></param>
        /// <returns></returns>
        private long SavePatient(ref CDataConnection conn,
                                 ref string strPatientID)
        {
            long lStatusCode = 0;
            string strStatusComment = String.Empty;

            long lXferID = -1;

            CDataParameterList pList = new CDataParameterList();
            pList.AddInputParameter("pi_vFileName", this.FileName);
            pList.AddInputParameter("pi_vExternalID", this.PatID);
            pList.AddInputParameter("pi_vFirstName", this.PatFirstName);
            pList.AddInputParameter("pi_vLastName", this.PatLastName);
            pList.AddInputParameter("pi_vMiddleName", this.PatMiddleName);
            pList.AddInputParameter("pi_vFacilityID", this.PatFacilityID);

            //
            CDataSet cds = new CDataSet();
            DataSet ds = cds.GetOracleDataSet(conn,
                                              "PCK_XFER.ImportPhilipsPatientRS",
                                              pList,
                                              out lStatusCode,
                                              out strStatusComment);

            //check response
            if (lStatusCode == 0)
            {
                if (ds != null)
                {
                    foreach (DataTable table in ds.Tables)
                    {
                        foreach (DataRow row in table.Rows)
                        {
                            if (!row.IsNull("XFERID"))
                            {
                                lXferID = Convert.ToInt64(row["XFERID"].ToString());
                            }
                            if (!row.IsNull("PATIENTID"))
                            {
                                strPatientID = row["PATIENTID"].ToString();
                            }
                        }
                    }
                }
            }
            else
            {
                CLogEvent evt = new CLogEvent();
                evt.LogEvent(strStatusComment, EventLogEntryType.Warning);
            }

            return lXferID;

        }

        /// <summary>
        /// Save all prescription. The serial number of the device is there
        /// </summary>
        /// <param name="conn"></param>
        /// <param name="lXferID"></param>
        /// <returns></returns>
        private bool SavePrescription(ref CDataConnection conn,
                                      long lXferID)
        {
            bool bStatus = true;

            if (String.IsNullOrEmpty(this.PrescriptionsXML))
            {
                return bStatus;
            }

            //sessions reader
            //security - XML External Entities attacks benefit from an XML feature to build documents dynamically at the time of processing. 
            XmlReaderSettings settings = new XmlReaderSettings();
            settings.DtdProcessing = DtdProcessing.Prohibit;
            settings.XmlResolver = null;

            XmlReader readerPrescriptions = XmlTextReader.Create(new System.IO.StringReader(this.PrescriptionsXML), settings);
            while (!readerPrescriptions.EOF)
            {
                if (readerPrescriptions.NodeType == XmlNodeType.Element && readerPrescriptions.Name.ToLower() == "prescription")
                {
                    //get the session XML
                    string strPrescriptionXML = readerPrescriptions.ReadOuterXml();

                    XmlReader readerPrescription = XmlTextReader.Create(new System.IO.StringReader(strPrescriptionXML), settings);
                    while (!readerPrescription.EOF)
                    {
                        if (readerPrescription.NodeType == XmlNodeType.Element && readerPrescription.Name.ToLower() == "device")
                        {
                            //device issued date
                            string strDeviceIssuedDate = readerPrescription.GetAttribute("Issued");
                            DateTime dtDeviceIssuedDate = DateTime.Parse(strDeviceIssuedDate,
                                                                   null,
                                                                   System.Globalization.DateTimeStyles.RoundtripKind);

                            string strModel = String.Empty;
                            string strSerial = String.Empty;
                            string strTherapyMode = String.Empty;
                            string strModem = String.Empty;
                            string strHumidifer = String.Empty;
                            string strMinPressure = String.Empty;
                            string strMaxPressue = String.Empty;

                            //get the values
                            XmlReader readerSettings = XmlTextReader.Create(new System.IO.StringReader(readerPrescription.ReadInnerXml()), settings);
                            while (!readerSettings.EOF)
                            {
                                if (readerSettings.NodeType == XmlNodeType.Element && readerSettings.Name.ToLower() == "setting")
                                {
                                    string strName = readerSettings.GetAttribute("Name").ToLower();
                                    string strValue = readerSettings.ReadInnerXml();

                                    if (strName == "model")
                                    {
                                        strModel = strValue;
                                    }
                                    if (strName == "serial")
                                    {
                                        strSerial = strValue;
                                    }
                                    if (strName == "therapy mode")
                                    {
                                        strTherapyMode = strValue;
                                    }
                                    if (strName == "use modem")
                                    {
                                        strModem = strValue;
                                    }
                                    if (strName == "humidifer setting")
                                    {
                                        strHumidifer = strValue;
                                    }
                                    if (strName == "minimum cpap pressure")
                                    {
                                        strMinPressure = strValue;
                                    }
                                    if (strName == "maximum cpap pressure")
                                    {
                                        strMaxPressue = strValue;
                                    }
                                }
                                else
                                {
                                    readerSettings.Read();
                                }
                            }
                            readerSettings.Close();

                            //insert prescription
                            CDataParameterList pList = new CDataParameterList();
                            pList.AddInputParameter("pi_vFileName", this.FileName);
                            pList.AddInputParameter("pi_vExternalID", this.PatID);
                            pList.AddInputParameter("pi_nXferID", lXferID);
                            pList.AddInputParameter("pi_dtDeviceIssuedDate", dtDeviceIssuedDate);
                            pList.AddInputParameter("pi_vModel", strModel);
                            pList.AddInputParameter("pi_vSerial", strSerial);
                            pList.AddInputParameter("pi_vTherapyMode", strTherapyMode);
                            pList.AddInputParameter("pi_vUseModem", strModem);
                            pList.AddInputParameter("pi_vHumidifer", strHumidifer);
                            pList.AddInputParameter("pi_vMinCPAPPressure", strMinPressure);
                            pList.AddInputParameter("pi_vMaxCPAPPressure", strMaxPressue);

                            long lStatusCode = 0;
                            string strStatus = "";
                            conn.ExecuteOracleSP("PCK_XFER.ImportPhilipsPrescription",
                                                  pList,
                                                  out lStatusCode,
                                                  out strStatus);

                            if (lStatusCode != 0)
                            {
                                //errors will be handled on the db side so that this method
                                //runs as quickly as possible
                            }

                        }
                        else
                        {
                            readerPrescription.Read();
                        }
                    }
                    readerPrescription.Close();
                }
                else
                {
                    readerPrescriptions.Read();
                }
            }
            readerPrescriptions.Close();

            return bStatus;
        }

        /// <summary>
        /// Save all sessions
        /// </summary>
        /// <param name="conn"></param>
        /// <param name="lXferID"></param>
        /// <returns></returns>
        private bool SaveSessions(ref CDataConnection conn,
                                  long lXferID)
        {
            //sessions reader
            //security - XML External Entities attacks benefit from an XML feature to build documents dynamically at the time of processing. 
            XmlReaderSettings settings = new XmlReaderSettings();
            settings.DtdProcessing = DtdProcessing.Prohibit;
            settings.XmlResolver = null;

            if (String.IsNullOrEmpty(SessionsXML))
            {
                return false;
            }

            XmlReader readerSessions = XmlTextReader.Create(new System.IO.StringReader(SessionsXML), settings);
            while (!readerSessions.EOF)
            {
                if (readerSessions.NodeType == XmlNodeType.Element && readerSessions.Name.ToLower() == "session")
                {
                    //get the session XML
                    string strSessionXML = readerSessions.ReadOuterXml();

                    //philips structure has 2 roots settings and events
                    //for the inner xml calls to work we need to split them 
                    //out so we have 1 root for each
                    string strSettingsXML = String.Empty;
                    string strEventsXML = String.Empty;
                    XmlReader readerSettingsXML = XmlTextReader.Create(new System.IO.StringReader(strSessionXML), settings);
                    while (!readerSettingsXML.EOF)
                    {
                        if (readerSettingsXML.NodeType == XmlNodeType.Element)
                        {
                            if (readerSettingsXML.Name.ToLower() == "settings")
                            {
                                strSettingsXML = readerSettingsXML.ReadOuterXml();
                            }
                            else if (readerSettingsXML.Name.ToLower() == "events")
                            {
                                strEventsXML = readerSettingsXML.ReadOuterXml();
                            }
                            else
                            {
                                readerSettingsXML.Read();
                            }
                        }
                        else
                        {
                            readerSettingsXML.Read();
                        }
                    }
                    readerSettingsXML.Close();

                    //Settings XML
                    string strDeviceTherapyMode = "";

                    #region process Settings
                    //settings
                    XmlReader readerSettings = XmlTextReader.Create(new System.IO.StringReader(strSettingsXML), settings);
                    while (!readerSettings.EOF)
                    {
                        if (readerSettings.NodeType == XmlNodeType.Element && readerSettings.Name.ToLower() == "setting")
                        {
                            string strSetName = readerSettings.GetAttribute("Name").ToLower();
                            string strValue = readerSettings.ReadInnerXml();

                            if (strSetName == "device therapy mode")
                            {
                                strDeviceTherapyMode = strValue;
                            }
                        }
                        else
                        {
                            readerSettings.Read();
                        }
                    }
                    readerSettings.Close();
                    #endregion

                    #region process Events
                    //we will only use Mask On/Off from this section, however we will pass all events to the 
                    //sp call in case they are needed later
                    //
                    //The Mask On/Off data is in the Events with the Name=”Connect”. 
                    //
                    //The one with the Name=”Blower” is actually the total timespan from the Connect. 
                    //Meaning we could have multiple events with Name=”Connect” but only one Name=”Blower” 
                    //with the start datetime of the first Name=”Connect” and the end datetime of the last connect.
                    //Event=”Connect”.Start -> PATIENT_CPAP_USAGE.MASK_ON
                    //Event=”Connect”.End -> PATIENT_CPAP_USAGE.MASK_OFF

                    //events
                    XmlReader readerEvents = XmlTextReader.Create(new System.IO.StringReader(strEventsXML), settings);
                    while (readerEvents.Read())
                    {
                        if (readerEvents.NodeType == XmlNodeType.Element && readerEvents.Name.ToLower() == "event")
                        {
                            //Event properties 
                            string strEventName = readerEvents.GetAttribute("Name");

                            string strEventEnd = readerEvents.GetAttribute("End");
                            DateTime dtEventEnd = DateTime.Parse(strEventEnd,
                                                                 null,
                                                                 System.Globalization.DateTimeStyles.RoundtripKind);

                            string strEventStart = readerEvents.GetAttribute("Start");
                            DateTime dtEventStart = DateTime.Parse(strEventStart,
                                                                   null,
                                                                   System.Globalization.DateTimeStyles.RoundtripKind);


                            //save this one session/event
                            CDataParameterList pList = new CDataParameterList();
                            pList.AddInputParameter("pi_vFileName", FileName);
                            pList.AddInputParameter("pi_vExternalID", PatID);
                            pList.AddInputParameter("pi_nXferID", lXferID);
                            pList.AddInputParameter("pi_vPatFacilityID", this.PatFacilityID);
                            pList.AddInputParameter("pi_vDeviceTherapyMode", strDeviceTherapyMode);
                            pList.AddInputParameter("pi_vEventName", strEventName);
                            pList.AddInputParameter("pi_dtEventStart", dtEventStart);
                            pList.AddInputParameter("pi_dtEventEnd", dtEventEnd);

                            long lStatusCode = 0;
                            string strStatus = "";
                            conn.ExecuteOracleSP("PCK_XFER.ImportPhilipsSession",
                                                  pList,
                                                  out lStatusCode,
                                                  out strStatus);

                            if (lStatusCode != 0)
                            {
                                //errors will be handled on the db side so that this method
                                //runs as quickly as possible
                            }

                        }
                    }
                    readerEvents.Close();
                    #endregion
                }
                else
                {
                    readerSessions.Read();
                }
            }//end while (readerSessions.Read())
            readerSessions.Close();

            return true;
        }

        /// <summary>
        /// Save all the patient days
        /// </summary>
        /// <param name="conn"></param>
        /// <param name="lXferID"></param>
        /// <returns></returns>
        private bool SaveDays(ref CDataConnection conn,
                              long lXferID)
        {
            //security - XML External Entities attacks benefit from an XML feature to build documents dynamically at the time of processing. 
            XmlReaderSettings settings = new XmlReaderSettings();
            settings.DtdProcessing = DtdProcessing.Prohibit;
            settings.XmlResolver = null;

            if (String.IsNullOrEmpty(DaysXML))
            {
                return false;
            }

            XmlReader readerDays = XmlTextReader.Create(new System.IO.StringReader(DaysXML), settings);
            while (!readerDays.EOF)
            {
                if (readerDays.NodeType == XmlNodeType.Element && readerDays.Name.ToLower() == "day")
                {
                    //day properties 2015-12-03T17:00:00Z
                    string strDayStart = readerDays.GetAttribute("Start");
                    DateTime dtStart = DateTime.Parse(strDayStart,
                                                       null,
                                                       System.Globalization.DateTimeStyles.RoundtripKind);

                    string strDayEnd = readerDays.GetAttribute("End");
                    DateTime dtEnd = DateTime.Parse(strDayEnd,
                                                     null,
                                                     System.Globalization.DateTimeStyles.RoundtripKind);


                    string strDayIsActive = readerDays.GetAttribute("IsActive");

                    //statistics vars
                    string strTotalConnTime = "";
                    string strTotalBlowerTime = "";
                    string strApneaHypoIndex = "";
                    string strApneaIdx = "";
                    string strHypopneaIndx = "";
                    string strObstrAirApneaIdx = "";
                    string strClearAirwApneaIdx = "";
                    string strFlowLimitIdx = "";
                    string strVibratorySnoreIdx = "";
                    string strRERAIdx = "";
                    string strPctPeriodBreath = "";
                    string strAverageLeak = "";
                    string strMaximumLeak = "";
                    string str90PctLeak = "";
                    string strTotalLargeLeakDuration = "";
                    string strAvgCPAPPressure = "";
                    string strMaxCPAPPressure = "";
                    string strMinCPAPPressure = "";
                    string str90PctCPAPPressure = "";
                    string strTotClassHumidTime = "";
                    string strTotSystemOneHumidTime = "";
                    string strTotHeatedTubeHumidTime = "";
                    string strTotPassOverHumidTime = "";
                    string strTotDisconnectedHumidTime = "";
                    string strTotErrorHumidiTime = "";
                    string strAverageHumidSetting = "";
                    string strAverageAmbientTemp = "";
                    string strAverageAmbientHumid = "";

                    //statistics XML
                    string strStatisticsXML = readerDays.ReadInnerXml();

                    XmlReader readerStat = XmlTextReader.Create(new System.IO.StringReader(strStatisticsXML), settings);
                    while (!readerStat.EOF)
                    {
                        if (readerStat.NodeType == XmlNodeType.Element && readerStat.Name.ToLower() == "statistic")
                        {
                            //statistic name and type attributes
                            string strStatName = readerStat.GetAttribute("Name");

                            //get the value
                            string strValue = readerStat.ReadInnerXml();

                            if (strStatName == "Total Connected Time")
                            {
                                strTotalConnTime = strValue;
                            }
                            else if (strStatName == "Total Blower Time")
                            {
                                strTotalBlowerTime = strValue;
                            }
                            else if (strStatName == "Apnea/Hypopnea Index")
                            {
                                strApneaHypoIndex = strValue;
                            }
                            else if (strStatName == "Apnea Index")
                            {
                                strApneaIdx = strValue;
                            }
                            else if (strStatName == "Hypopnea Index")
                            {
                                strHypopneaIndx = strValue;
                            }
                            else if (strStatName == "Obstructed Airway Apnea Index")
                            {
                                strObstrAirApneaIdx = strValue;
                            }
                            else if (strStatName == "Clear Airway Apnea Index")
                            {
                                strClearAirwApneaIdx = strValue;
                            }
                            else if (strStatName == "Flow Limitation Index")
                            {
                                strFlowLimitIdx = strValue;
                            }
                            else if (strStatName == "Vibratory Snore Index")
                            {
                                strVibratorySnoreIdx = strValue;
                            }
                            else if (strStatName == "RERA Index")
                            {
                                strRERAIdx = strValue;
                            }
                            else if (strStatName == "% Periodic Breathing")
                            {
                                strPctPeriodBreath = strValue;
                            }
                            else if (strStatName == "Average Leak")
                            {
                                strAverageLeak = strValue;
                            }
                            else if (strStatName == "Maximum Leak")
                            {
                                strMaximumLeak = strValue;
                            }
                            else if (strStatName == "90% Leak")
                            {
                                str90PctLeak = strValue;
                            }
                            else if (strStatName == "Total Large Leak Duration")
                            {
                                strTotalLargeLeakDuration = strValue;
                            }
                            else if (strStatName == "Average CPAP Pressure")
                            {
                                strAvgCPAPPressure = strValue;
                            }
                            else if (strStatName == "Maximum CPAP Pressure")
                            {
                                strMaxCPAPPressure = strValue;
                            }
                            else if (strStatName == "Minimum CPAP Pressure")
                            {
                                strMinCPAPPressure = strValue;
                            }
                            else if (strStatName == "90% CPAP Pressure")
                            {
                                str90PctCPAPPressure = strValue;
                            }
                            else if (strStatName == "Total Classic Humidification Time")
                            {
                                strTotClassHumidTime = strValue;
                            }
                            else if (strStatName == "Total System One Humidification Time")
                            {
                                strTotSystemOneHumidTime = strValue;
                            }
                            else if (strStatName == "Total Heated Tube Humidification Time")
                            {
                                strTotHeatedTubeHumidTime = strValue;
                            }
                            else if (strStatName == "Total Pass Over Humidification Time")
                            {
                                strTotPassOverHumidTime = strValue;
                            }
                            else if (strStatName == "Total Disconnected Humidification Time")
                            {
                                strTotDisconnectedHumidTime = strValue;
                            }
                            else if (strStatName == "Total Error Humidification Time")
                            {
                                strTotErrorHumidiTime = strValue;
                            }
                            else if (strStatName == "Average Humidifier Setting")
                            {
                                strAverageHumidSetting = strValue;
                            }
                            else if (strStatName == "Average Ambient Temperature")
                            {
                                strAverageAmbientTemp = strValue;
                            }
                            else if (strStatName == "Average Ambient Humidity")
                            {
                                strAverageAmbientHumid = strValue;
                            }
                            else
                            {
                                //property we are not interested in
                            }
                        }
                        else
                        {
                            readerStat.Read();
                        }
                    }
                    readerStat.Close();

                    //at this point we have all the statistics so we can save the day record
                    //do work and then close
                    CDataParameterList pList = new CDataParameterList();

                    pList.AddInputParameter("pi_vFileName", FileName);
                    pList.AddInputParameter("pi_vExternalID", PatID);
                    pList.AddInputParameter("pi_nXferID", lXferID);
                    pList.AddInputParameter("pi_vPatFacilityID", PatFacilityID);

                    pList.AddInputParameter("pi_dtDayStart", dtStart);
                    pList.AddInputParameter("pi_dtDayEnd", dtEnd);

                    pList.AddInputParameter("pi_vTotalConnTime", strTotalConnTime);
                    pList.AddInputParameter("pi_vTotalBlowerTime", strTotalBlowerTime);
                    pList.AddInputParameter("pi_vApneaHypoIndex", strApneaHypoIndex);
                    pList.AddInputParameter("pi_vApneaIdx", strApneaIdx);
                    pList.AddInputParameter("pi_vHypopneaIndx", strHypopneaIndx);
                    pList.AddInputParameter("pi_vObstrAirApneaIdx", strObstrAirApneaIdx);
                    pList.AddInputParameter("pi_vClearAirwApneaIdx", strClearAirwApneaIdx);
                    pList.AddInputParameter("pi_vFlowLimitIdx", strFlowLimitIdx);
                    pList.AddInputParameter("pi_vVibratorySnoreIdx", strVibratorySnoreIdx);
                    pList.AddInputParameter("pi_vRERAIdx", strRERAIdx);
                    pList.AddInputParameter("pi_vPctPeriodBreath", strPctPeriodBreath);
                    pList.AddInputParameter("pi_vAverageLeak", strAverageLeak);
                    pList.AddInputParameter("pi_vMaximumLeak", strMaximumLeak);
                    pList.AddInputParameter("pi_v90PctLeak", str90PctLeak);
                    pList.AddInputParameter("pi_vTotalLargeLeakDuration", strTotalLargeLeakDuration);
                    pList.AddInputParameter("pi_vAvgCPAPPressure", strAvgCPAPPressure);
                    pList.AddInputParameter("pi_vMaxCPAPPressure", strMaxCPAPPressure);
                    pList.AddInputParameter("pi_vMinCPAPPressure", strMinCPAPPressure);
                    pList.AddInputParameter("pi_v90PctCPAPPressure", str90PctCPAPPressure);
                    pList.AddInputParameter("pi_vTotClassHumidTime", strTotClassHumidTime);
                    pList.AddInputParameter("pi_vTotSystemOneHumidTime", strTotSystemOneHumidTime);
                    pList.AddInputParameter("pi_vTotHeatedTubeHumidTime", strTotHeatedTubeHumidTime);
                    pList.AddInputParameter("pi_vTotPassOverHumidTime", strTotPassOverHumidTime);
                    pList.AddInputParameter("pi_vTotDisconnectedHumidTime", strTotDisconnectedHumidTime);
                    pList.AddInputParameter("pi_vTotErrorHumidiTime", strTotErrorHumidiTime);
                    pList.AddInputParameter("pi_vAverageHumidSetting", strAverageHumidSetting);
                    pList.AddInputParameter("pi_vAverageAmbientTemp", strAverageAmbientTemp);
                    pList.AddInputParameter("pi_vAverageAmbientHumid", strAverageAmbientHumid);

                    long lStatusCode = 0;
                    string strStatus = "";
                    conn.ExecuteOracleSP("PCK_XFER.ImportPhilipsDay",
                                          pList,
                                          out lStatusCode,
                                          out strStatus);

                    if (lStatusCode != 0)
                    {
                        //errors will be handled on the db side so that this method
                        //runs as quickly as possible
                        CLogEvent evt = new CLogEvent();
                        evt.LogEvent(strStatus, EventLogEntryType.Warning);
                    }


                }
                else
                {
                    readerDays.Read();
                }
            }
            readerDays.Close();

            return true;
        }

        /// <summary>
        /// Transfer the data to real tables
        /// </summary>
        /// <param name="conn"></param>
        /// <param name="lXferID"></param>
        /// <param name="strPatientID"></param>
        private void SignalXfer(ref CDataConnection conn,
                                long lXferID,
                                string strPatientID)
        {
            long lStatusCode = 0;
            string strStatus = String.Empty;

            CDataParameterList pList = new CDataParameterList();
            pList.AddInputParameter("pi_vFileName", FileName);
            pList.AddInputParameter("pi_vExternalID", PatID);
            pList.AddInputParameter("pi_nXferID", lXferID);
            pList.AddInputParameter("pi_vPatientID", strPatientID);


            conn.ExecuteOracleSP("PCK_XFER.XferPhilipsCPAP",
                                  pList,
                                  out lStatusCode,
                                  out strStatus);

            if (lStatusCode != 0)
            {
                //errors will be handled on the db side so that this method
                //runs as quickly as possible
            }

        }

    }

}