define(['angular', 'app', 'formatter'], function (angular, app) {
	"use strict";
	app.service('BaseCRUDService', function ($http, $q, $parse, $filter, formatter) {

		//Constructor
		var BaseCRUDService = function(config) {

			this.config = config;
			
			this.requests = {
				change : [],
				retrieve : []
			};

			this.idGetter = $parse(this.config.object.uidKey);

		};

		//Methods
		BaseCRUDService.prototype = {};

		BaseCRUDService.prototype.transformRequest = function(item) {
			return this.config.object.requestTransform ? this.config.object.requestTransform(item) : item;
		};

		BaseCRUDService.prototype._makeRequest = function (httpConfig, linkTitle) {
			var that = this;
			var defer = $q.defer();

			function sendRequest(httpConfig) {

				if(httpConfig.method === "POST" || httpConfig.method === "PUT") {
					httpConfig.data = that.transformRequest(httpConfig.data);
					delete httpConfig.data.link;
				}

				var waitForIt = httpConfig.method === "GET" ? that.requests.change : that.requests.retrieve;

				$q.all(waitForIt).finally(function(){

					var request = $http(httpConfig).success(function (data, status, headers, config, statusText) {
						defer.resolve({data : data, status : status, headers : headers, config: config, statusText : statusText});
					}).error(function (data, status, headers, config, statusText) {
						defer.reject({data : data, status : status, headers : headers, config: config, statusText : statusText});
					}).finally(function(){
						waitForIt.splice(waitForIt.indexOf(request), 1);
					})

					if(httpConfig.method === "GET") {
						that.requests.retrieve.push(request);
					} else {
						that.requests.change.push(request);
					}

				});
			}


			if (httpConfig.url) {
				sendRequest(httpConfig);
			} else {
				that.config.dependency.service.fetch().then(
					function (resource) {
						var linkTitle = that.config.dependency.linkTitles[httpConfig.method] || that.config.dependency.linkTitles["GET"];
						httpConfig.url = resource.link ? resource.link[linkTitle] : resource[linkTitle];
						if (httpConfig.pathParamString) {
							var i = httpConfig.url.indexOf('?');
							if (i > 0)
								httpConfig.url = httpConfig.url.substring(0, i) + httpConfig.pathParamString + httpConfig.url.substring(i);
							else
								httpConfig.url += httpConfig.pathParamString || "";
						}
						sendRequest(httpConfig);
					});
			}

			return defer.promise;
		};

		//Requires Override
		BaseCRUDService.prototype.createEmpty = function() {
			return {};
		};

		BaseCRUDService.prototype.fetch = function (queryParams, preventLocalUpdate) {
			return this._makeRequest({method: "GET", params: queryParams});
		};

		BaseCRUDService.prototype.fetchLatest = function (queryParams) {
			return this._makeRequest({method: "GET", params: queryParams, pathParamString: "/latest"});
		};

		BaseCRUDService.prototype.save = function (item) {
			var method = !item.link || this.config.dependency.linkTitles.PUT === "none" ? "POST" : "PUT";
			return this._makeRequest({method: method, data: item, url : (method === "PUT" ? item.link[0].href : null)});
		};

		BaseCRUDService.prototype.delete = function (item) {
			if(this.idGetter(item)) {
				return this._makeRequest({method: "DELETE", url : item.link[0].href});
			} else if(item.pathParamString || item.queryParams) {
				return this._makeRequest({method: "DELETE", params: item.queryParams, pathParamString: item.pathParamString});
			} else {
				var defer = $q.defer();
				defer.reject();
				return defer.promise;
			}

		};

		return BaseCRUDService;
	});
});