define(['authentication_service'], function () {
	'use strict';

	describe('Authentication Service --', function () {
		var $scope,
			$http,
			$httpBackend,
			$q,
			$window,
			$location,
			$injector,
			service,
			resourceDirectory,
			resourceDirectoryServiceMock,
			localResourceDirectory,
			localResourceDirectoryServiceMock,
			localDeferred,
			pageServiceMock,
			windowOpenSpy;

		beforeEach(function() {
			module('angularTemplateApp');

			// resourceDirectory returns a hash of keys to URL values for MAE resources
			resourceDirectoryServiceMock = jasmine.createSpyObj('resourceDirectory', ['fetch']);
			localResourceDirectoryServiceMock = jasmine.createSpyObj('localResourceDirectoryService', ['fetch']);
			pageServiceMock = { appUri:'http://test.com/var-utility/' };

			resourceDirectory = {
				'mhpuser' 		: '/user',
				'token' 		: '/token',
				'oauth-login'	: '/oauth',
				'dummy'			: '/dummy'
			};

			localResourceDirectory = {
				'login': '/authorize',
				'logout': '/logout',
				'rest-v1': '/rest/v1',
				'oauth-info': '/oauth/token',
				'oauth-token': '/oauth/info'
			};

			module(function($provide) {
				$provide.value('resourceDirectory', resourceDirectoryServiceMock);
				$provide.value('localResourceDirectoryService', localResourceDirectoryServiceMock);
				$provide.value('pageService', pageServiceMock);
			});

			inject(function($rootScope, _$http_, _$httpBackend_, _$q_, _$window_, _$location_, _$injector_, authenticationService) {
				$scope = $rootScope;
				$http = _$http_;
				$httpBackend = _$httpBackend_;
				$q = _$q_;
				$window = _$window_;
				$location = _$location_;
				$injector = _$injector_;
				service = authenticationService;

				var resourceDeferred = $q.defer();
				resourceDeferred.resolve(resourceDirectory);
				resourceDirectoryServiceMock.fetch.andReturn(resourceDeferred.promise);
				var localResourceDeferred = $q.defer();
				localResourceDeferred.resolve(localResourceDirectory);
				localResourceDirectoryServiceMock.fetch.andReturn(localResourceDeferred.promise);

				$httpBackend.whenGET('/MobileHealthPlatformWeb/rest/public/resource-directory/json').respond(resourceDirectory);
				$httpBackend.whenGET(resourceDirectory['mhpuser']).respond('200', {'user':'patient1'});
				$httpBackend.whenDELETE(resourceDirectory['token']).respond('200', {});

				$httpBackend.whenGET('resources.json').respond(localResourceDirectory);
				$httpBackend.whenGET(localResourceDirectory['oauth-info']).respond('200', {});
			});
		});

		/* GENERAL FLOW OF AUTH SERVICE USAGE */
		// From splash page, beginLogon() --> service.authenticate() --> service.checkAuthStatus()
		// checkAuthStatus: calls validateToken()
		// - validateToken: returns 200 response if valid, calls various other authService functions related to tokens and fetches resourceDirectory
		// -- sets http.defaults header with currentToken
		// -- then does a GET on link mapped to resourceDirectory item 'mhpuser' -- the actual authentication, if its promise resolves we're good to go

		describe('the service', function () {
			beforeEach(function () {
				windowOpenSpy = spyOn(window, 'open');
				sessionStorage.setItem('token', null);

				localDeferred = $q.defer();
				localDeferred.resolve('success'); // general resolved/success promise
			});

			it('should have some commonly used methods defined', function () {
				expect(service.readLocalResourceDirectory).toBeDefined();

				// used in other parts of the app
				expect(service.authenticate).toBeDefined();
				expect(service.checkForAuthCode).toBeDefined();
				expect(service.getNewToken).toBeDefined();
				expect(service.checkAuthStatus).toBeDefined();
				expect(service.isAuthenticated).toBeDefined();
				expect(service.isAuthorized).toBeDefined();
				expect(service.checkAuthorizedKey).toBeDefined();
				expect(service.wipeSessionData).toBeDefined();
				expect(service.logoutRedirectToLaunchpad).toBeDefined();
			});

			it('should store session token', function () {
				expect(sessionStorage.getItem('token')).toEqual(JSON.stringify(null));
				service.storeSessionToken('fakeToken12345');
				expect(sessionStorage.getItem('token')).toEqual(JSON.stringify('fakeToken12345'));
			});

			it('should get current session token', function () {
				expect(service.getCurrentToken()).toEqual(null);
				service.storeSessionToken('fakeToken12345');
				expect(service.getCurrentToken()).toEqual('fakeToken12345');
			})

			it('should be able to delete cookie', function () {
				var testCookie = 'fakeCookie=12345';
				document.cookie = testCookie;
				expect(document.cookie).toEqual(testCookie);
				service.deleteCookie('fakeCookie');
				expect(document.cookie).toEqual('');
			});

			it('should be able to delete token', function () {
				$httpBackend.expectDELETE(resourceDirectory['token']);

				service.deleteToken();
				$scope.$digest();

				// flushing ensures that if the expected http request above wasn't made, test will fail
				$httpBackend.flush();

				expect(resourceDirectoryServiceMock.fetch).toHaveBeenCalled();
			});

			it('should be able to validate token', function () {
				spyOn(service, 'setAuthorizationRequestHeader');
				$httpBackend.expectGET(resourceDirectory['mhpuser']);

				service.storeSessionToken('fakeToken12345');
				service.validateToken();
				$scope.$digest();
				$httpBackend.flush();

				expect(resourceDirectoryServiceMock.fetch).toHaveBeenCalled();
				expect(service.setAuthorizationRequestHeader).toHaveBeenCalledWith('fakeToken12345');
			});

			it('should wipe session data', function () {
				spyOn(service, 'deleteToken').andReturn(localDeferred.promise);
				spyOn(service, 'deleteCookie').andCallThrough();

				sessionStorage.setItem('token', JSON.stringify('fakeToken'));
				service.wipeSessionData();

				$scope.$digest();

				expect(sessionStorage.getItem('token')).toEqual(null);
				expect(service.deleteToken).toHaveBeenCalled();
				expect(service.deleteCookie).toHaveBeenCalled();
			});

			it('should login with redirect', function () {
				$httpBackend.expectGET(localResourceDirectory['oauth-info']);
				service.authenticate();

				$scope.$digest();
				$httpBackend.flush();

				expect(window.open).toHaveBeenCalled();
			});

			// call to checkAuthStatus is in router.js

			it('should logout with redirect', function () {
				spyOn(service, 'wipeSessionData').andReturn(localDeferred.promise);
				service.logoutRedirectToLaunchpad();

				$scope.$digest();

				expect(window.open).toHaveBeenCalled();
				expect(service.wipeSessionData).toHaveBeenCalled();
			});

			it('should be able to authorize users', function () {
				$httpBackend.expectGET(resourceDirectory['rest-v1']).respond(function (method, url, data, headers) {
					return [200, {}, {}, ''];
				});

				expect(service.isAuthorized()).toBe(false);

				service.checkAuthorizedKey();
				$scope.$digest();
				$httpBackend.flush();

				expect(localResourceDirectoryServiceMock.fetch).toHaveBeenCalled();
				expect(service.isAuthorized()).toBe(true);
			});

			it('should be able to confirm unauthorized users', function () {
				$httpBackend.expectGET(resourceDirectory['rest-v1']).respond(function (method, url, data, headers) {
					return [404, {}, {}, ''];
				});

				expect(service.isAuthorized()).toBe(false);

				service.checkAuthorizedKey();
				$scope.$digest();
				$httpBackend.flush();

				expect(localResourceDirectoryServiceMock.fetch).toHaveBeenCalled();
				expect(service.isAuthorized()).toBe(false);
			});
		});
	});
});