package gov.va.vamf.scheduling.varutility.domain;

import agilex.commons.NullChecker;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

@XmlAccessorType(value = XmlAccessType.NONE)
@XmlRootElement(namespace = Namespace.VarUtility)
public class CoreSettings extends DomainBaseObjects<CoreSetting> {

    public static final String PRIMARY_CARE_ID = "323";

    String[] booleans   = {"Yes", "No"};
    Integer[] durations = {365, 730};
    ArrayList<String> careTypeIdsList = new ArrayList<String>();
    ArrayList<String> validBooleans = new ArrayList<String>(Arrays.asList(booleans));
    ArrayList<Integer> validDurations = new ArrayList<Integer>(Arrays.asList(durations));

    public ArrayList<ValidationError> validate(List<ClinicalServices> clinicalServices) {
        ArrayList<ValidationError> errors = new ArrayList<ValidationError>();

        validateTypesOfCare(errors, clinicalServices);
        validatePatientHistoryRequired(errors);
        validateDurations(errors);

        return errors;
    }

    private void validateDurations(ArrayList<ValidationError> errors) {
        for(CoreSetting c : this) {
            if(NullChecker.isNotNullish(c.getPatientHistoryRequired()) &&
                c.getPatientHistoryRequired().equals("Yes") &&
                NullChecker.isNullish(c.getPatientHistoryDuration())) {
                errors.add(new ValidationError(c.getTypeOfCare(),
                    "PatientHistoryDuration is required."));
            }
            if(NullChecker.isNotNullish(c.getPatientHistoryDuration()) &&
                ! validDurations.contains(c.getPatientHistoryDuration())) {
                errors.add(new ValidationError(c.getTypeOfCare(),
                    "Valid values for patientHistoryDuration are 365, 730, or null"));
            }
        }
    }

    private void validatePatientHistoryRequired(ArrayList<ValidationError> errors) {
        for(CoreSetting c : this) {
            if(NullChecker.isNotNullish(c.getPatientHistoryRequired()) &&
                ! validBooleans.contains(c.getPatientHistoryRequired())) {
                errors.add(new ValidationError(c.getTypeOfCare(),
                    "Valid values for patientHistoryRequired are Yes, No, or null"));
            }
        }
    }

    private void validateTypesOfCare(ArrayList<ValidationError> errors, List<ClinicalServices> clinicalServices) {
        for (ClinicalServices clinicalService : clinicalServices) {
            careTypeIdsList.add(clinicalService.getId());
        }
        ArrayList<String> inputCareTypeIdsList = new ArrayList<String>();

        for(CoreSetting c : this) {
            inputCareTypeIdsList.add(c.getId());
        }

        if(! inputCareTypeIdsList.containsAll(careTypeIdsList)) {
            errors.add(new ValidationError("CoreSettings does not contain all care types."));
        }

        if(inputCareTypeIdsList.size() != careTypeIdsList.size()) {
            errors.add(new ValidationError("Invalid care types."));
        }
    }

    public void cleanupDataForSave() {
        for(CoreSetting cs : this) {
            if(cs.getId().equals(PRIMARY_CARE_ID)) {
                cs.setPatientHistoryRequired(null);
                cs.setPatientHistoryDuration(null);
            } else {
                if(NullChecker.isNullish(cs.getPatientHistoryRequired()) || cs.getPatientHistoryRequired().equals("No")) {
                    cs.setPatientHistoryDuration(0);
                }
            }
        }
    }
}
