#! /bin/bash
# ==== Begin Header ====
# tas-init.sh v2.1 2017-Dec-27 PII                      )
#  Add dos2unix, wget, yum-utils to standard builds
#  Remove password requirement for non-NFS/FPC builds
# tas-init.sh v2.0 2017-Dec-11 PII                      )
#  Remove Security settings and other parts that are now provided in
#  VA "Gold Image" (or self-created with SysAdmin mag-init.sh)
# tas-init.sh v1.0 2017-Dec-08 PII                      )
#  First real release
#
# These are the available command-line options:
#  -c or --nocolor : Turn Off Color Debug output
#  -l or --logfile : Change logfile (must come next)
#  -f or --fqdn    : Use FQDN instead of detecting it (must come next)
#  -u or --user    : Username to use to access BitBucket (mccf_devops)  
#  -p or --pass    : Password to go with username above
#  -v or --vpass   : Vault Password for Jenkins passwords & Private SSL Key
#
# This script assumes a system in one of the TAS environments
# - VA AITC (aka EDE) is the Austin IT Center (VA Private Cloud)
# - VA Azure (aka MAG) is the Microsoft Azure Gov Cloud
# - Halfaker AWS (aka AWS) is the Halfaker AWS Cloud (temporary)
# - Halfaker Azure (aka HAC) is the Halfaker Azure Cloud (undefined)
# - Individual dev systems are NOT supported and will likely be detected as AWS
#
# The tasks this script performs are essentially to create a system-start-point
# from which the TAS Jenkins Server can access the system and configure it 
# based on the requirements needed by the TAS Configuration Management Team
#
# A log of the progress and results of this script is located at $LOGTXT
# If run interactively, the progress & results (different from the log file)
#      are shown on the screen. If the colorized output of the interactive
#      screen are not working, you can turn off the color with a -c option
#
# ==== End Header ====
# Don't waste any time -- if you're not root, exit now
if [ "${UID}" != "0" ]; then
    echo "$0 Error: You must run this script as root" >&2 
    exit 1
fi

LOGFILE=/var/log/$0.log ; cp /dev/null ${LOGFILE} # create & blank the log file
DeployEnv=
MyIP=
MyHostname=
MyDomain=
MyFQDN=
MyDNS1=
MyDNS2=
MyCIServer=localhost
IsCIServer=
IsVA=No
VARH7BSTRAP="http://URL/pub/bootstrap/rhel7.sh"
VARH7LIC=
PreInstalledPkgs="git ansible dos2unix wget yum-utils"
#
# VA DNS Servers & TAS CI servers for EDE Environment
VAEDEDNS1=IP         
VAEDEDNS2=IP         
VAEDECI=URL
#
# VA DNS Servers & TAS CI servers for MAG Environment
# NOTE: For now, the DNS servers are identical to EDE DNS servers
#   That is expected to change at some time in the future
VAMAGDNS1=IP         
VAMAGDNS2=IP         
VAMAGCI=URL
#
# Someday we will have a HAC environment -- until then, treat as AWS
HACDNS1=8.8.4.4
HACDNS2=4.2.2.3
HACCI=aws-master.tas.local.
HACCIIP=IP          
#
# If we're not in the EDE or MAG, we assume AWS
AWSDNS1=8.8.4.4
AWSDNS2=4.2.2.3
AWSCI=aws-master.mccf.halfakerlabs.com
AWSCIIP=IP          
#
# Jenkins Public Key
JENKINSUSER=AI
JENKINSUID=410
JENKINSHOME=/home/jenkins
JENKINSSHELL=/bin/bash
JENKINSPUB="AI"
#
YUMSRC=bitbucket.org/halfaker/mccf_yum
YUMDIR=mccf_yum
YUMSH=./mvrepo.sh
MISCSRC=bitbucket.org/halfaker/mccf_misc_repo
MISCDIR=mccf_misc_repo
MISCSH=./mvrepo.sh
JENKINSSRC=bitbucket.org/halfaker/mccf_jenkins
JENKINSDIR=mccf_jenkins
JENKINSSH=./mvrepo.sh
DEVOPSSRC=bitbucket.org/halfaker/mccf_devops
DEVOPSDIR=mccf_devops
DEVOPSDIR2=playbooks
DEVOPSSH=./installJenkins.sh
#
# Define colors and terminal escape sequences for intereactive use
UseColor=Yes
CLEAR="\\033c"
COL60="\\033[60G"
NORMAL="\\033[0;39m"
WHITE="\\033[1;39m"
GREEN="\\033[1;32m"
RED="\\033[1;31m"
YELLOW="\\033[1;33m"
BLUE="\\033[1;34m"

SHOWPROCESS="${COL60}${BLUE}Processing${NORMAL}"
SHOWBUILD="${COL60}${BLUE}Building..${NORMAL}"
SHOWINSTALL="${COL60}${BLUE}Installing${NORMAL}"
SHOWSKIP="${COL60}${WHITE}[${YELLOW} SKIP ${WHITE}] ${NORMAL}"
SHOWOK="${COL60}${WHITE}[${GREEN} -OK- ${WHITE}] ${NORMAL}"
SHOWDONE="${COL60}${WHITE}[${GREEN} DONE ${WHITE}] ${NORMAL}"
SHOWPASS="${COL60}${WHITE}[${GREEN} PASS ${WHITE}] ${NORMAL}"
SHOWISSUE="${COL60}${WHITE}[${YELLOW}ISSUE ${WHITE}] ${NORMAL}"
SHOWFAIL="${COL60}${WHITE}[${RED} FAIL ${WHITE}] ${NORMAL}"
#
# LOGTXT sends arguments passed to the defined logfile
LOGTXT() {
   echo "$*" >> ${LOGFILE}
}
# DEBUG sends the arguments passed to the terminal with an EOL at the end
DEBUG() {
   if [ -t 1 ] ; then echo -e "$*" ; fi
}
# EDEBUG sends the arguments passed to the terminal WITH NO EOL character
EDEBUG() {
   if [ -t 1 ] ; then echo -en "$*" ; fi
}

# ReadArgs reads the command line arguments passed to it
# and polulates shell variables accordingly
ReadArgs() {
  LOGTXT "Entered ReadArgs `date`"
  while [[ $# -gt 0 ]] ; do
    key="$1"

    case ${key} in
      -c | --nocolor ) 
           UseColor=No
           CLEAR=""
           COL60=""
           NORMAL=""
           WHITE=""
           GREEN=""
           RED=""
           YELLOW=""
           BLUE=""
           shift # past argument
           LOGTXT "Processing without color"
           ;;
      -l | --logfile )
           if [[ $# -ge 2 ]] ; then
             LOGFILE="$2"
           else
             DEBUG "$1 argument requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           LOGTXT "Using LogFile ${LOGFILE} instead"
           # DEBUG  "Log output redirected to ${LOGFILE}"
           shift # past argument
           shift # past logfile entry
           ;;
      -f | --fqdn )
           if [[ $# -ge 2 ]] ; then
             MyFQDN="$2"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           LOGTXT "Overriding FQDN lookup. Using ${MyFQDN} instead"
           # DEBUG  "Overriding FQDN lookup. Using ${MyFQDN} instead"
           shift # past argument
           shift # past fqdn entry
           ;;
      -u | --user )
           if [[ $# -ge 2 ]] ; then
             BBUSER="$2"
             LOGTXT "Setting BitBucket User from Command Line"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           shift # past argument
           shift # past user entry
           ;;
      -p | --pass | --password )
           if [[ $# -ge 2 ]] ; then
             BBPASS="$2"
             DEBUG " "
             LOGTXT "Setting BitBucket Password from Command Line"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           shift # past argument
           shift # past password entry
           ;;
      -v | --vpass | --vault | --vault-password )
           if [[ $# -ge 2 ]] ; then
             JVPASS="$2"
             LOGTXT "Setting Jenkins Vault Password from Command Line"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           shift # past argument
           shift # past vault password entry
           ;;
      * )  DEBUG "$1 is not a valid argument"
           LOGTXT "$1 is not a valid argument"
           exit 9
           ;;
    esac
  done
  LOGTXT "Exited ReadArgs"
}

# DetectSystem attempts to identify the system we're on based on IP address
# and PTR record (if any)
DetectSystem() {
  LOGTXT "Entering DetectSystem `date`"
  EDEBUG "Detecting System IP: "
  # This method uses our defined gateway to determine the IP address used
  # for getting out... in most cases, this will be our primary IP address
  InternetIP="`ip route get 8.8.8.8 | awk '{print $NF; exit}'`"
  LOGTXT "InternetIP found to be ${InternetIP}"
  # The hostname -I option returns ALL IPs (other than loopback)
  # In the EDE environment there are often 2 interfaces
  IPs="`hostname -I`"
  LOGTXT "List of all IPs on our interfaces is: ${IPs}"
  MyIP=${InternetIP}
  #
  # If we've been given the FQDN, don't bother detecting it
  if [ ! -z "${MyFQDN}" ] ; then
    LOGTXT "Using provided FQDN ${MyFQDN} and routable IP ${MyIP}"
    EDEBUG "${WHITE}${MyIP}${NORMAL}"
    DEBUG "${SHOWOK}"
  # We weren't told our FQDN, so we have to try to GET it
  # Since we're primarily designed for operating in VA environments, the
  # first way we try to determine our FQDN (and IP) is to try to get it from
  # the VA DNS servers
  elif PTR="`host ${MyIP} ${VAMAGDNS1} | grep 'va.gov'`" ; then
    MyFQDN="`echo ${PTR} | sed 's/.* pointer //'`"
    LOGTXT "${MyIP} has PTR of ${MyFQDN} from ${VAMAGDNS1}"
    EDEBUG "${WHITE}${MyIP}${NORMAL}"
    DEBUG "${SHOWOK}"
  # If our IP isn't listed in the VA DNS server (and it wasn't provided)
  # We have little choice (until HAC is created) but to assume AWS
  elif PTR="`host ${MyIP} ${AWSCI} | grep 'tas.local'`" ; then
    MyFQDN="`echo ${PTR} | sed 's/.* pointer //'`"
    LOGTXT "${MyIP} has PTR of ${MyFQDN} from AWS DNS"
    EDEBUG "${WHITE}${MyIP}${NORMAL}"
    DEBUG "${SHOWOK}"
  else 
    LOGTXT "Failed to determine which IP to use"
    EDEBUG "${YELLOW}${IPs}${NORMAL}"
    DEBUG "${SHOWISSUE}"
    exit 10
  fi
  #
  # At this point we should have defined MyFQDN and MyIP
  # Lets verify and set the MyHostname & MyDomain settings
  EDEBUG "Detecting System names: "
  if [ -z "${MyFQDN}" ] ; then
    LOGTXT "Failed to find an FQDN: fix DNS/PTR or specifiy on the command line with -f"
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failed to find your FQDN in DNS"
    DEBUG  " - Fix your DNS (PTR) entries for this IP address, or"
    DEBUG  " - use the --fqdn option to force the value in this script"
    exit 10
  elif ! MyHostname="`echo ${MyFQDN} | awk -F. '{ print $1 }'`" ; then
    LOGTXT "Error getting hostname from ${MyFQDN}"
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failed to get a valid hostname from your FQDN: ${MyFQDN}"
    exit 10
  elif ! MyDomain="`echo ${MyFQDN} | sed s/^${MyHostname}\.//`" ; then
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failed to get a valid domain name from your FQDN: ${MyFQDN}"
    LOGTXT "Error getting domain from ${MyFQDN}"
    exit 10
  fi
  EDEBUG "${WHITE}${MyFQDN}${NORMAL}"
  DEBUG "${SHOWOK}"

  # Next we need to set the hostname
  EDEBUG "Setting Hostname to: "
  if ! hostnamectl set-hostname ${MyFQDN} ; then
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failed to set hostname to ${MyFQDN} - Unknown cause"
    LOGTXT "Error setting hostname to ${MyFQDN}"
    exit 10
  fi
  # Ensure that we have ourselves listed in /etc/hosts
  if ! grep -q ${MyIP} /etc/hosts ; then
    echo ${MyIP} ${MyHostname} ${MyFQDN} >> /etc/hosts
  fi
  EDEBUG "${WHITE}${MyFQDN}${NORMAL}"
  DEBUG "${SHOWOK}"

  # Finally we need to determine our enviroment: MAG, EDE, AWS, or HAC
  EDEBUG "Detected System Environment is: "
  if `echo ${MyFQDN} | grep '.va.gov' > /dev/null 2>&1` ; then
    IsVA=Yes
    if   [ "${MyHostname:2:3}" == "c20" ] ; then
      DeployEnv=MAG
      MyCIServer=http://URL
      EDEBUG "${WHITE}MAG${NORMAL}"
      DEBUG  "${SHOWOK}"
    elif [ "${MyHostname:2:3}" == "aus" ] ; then
      DeployEnv=EDE
      MyCIServer=http://URL
      EDEBUG "${WHITE}EDE${NORMAL}"
      DEBUG  "${SHOWOK}"
    else
      DEBUG "${SHOWFAIL}"
      DEBUG  "Error determining VA install environment -- check hostname"
      LOGTXT "Error determining VA install environment -- check hostname"
      exit 10
    fi
    if [ "${MyHostname:5:3}" == "fpc" ] ; then
      IsCIServer=true
    fi
  # ===
  # At some point in the future, we will detect Halfaker Azure Cloud env here
  # ===
  elif `echo ${MyFQDN} | grep '.halfakerlabs.com' > /dev/null 2>&1` ; then
    IsVA=No
    DeployEnv=AWS 
    MyCIServer=http://aws-master.mccf.halfakerlabs.com
    if [ "${MyHostname:4:6}" == "master" ] ; then
      IsCIServer=true
    fi
    EDEBUG "${WHITE}AWS${NORMAL}"
    DEBUG  "${SHOWOK}"
  elif `echo ${MyFQDN} | grep '.tas.local' > /dev/null 2>&1` ; then
    IsVA=No
    DeployEnv=AWS 
    MyCIServer=http://aws-master.mccf.halfakerlabs.com
    if [ "${MyHostname:4:6}" == "master" ] ; then
      IsCIServer=true
    fi
    EDEBUG "${WHITE}AWS${NORMAL}"
    DEBUG  "${SHOWOK}"
  else
    DEBUG "${SHOWFAIL}"
    DEBUG  "Error determining Non-VA install environment -- check domain name"
    LOGTXT "Error determining Non-VA install environment -- check domain name"
    exit 10
  fi
  LOGTXT "${MyIP} is ${MyHostname}.${MyDomain} in ${DeployEnv}"
  LOGTXT "DetectSystem exited"
}
# Make use of ALL of the allocated storage!
GrowFS() {
  LOGTXT "Entering GrowFS `date`"
  EDEBUG "Verifiying and Growing Filesystems: "
  Err=
  #
  # Grow Home to 10GB (from default 2GB)
  FSYS=/home
  SIZE=10GB
  EXTENT="`df -h ${FSYS} | grep ${FSYS}  | awk '{ print $1 }'`"
  LOGTXT "Setting size of ${FSYS} (${EXTENT}) to ${SIZE}"
  EDEBUG "${WHITE}${FSYS} "
  if ! lvextend -r -L ${SIZE} ${EXTENT} >> ${LOGFILE} 2>&1 ; then
    LOGTXT "Error returned from lvextend -- continuing though"
    DEBUG "\n${WHITE}Problem growing ${FSYS} to ${SIZE}${SHOWISSUE}"
    Err=1
  fi
  #
  # Grow Opt to 10GB (from default 2GB)
  FSYS=/opt
  SIZE=10GB
  EXTENT="`df -h ${FSYS} | grep ${FSYS} | awk '{ print $1 }'`"
  LOGTXT "Setting size of ${FSYS} (${EXTENT}) to ${SIZE}"
  EDEBUG "${WHITE}${FSYS} "
  if ! lvextend -r -L ${SIZE} ${EXTENT} >> ${LOGFILE} 2>&1 ; then
    LOGTXT "Error returned from lvextend -- continuing though"
    DEBUG "\n${WHITE}Problem growing ${FSYS} to ${SIZE}${SHOWISSUE}"
    Err=1
  fi
  #
  # Grow Var to 20GB (this is where web pages go!)
  FSYS=/var
  SIZE=20GB
  EXTENT="`df -h ${FSYS} | grep ${FSYS} | awk '{ print $1 }'`"
  LOGTXT "Setting size of ${FSYS} (${EXTENT}) to ${SIZE}"
  EDEBUG "${WHITE}${FSYS} "
  if ! lvextend -r -L ${SIZE} ${EXTENT} >> ${LOGFILE} 2>&1 ; then
    LOGTXT "Error returned from lvextend -- continuing though"
    DEBUG "\n${WHITE}Problem growing ${FSYS} to ${SIZE}${SHOWISSUE}"
    Err=1
  fi
  LOGTXT "GrowFS Exited `date`"
  if [ -z ${Err} ] ; then DEBUG "${SHOWOK}" ; fi
}

GetPackages() {
  LOGTXT "Entering GetPackages `date`"
  # The playbooks that complete startup require Ansible and are retrieved with
  # GIT -- so we need to fetch them with YUM
  EDEBUG "Installing Pre-Install Packages..."
  if ! yum install ${PreInstalledPkgs} -y --disablerepo=\*mccf\* >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failure installing Pre-Install Packages - See Log File"
    LOGTXT "Failure installing Pre-Install Packages"
    exit 15
  fi
  DEBUG "${SHOWOK}"

  LOGTXT "Exiting GetPackages"
}
#
# EnableJenkinsAcess creates the MCCF Jenkins user on the system and
# installs SSH KEY level access to it. We also make sure that this
# Jenkins user cannot login with any password, but can with a key
# Finally, we make this Jenkins user a SUDOer with no need for a password
EnableJenkinsAccess() {
  LOGTXT "Entering EnableJenkinsAccess"
  EDEBUG "Enabling MCCF Jenkins user: "
  # Check if group exists, and if not create it
  EDEBUG "${WHITE}Group ${NORMAL}"
  if [ "`awk -F: '{ print $1 }' /etc/group | grep -c jenkins`" -eq 0 ] ; then
    if ! groupadd -g ${JENKINSUID} ${JENKINSUSER} >> ${LOGFILE} 2>&1 ; then
      DEBUG "${SHOWFAIL}"
      DEBUG "Failed to add Jenkins group -- check error logs"
      exit 12
    fi
  fi
  # Check if user exists, and if not create it
  EDEBUG "${WHITE}User ${NORMAL}"
  if [ "`awk -F: '{ print $1 }' /etc/passwd | grep -c jenkins`" -eq 0 ] ; then
    if ! useradd -s ${JENKINSSHELL} -m -d ${JENKINSHOME} -c "Jenkins Access System Account" -u ${JENKINSUID} -g ${JENKINSUID} ${JENKINSUSER} >> ${LOGFILE} 2>&1 ; then
      DEBUG "${SHOWFAIL}"
      DEBUG "Failed to add Jenkins user -- check error logs"
      exit 12
    fi
  fi
  # Create .ssh folder 
  EDEBUG "${WHITE}.ssh ${NORMAL}"
  if [ ! -d ${JENKINSHOME}/.ssh ] ; then
    if ! mkdir -p ${JENKINSHOME}/.ssh ; then
      DEBUG "${SHOWFAIL}"
      DEBUG "Failed to create Jenkins .ssh"
      exit 12
    fi
  fi
  EDEBUG "${WHITE}auth_key ${NORMAL}"
  if [ ! -f ${JENKINSHOME}/.ssh/authorized_keys ] ; then
    echo ${JENKINSPUB} > ${JENKINSHOME}/.ssh/authorized_keys
  fi
  if ! /bin/chown -R jenkins:jenkins ${JENKINSHOME}/.ssh ; then
    DEBUG "${SHOWFAIL}"
    DEBUG "Failed to set owner for Jenkins .ssh"
    exit 12
  elif ! /bin/chmod 700 ${JENKINSHOME}/.ssh ; then
    DEBUG "${SHOWFAIL}"
    DEBUG "Failed to set permissions for Jenkins .ssh"
    exit 12
  elif ! /bin/chmod 400 ${JENKINSHOME}/.ssh/authorized_keys ; then
    DEBUG "${SHOWFAIL}"
    DEBUG "Failed to set permissions for Jenkins keys"
    exit 12
  fi
  EDEBUG "${WHITE}sudo ${NORMAL}"
  JenkinsSudo=/etc/sudoers.d/jenkins
  if [ ! -f ${JenkinsSudo} ] ; then
    echo "jenkins ALL=(ALL) NOPASSWD:ALL" > ${JenkinsSudo}
    chmod 400 ${JenkinsSudo}
  fi
  DEBUG "${SHOWOK}"
  LOGTXT "Exiting EnableJenkinsAccess"
}

#
DeployYUMRepo() {
  LOGTXT "Entering DeployYUMRepo"
  EDEBUG "Downloading YUM Repo for CI Server..."
  STARTDIR="`pwd`"
  # Clean out any prior git operation for this project
  if [ -d "${YUMDIR}" ] ; then rm -rf "${YUMDIR}" ; fi
  # Fetch the project using git with the provided credentials
  if ! git clone https://${BBUSER}:${BBPASS}@${YUMSRC} >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}" 
    DEBUG " Error obtaining playbook -- check the logfile @ ${LOGFILE}"
    DEBUG " However, the most likely error is bad BitBucket permissions"
    LOGTXT "Sorry! git clone failed..."
    exit 21
  else
    DEBUG "${SHOWOK}"
  fi
  if [ ! -d "${YUMDIR}" ] ; then
    DEBUG "${SHOWFAIL}"
    DEBUG " Error accessing folder `pwd`/${YUMDIR}"
    exit 21
  elif [ ! -f "${YUMDIR}/${YUMSH}" ] ; then
    DEBUG "${SHOWFAIL}"
    DEBUG " Error accessing file `pwd`/${YUMDIR}/${YUMSH}" 
    exit 21
  fi
  cd "${YUMDIR}"
  EDEBUG "Deploying YUM Repo Locally..."
  if ! ${YUMSH} >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}"
    DEBUG "YUM Installer Failed -- check logs: ${LOGFILE}"
    exit 21
  else
    DEBUG "${SHOWOK}"
  fi
  cd "${STARTDIR}"
  rm -rf "${YUMDIR}"
  EDEBUG "Starting Apache HTTP service for YUM repo..."
  if ! yum install httpd -y --disablerepo=\*mccf\* >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}"
    DEBUG "Apache httpd Installer Failed -- check logs: ${LOGFILE}"
    exit 21
  elif ! systemctl enable httpd >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}"
    DEBUG "Apache httpd Installer Failed -- check logs: ${LOGFILE}"
    exit 21
  elif ! systemctl start httpd >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}"
    DEBUG "Apache httpd Startup Failed -- check logs: ${LOGFILE}"
    exit 21
  elif ! firewall-cmd --zone=public --add-service=http --permanent >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}"
    DEBUG "Firewalld add for httpd Failed -- check logs: ${LOGFILE}"
    exit 21
  else
    DEBUG "${SHOWOK}"
  fi
  LOGTXT "Exiting DeployYUMRepo"
}
#
# SubscribeMCCFRepo 
SubscribeMCCFRepo() {
  LOGTXT "Entering SubscribeMCCFRepo"
  EDEBUG "Subscribing to additional YUM Repos: ${WHITE}MCCF ${NORMAL}"
  cat > /etc/yum.repos.d/mccf.repo << -END
[mccf]
name=Halfaker MCCF Distributable (to the VA)
baseurl=${MyCIServer}/pub/repo/yum/MCCF
gpgcheck=1
gpgkey=${MyCIServer}/pub/repo/yum/RPM-GPG-KEY-MCCF
enabled=1
-END
  LOGTXT "Created mccf.repo"
  if [ "${DeployEnv}" == "AWS" ] ; then
    EDEBUG "${YELLOW}MCCF-VA ${NORMAL}"
    cat >> /etc/yum.repos.d/mccf.repo << -END
[mccf-va]
name=Halfaker MCCF Dev Repo for VA supplied packages
baseurl=${MyCIServer}/dev/repo/yum/MCCF-VA
gpgcheck=1
gpgkey=${MyCIServer}/pub/repo/yum/RPM-GPG-KEY-MCCF
enabled=1
-END
    LOGTXT "Added mccf-va to  mccf.repo"
    EDEBUG "${YELLOW}MCCF-DEV ${NORMAL}"
    cat >> /etc/yum.repos.d/mccf.repo << -END
[mccf-dev]
name=Halfaker MCCF Dev Repo 
baseurl=${MyCIServer}/dev/repo/yum/MCCF-DEV
gpgcheck=1
gpgkey=${MyCIServer}/pub/repo/yum/RPM-GPG-KEY-MCCF
enabled=0
-END
    LOGTXT "Added mccf-dev to  mccf.repo"
  fi
  DEBUG "${SHOWOK}"
  LOGTXT "Exiting SubscribeMCCFRepo"
}
#
# InstallCI Server (aka NFS or Master)
DeployJenkins() {
  LOGTXT "Entering DeployJenkins"
  #
  # If BitBucket Username, Password, or Vault Passwords are not provided...
  if [ -z "${BBUSER}" ] ; then
    EDEBUG "${WHITE}Please enter your Atlassian BitBucket Account User Name: ${YELLOW}"
    read BBUSER
    EDEBUG "${NORMAL}"
  fi
  if [ -z "${BBPASS}" ] ; then
    EDEBUG "${WHITE}Please enter your password for the account ${BBUSER}: ${NORMAL}"
    read -s BBPASS
    DEBUG " "
  fi
  if [ -z "${JVPASS}" ] ; then
    EDEBUG "${WHITE}For CI Servers, please enter the Jenkins Vault Password: ${NORMAL}"
    read -s JVPASS
    DEBUG " "
  fi
  EDEBUG "Setting Jenkins Vault Passwords..."
  echo "${JVPASS}" > ~root/.vault_pass    ; chmod 400 ~root/.vault_pass
  echo "${JVPASS}" > ~jenkins/.vault_pass ; chmod 400 ~jenkins/.vault_pass ; chown -R jenkins:jenkins ~jenkins/.vault_pass
  DEBUG "${SHOWOK}"
  EDEBUG "Getting Jenkins Repo for CI Server..."
  STARTDIR="`pwd`"
  # Clean out any prior git operation for this project
  if [ -d "${JENKINSDIR}" ] ; then rm -rf "${JENKINSDIR}" ; fi
  # Fetch the project using git with the provided credentials
  if ! git clone https://${BBUSER}:${BBPASS}@${JENKINSSRC} >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}" 
    DEBUG " Error obtaining playbook -- check the logfile @ ${LOGFILE}"
    exit 21
  else
    DEBUG "${SHOWOK}"
  fi
  EDEBUG "Running Jenkins Repo Installer..."
  if [ ! -d "${JENKINSDIR}" ] ; then
    DEBUG "${SHOWFAIL}"
    DEBUG " Error accessing folder `pwd`/${JENKINSDIR}"
    exit 21
  elif [ ! -f "${JENKINSDIR}/${JENKINSSH}" ] ; then
    DEBUG "${SHOWFAIL}"
    DEBUG " Error accessing file `pwd`/${JENKINSDIR}/${JENKINSSH}" 
    exit 21
  fi
  cd "${JENKINSDIR}"
  if ${JENKINSSH} >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWOK}"
  else
    DEBUG "${SHOWFAIL}"
    DEBUG "Jenkins Repo Installer Failed -- check logs: ${LOGFILE}"
    exit 21
  fi
  cd "${STARTDIR}"
  rm -rf "${JENKINSDIR}"
  #
  EDEBUG "Getting DevOps Repo for CI Server..."
  # Clean out any prior git operation for this project
  if [ -d "${DEVOPSDIR}" ] ; then rm -rf "${DEVOPSDIR}" ; fi
  # Fetch the project using git with the provided credentials
  if ! git clone https://${BBUSER}:${BBPASS}@${DEVOPSSRC} >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}" 
    DEBUG " Error obtaining playbook -- check the logfile @ ${LOGFILE}"
    exit 21
  else
    DEBUG "${SHOWOK}"
  fi
  EDEBUG "Running Devops Jenkins Installer..."
  if [ ! -d "${DEVOPSDIR}" ] ; then
    DEBUG "${SHOWFAIL}"
    DEBUG " Error accessing folder `pwd`/${DEVOPSDIR}"
    exit 21
  elif [ ! -d "${DEVOPSDIR}/${DEVOPSDIR2}" ] ; then
    DEBUG "${SHOWFAIL}"
    DEBUG " Error accessing folder `pwd`/${DEVOPSDIR}/${DEVOPSDIR2}"
    exit 21
  else
    cd "${DEVOPSDIR}/${DEVOPSDIR2}"
    if [ ! -f "${DEVOPSSH}" ] ; then
      DEBUG "${SHOWFAIL}"
      DEBUG " Error accessing file `pwd`/${DEVOPSSH}" 
      exit 21
    fi
  fi
#
#
# THIS HAS TO BE HASHED OUT BEFORE RUNNING ANYWHERE ELSE BUT MAG
#
#
  if ${DEVOPSSH} -e MAG_DEV -u ${BBUSER} -p ${BBPASS} -l /var/log/${DEVOPSSH}.log >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWOK}"
  else
    DEBUG "${SHOWFAIL}"
    DEBUG " Installer Failed -- check logs: ${LOGFILE}"
    exit 21
  fi
  cd "${STARTDIR}"
  rm -rf "${DEVOPSDIR}"
  DEBUG "${SHOWOK}"
  LOGTXT "Exiting DeployJenkins"
}
#
# 
#
# THIS IS THE START OF ACTUAL PROCESSING
#
#

LOGTXT "Starting $0 at `date`"
EDEBUG ${CLEAR}
DEBUG "Welcome to the TAS System Preparation Script."
DEBUG "We need to confirm and/or set some things up before we can integrate"
DEBUG "this system into TAS. Please pay close attention to the output below:"
DEBUG "Additionally, a log file is created at ${LOGFILE}"
DEBUG " "

ReadArgs $*
DetectSystem
GrowFS
EnableJenkinsAccess
GetPackages
if [ ! -z ${IsCIServer} ] ; then
  DeployYUMRepo
  SubscribeMCCFRepo
  DeployJenkins
else
  SubscribeMCCFRepo
fi

DEBUG "All done... everything passed. Rebooting system in 30 seconds for initial use"
DEBUG "Press CTRL-C to abort the reboot"
LOGTXT "Completed $0 at `date`"
sleep 30
PARENT="`pwd | sed 's#/mccf_devops/.*#/#'`"
echo $PARENT
cd "${PARENT}"
rm -rf mccf_devops
init 6
