#!/bin/bash
#
#set -x

scriptname=${0}
scriptBase="$( cd "$( dirname "$scriptname" )";  pwd)"
VAULTPASS_PATH="~/.vault_pass"
REPO_FILE="jenkins-requirements.yml"
ROOT_DIR="$( cd ..;  pwd)"
TIMESTAMP=`date "+%Y%m%d_%H%M%S"`
LOGFILE="${scriptBase}/jenkinsInstall_${TIMESTAMP}.log"

function usage() {
  echo "usage: $scriptname -e ENV"
  echo "      where ENV = AWS_DEV | EDE_ALL | EDE_CI | EDE_CIT | EDE_DEV | EDE_SQA | EDE_UAT | MAG_ALL | MAG_CI | MAG_CIT | MAG_DEV | MAG_SQA | MAG_UAT | local"
  echo "optional arguments: "
  echo "  -u|--user Username"
  echo "  -p|--pass Password"
  echo "  -l|--logfile logfile"
  echo "  -c|--nocolor (for nocolor)]"
  exit 1
}

function runPlaybook() {
  if [ $# -eq 1 ]
  then
    LOGTXT "Running ${ENV} install for ${1}"
    cd "${ROOT_DIR}/playbooks"
    if [[ "${ENV}" == "local" ]]
    then
      DEBUG "ansible-playbook ${1} -i ../environments/${ENV} --extra-vars var_hosts=${HOSTS}"
      ansible-playbook ${1} -i "../environments/${ENV}" --extra-vars "var_hosts=${HOSTS}"
    else
      DEBUG "ansible-playbook ${1} -i ../environments/${ENV} --extra-vars var_hosts=${HOSTS} --vault-password-file ${VAULTPASS_PATH}"
      ansible-playbook ${1} -i "../environments/${ENV}" --extra-vars "var_hosts=${HOSTS}" --vault-password-file ${VAULTPASS_PATH}
    fi
  else 
    echo "No playbook specified"
    exit 1
  fi
}

function getCredentials(){
  DEBUG "Collecting credentials to pull source from bitbucket"
  echo -n "Please enter bitbucket username: "
  read USERNAME
  # 8m is the control code to hide characters
  echo -n "Please enter bitbucket password: "
  read -s PASSWD
}

function installRoles(){
  if [[ ! -e ${REPO_FILE} ]]
  then
    DEBUG "${REPO_FILE} not found, exiting"
    exit 1
  fi
  echo "processing: ${REPO_FILE}"
  REPOS=$(sed -n -e '/^\s*name/p' ${REPO_FILE} | cut -d':' -f 2 | sed 's/^ *//;s/ *$//')
  for repo in ${REPOS} ; do
    PROJECT_NAME="${repo}"
    DEBUG "--> ${PROJECT_NAME}"
    PROJECT_PATH="${ROOT_DIR}/roles/${PROJECT_NAME}"
    DEBUG "--> ${PROJECT_PATH}"
    # Make sure the required directories don't exist
    if [ -e "${PROJECT_PATH}" ] ; then
      DEBUG -e "Role ${PROJECT_NAME} already exists!"
      cd "${ROOT_DIR}/roles"
      rm -rf "${PROJECT_NAME}"
    fi
      # create get the role
    echo -e "INFO: getting role ${PROJECT_NAME}..."
    echo -e "===================================================="
    cd "${ROOT_DIR}/roles"
    mkdir ${PROJECT_NAME}

    USERORIGIN="https://${USERNAME}:${PASSWD}@bitbucket.org/halfaker/${PROJECT_NAME}.git"
    git clone ${USERORIGIN} "/tmp/${PROJECT_NAME}"
    if (( $? == 0 )); then
      rsync -av --exclude ".git*" --exclude ".git/" "/tmp/${PROJECT_NAME}" .
      rm -rf "/tmp/${PROJECT_NAME}"
    else
      DEBUG "problem collecting repo for ${USERORIGIN}"
      DEBUG "please verify you entered the username and password correctly"
      echo "all specified repos are required, exiting"
      exit 1
    fi
    echo
  done
}
#
# LOGTXT sends arguments passed to the defined logfile
LOGTXT() {
   echo "$(date) $*" >> $LOGFILE
}
# DEBUG sends the arguments passed to the terminal with an EOL at the end
DEBUG() {
   if [ -t 1 ] ; then echo -e "$(date) $*" ; fi
}
# EDEBUG sends the arguments passed to the terminal WITH NO EOL character
EDEBUG() {
   if [ -t 1 ] ; then echo -en "$*" ; fi
}
# ReadArgs reads the command line arguments passed to it
# and polulates shell variables accordingly
ReadArgs() {
  LOGTXT "Entered ReadArg"
  while [[ $# -gt 0 ]] ; do
    key="$1"
    case $key in
      -c | --nocolor )
           UseColor=No
           CLEAR=""
           COL60=""
           NORMAL=""
           WHITE=""
           GREEN=""
           RED=""
           YELLOW=""
           BLUE=""
           shift # past argument
           LOGTXT "Processing without color"
           ;;
      -l | --logfile )
           if [[ $# -ge 2 ]] ; then
             LOGFILE="$2"
           else
             DEBUG "$1 argument requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           LOGTXT "Using LogFile $LOGFILE instead"
           # DEBUG  "Log output redirected to $LOGFILE"
           shift # past argument
           shift # past logfile entry
          ;;
      -u | --user )
           if [[ $# -ge 2 ]] ; then
             USERNAME="$2"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           LOGTXT "USERNAME: $USERNAME "
           # DEBUG  "USERNAME: $USERNAME "
           shift # past argument
           shift # past fqdn entry
           ;;
      -p | --pass )
           if [[ $# -ge 2 ]] ; then
             PASSWD="$2"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           LOGTXT "Using supplied password "
           # DEBUG  "Using supplied password $PASSWD "
           shift # past argument
           shift # past fqdn entry
           ;;
      -e | --env )
           if [[ $# -ge 2 ]] ; then
             ENV="$2"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           LOGTXT "Environment: $ENV"
           # DEBUG  "Environment $ENV"
           shift # past argument
           shift # past fqdn entry
           ;;
      * )  DEBUG "$1 is not a valid argument"
           LOGTXT "$1 is not a valid argument"
           exit 9
           ;;
    esac
  done
  LOGTXT "Exited ReadArgs"
}

ReadArgs $*

DEBUG "vault pass path: ${VAULTPASS_PATH}"
DEBUG "logging to: ${LOGFILE}"

if [[ -z "${ENV}" ]]; then
  DEBUG "ENV required"
  usage
else
  DEBUG "operating on ENV ${ENV}"
fi

if [[ -z "${USERNAME}" ]]; then
  DEBUG "Credentials are required"
  getCredentials
fi

if [[ -z "${PASSWD}" ]]; then
  DEBUG "Credentials are required"
  getCredentials
fi

HOSTS=ci-server
playbooks=("install-jenkins.yml")

DEBUG "Starting to process playbooks on ${HOSTS} for $ENV environment"

DEBUG "Install roles from ${REPO_FILE}"
installRoles ${REPO_FILE}
  
for item in "${playbooks[@]}"
do
  playBook=${item}
  LOGTXT "execute playbook ${playBook} for ${ENV} ${HOSTS}"
  runPlaybook ${playBook}
  rtnCode=$?
  if [ $rtnCode -eq 0 ]
  then
    DEBUG "${playBook} completed successfully"
    LOGTXT "${playBook} completed successfully for ${ENV} on ${HOSTS}"
  else
    DEBUG "${playBook} returned: ${rtnCode}"
    exit $rtnCode
  fi
done

if [[ "${ENV}" == "local" ]]
then
  DEBUG "Admin User is is 'jenkins_admin'"
  DEBUG "Password is 'Password123!'"
fi
      # CONTACT INFO
CONTACT="Lee Benhart: Lee.Benhart@Halfaker.com or PII"

exit
