import { Injectable } from '@angular/core'
import { Http, Response } from '@angular/http'
import { Observable } from 'rxjs/Rx'
import 'rxjs/add/operator/map'
import 'rxjs/add/operator/toPromise'
import { environment } from '../../../../environments/environment'
import { ErrorService } from '../../../core/error-messages/error.service'
import { Resource } from './resource'
import { FhirOrganization } from './fhir-organization'
import { FhirPatient } from './fhir-patient'
import { FhirPractitioner } from './fhir-practitioner'
import { FhirCondition } from './fhir-condition'
import { FhirFind } from './fhir-find'
import { Record, RecordItem } from './record'

export interface ResourceInt {
  id: string,
  util: string,
  type: string
}

export const FhirServiceConst = {
  utilGetRecord: "RECORD",
  utilFind: "FIND",
  typeOrg: "ORGANIZATION",
  typePrc: "PRACTITIONER",
  typePat: "PATIENT",
  typeCond: "CONDITION",
}

@Injectable()
export class FhirService {
  fhirUrl = ''
  util = ''
  type = ''

  constructor(private http: Http, private errorService: ErrorService) { }

  async getResource(param: ResourceInt): Promise<Resource> {
    this.type = param.type
    this.util = param.util
    if (environment.fhirUrl) {
      this.fhirUrl = environment.fhirUrl
    } else {
      console.error("fhirUrl not set in environment")
    }
    let id = encodeURI(param.id)
    let url = this.fhirUrl + '/' + this.util + '/' + this.type + '/' + id

    if (this.util === FhirServiceConst.utilFind) {
      return await this.http
        .get(url)
        .map(mapFind)
        .catch(this.handleError)
        .toPromise()

    } else if (this.util === FhirServiceConst.utilGetRecord) {
      if (this.type === FhirServiceConst.typeOrg) {
        return await this.http
          .get(url)
          .map(mapOrganization)
          .catch(this.handleError)
          .toPromise()

      } else if (this.type === FhirServiceConst.typePrc) {
        return await this.http
          .get(url)
          .map(mapPractitioner)
          .catch(this.handleError)
          .toPromise()

      } else if (this.type === FhirServiceConst.typePat) {
        return await this.http
          .get(url)
          .map(mapPatient)
          .catch(this.handleError)
          .toPromise()

      } else if (this.type === FhirServiceConst.typeCond) {
        return await this.http
          .get(url)
          .map(mapCondition)
          .catch(this.handleError)
          .toPromise()

      } else {
        return null
      }

    } else {
      return null
    }

  }

  private handleError(error: any): Promise<any> {
    console.error('Error: ' + error)

    error = (error.message || JSON.parse(error._body) || error)
    if (error.error) {
      error = error.error
    }

    return Promise.reject(error)
  }

  async getRecord(params: string, id: string ): Promise<Array<Record>> {

    if (environment.fhirUrl) {
      this.fhirUrl = environment.fhirUrl
    } else {
      console.error("fhirUrl not set in environment")
    }

    let url = this.fhirUrl + '/RECORD/' + params + ';/' + id
//    console.log("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ record url="+url)

    return await this.http.get(url)
      .map(mapRecord)
      .catch(this.handleError)
      .toPromise()
  }

}

function mapRecord(response: Response): Array<Record> {
  let r = response.json()
  let records: Array<Record> = []

  if ( r.length === undefined) {
    let record = new Record()

    Object.keys(r).map(function(data) {
      let rec = new RecordItem()
      rec.title = data
      rec.value = r[data]
      record.records.push(rec)
    })
    records.push(record)

    return records
  }

  for (let i = 0; i < r.length; i++) {
    let record = new Record()

    let arr = Object.keys(r[i]).map(function(data) {
      let rec = new RecordItem()
      rec.title = data
      rec.value = r[i][data]
      record.records.push(rec)
    })

    records.push(record)
  }

  return records
}



function mapOrganization(response: Response): Resource {
  let r = response.json()
  let resource: Resource = new Resource()

  if (r.hasOwnProperty('401')) {
    resource.returnCode = '401'
    return resource
  }

  for (let i = 0; i < r.length; i++) {

    let org = new FhirOrganization()

    org.acosHospitalId = r[i]["ACOS HOSPITAL ID"]
    org.agencyCode = r[i]["AGENCY CODE"]
    org.billingFacilityName = r[i]["BILLING FACILITY NAME"]
    org.city = r[i]["CITY"]
    org.cityMailing = r[i]["CITY (MAILING)"]
    org.currentLocation = r[i]["CURRENT LOCATION"]
    org.district = r[i]["DISTRICT"]
    org.domain = r[i]["DOMAIN"]
    org.facilityDeaExpirationDate = r[i]["FACILITY DEA EXPIRATION DATE"]
    org.facilityDeaNumber = r[i]["FACILITY DEA NUMBER"]
    org.facilityType = r[i]["FACILITY TYPE"]
    org.inactiveFacilityFlag = r[i]["INACTIVE FACILITY FLAG"]
    org.multiDivisionFacility = r[i]["MULTI-DIVISION FACILITY"]
    org.name = r[i]["NAME"]
    org.npi = r[i]["NPI"]
    org.officialVaName = r[i]["OFFICIAL VA NAME"]
    org.pointerToAgency = r[i]["POINTER TO AGENCY"]
    org.region = r[i]["REGION"]
    org.reportingStation = r[i]["REPORTING STATION"]
    org.shortName = r[i]["SHORT NAME"]
    org.stAddr_1Mailing = r[i]["ST. ADDR. 1 (MAILING)"]
    org.stAddr_2Mailing = r[i]["ST. ADDR. 2 (MAILING)"]
    org.state = r[i]["STATE"]
    org.stateMailing = r[i]["STATE (MAILING)"]
    org.stationNumber = r[i]["STATION NUMBER"]
    org.status = r[i]["STATUS"]
    org.streetAddr_1 = r[i]["STREET ADDR. 1"]
    org.streetAddr_2 = r[i]["STREET ADDR. 2"]
    org.vaTypeCode = r[i]["VA TYPE CODE"]
    org.zip = r[i]["ZIP"]
    org.zipMailing = r[i]["ZIP (MAILING)"]
    org.associationsAssociations_1 = r[i]["ASSOCIATIONS.ASSOCIATIONS[1]"]
    org.associationsParentOfAssociation_1 = r[i]["ASSOCIATIONS.PARENT OF ASSOCIATION[1]"]
    org.associationsAssociations_2 = r[i]["ASSOCIATIONS.ASSOCIATIONS[2]"]
    org.associationsParentOfAssociation_2 = r[i]["ASSOCIATIONS.PARENT OF ASSOCIATION[2]"]
    org.contactArea_1 = r[i]["CONTACT.AREA[1]"]
    org.contactContact_1 = r[i]["CONTACT.CONTACT[1]"]
    org.contactPhone_1 = r[i]["CONTACT.PHONE #[1]"]
    org.contactArea_2 = r[i]["CONTACT.AREA[2]"]
    org.contactContact_2 = r[i]["CONTACT.CONTACT[2]"]
    org.contactPhone_2 = r[i]["CONTACT.PHONE #[2]"]
    org.contactArea_3 = r[i]["CONTACT.AREA[3]"]
    org.contactContact_3 = r[i]["CONTACT.CONTACT[3]"]
    org.contactPhone_3 = r[i]["CONTACT.PHONE #[3]"]
    org.effectiveDatetimeEffectiveDatetime_1 = r[i]["EFFECTIVE DATE/TIME.EFFECTIVE DATE/TIME[1]"]
    org.effectiveDatetimeNpi_1 = r[i]["EFFECTIVE DATE/TIME.NPI[1]"]
    org.effectiveDatetimeStatus_1 = r[i]["EFFECTIVE DATE/TIME.STATUS[1]"]
    org.taxonomyCodePrimaryCode_1 = r[i]["TAXONOMY CODE.PRIMARY CODE[1]"]
    org.taxonomyCodeStatus_1 = r[i]["TAXONOMY CODE.STATUS[1]"]
    org.taxonomyCodeTaxonomyCode_1 = r[i]["TAXONOMY CODE.TAXONOMY CODE[1]"]
    org.historyActivatedFacility_3000701 = r[i]["HISTORY.ACTIVATED FACILITY[3000701]"]
    org.historyDeactivatedFacilitySta_3000701 = r[i]["HISTORY.DEACTIVATED FACILITY / STA #[3000701]"]
    org.historyEffectiveDate_3000701 = r[i]["HISTORY.EFFECTIVE DATE[3000701]"]
    org.historyNameChangedFrom_3000701 = r[i]["HISTORY.NAME (CHANGED FROM)[3000701]"]
    org.historyOfficalVaNameChangedFrom_3000701 = r[i]["HISTORY.OFFICAL VA NAME (CHANGED FROM)[3000701]"]
    org.historyRealignedFrom_3000701 = r[i]["HISTORY.REALIGNED FROM[3000701]"]
    org.historyRealignedTo_3000701 = r[i]["HISTORY.REALIGNED TO[3000701]"]
    org.identifierCodingSystem_1 = r[i]["IDENTIFIER.CODING SYSTEM[1]"]
    org.identifierEffectiveDatetime_1 = r[i]["IDENTIFIER.EFFECTIVE DATE/TIME[1]"]
    org.identifierId_1 = r[i]["IDENTIFIER.ID[1]"]
    org.identifierStatus_1 = r[i]["IDENTIFIER.STATUS[1]"]
    org.identifierCodingSystem_2 = r[i]["IDENTIFIER.CODING SYSTEM[2]"]
    org.identifierEffectiveDatetime_2 = r[i]["IDENTIFIER.EFFECTIVE DATE/TIME[2]"]
    org.identifierId_2 = r[i]["IDENTIFIER.ID[2]"]
    org.identifierStatus_2 = r[i]["IDENTIFIER.STATUS[2]"]

    resource.organization.push(org)

  }

  return resource
}


function mapPractitioner(response: Response): Resource {
  let r = response.json()
  let resource: Resource = new Resource()

  if (r.hasOwnProperty('401')) {
    resource.returnCode = '401'
    return resource
  }

  for (let i = 0; i < r.length; i++) {

    let prac = new FhirPractitioner()
    prac.accessCode = r[i]["ACCESS CODE"]
    prac.adupn = r[i]["ADUPN"]
    prac.allowAraAccess = r[i]["ALLOW ARA ACCESS"]
    prac.allowVerifyingOfOthers = r[i]["ALLOW VERIFYING OF OTHERS"]
    prac.allowedToUseSpooler = r[i]["ALLOWED TO USE SPOOLER"]
    prac.alwaysShowSecondaries = r[i]["ALWAYS SHOW SECONDARIES"]
    prac.appointmentStatus = r[i]["APPOINTMENT STATUS"]
    prac.araValue = r[i]["ARA VALUE"]
    prac.askDeviceTypeAtSignOn = r[i]["ASK DEVICE TYPE AT SIGN-ON"]
    prac.askTerminalTypeAtLmEntry = r[i]["ASK TERMINAL TYPE AT LM ENTRY"]
    prac.authorizeReleaseOfNpi = r[i]["AUTHORIZE RELEASE OF NPI"]
    prac.authorizedToWriteMedOrders = r[i]["AUTHORIZED TO WRITE MED ORDERS"]
    prac.autoMenu = r[i]["AUTO MENU"]
    prac.autoSignOn = r[i]["AUTO SIGN-ON"]
    prac.badgeNumber = r[i]["BADGE NUMBER"]
    prac.brightCharsAtExitFromLm = r[i]["BRIGHT CHARS AT EXIT FROM LM"]
    prac.canMakeIntoAMailMessage = r[i]["CAN MAKE INTO A MAIL MESSAGE"]
    prac.city = r[i]["CITY"]
    prac.clinicalCoreTrainee = r[i]["CLINICAL CORE TRAINEE"]
    prac.commercialPhone = r[i]["COMMERCIAL PHONE"]
    prac.creator = r[i]["CREATOR"]
    prac.currentDegreeLevel = r[i]["CURRENT DEGREE LEVEL"]
    prac.dateAccessCodeLastChanged = r[i]["DATE ACCESS CODE LAST CHANGED"]
    prac.dateESigLastChanged = r[i]["DATE E-SIG LAST CHANGED"]
    prac.dateEntered = r[i]["DATE ENTERED"]
    prac.dateExclusionaryListChecked = r[i]["DATE EXCLUSIONARY LIST CHECKED"]
    prac.dateHl7TraineeRecordBuilt = r[i]["DATE HL7 TRAINEE RECORD BUILT"]
    prac.dateLastAccessedLmWp = r[i]["DATE LAST ACCESSED LM WP"]
    prac.dateNoLongerTrainee = r[i]["DATE NO LONGER TRAINEE"]
    prac.dateVerifyCodeLastChanged = r[i]["DATE VERIFY CODE LAST CHANGED"]
    prac.deaExpirationDate = r[i]["DEA EXPIRATION DATE"]
    prac.dea = r[i]["DEA#"]
    prac.defaultTerminalTypeForLm = r[i]["DEFAULT TERMINAL TYPE FOR LM"]
    prac.degree = r[i]["DEGREE"]
    prac.delegateOf = r[i]["DELEGATE OF"]
    prac.delegationDate = r[i]["DELEGATION DATE"]
    prac.delegationLevel = r[i]["DELEGATION LEVEL"]
    prac.deleteAllMailAccess = r[i]["DELETE ALL MAIL ACCESS"]
    prac.deleteKeysAtTermination = r[i]["DELETE KEYS AT TERMINATION"]
    prac.detoxMaintenanceIdNumber = r[i]["DETOX/MAINTENANCE ID NUMBER"]
    prac.digitalPager = r[i]["DIGITAL PAGER"]
    prac.displayHelpAtEntryToLm = r[i]["DISPLAY HELP AT ENTRY TO LM"]
    prac.displayLmCommands = r[i]["DISPLAY LM COMMANDS"]
    prac.disuser = r[i]["DISUSER"]
    prac.dob = r[i]["DOB"]
    prac.electronicSignatureCode = r[i]["ELECTRONIC SIGNATURE CODE"]
    prac.emailAddress = r[i]["EMAIL ADDRESS"]
    prac.endDateOfTempAddress = r[i]["END DATE OF TEMP ADDRESS"]
    prac.exclusionaryCheckPerformed = r[i]["EXCLUSIONARY CHECK PERFORMED"]
    prac.exclusionaryCheckedBy = r[i]["EXCLUSIONARY CHECKED BY"]
    prac.entryLastEditDate = r[i]["Entry Last Edit Date"]
    prac.enumerationCompleted = r[i]["Enumeration Completed"]
    prac.enumerationInitiated = r[i]["Enumeration Initiated"]
    prac.faxNumber = r[i]["FAX NUMBER"]
    prac.fileManagerAccessCode = r[i]["FILE MANAGER ACCESS CODE"]
    prac.fileRange = r[i]["FILE RANGE"]
    prac.fyAssigned = r[i]["FY assigned"]
    prac.fyear = r[i]["FYear"]
    prac.hinqEmployeeNumber = r[i]["HINQ EMPLOYEE NUMBER"]
    prac.immediateSupervisor = r[i]["IMMEDIATE SUPERVISOR"]
    prac.inactiveDate = r[i]["INACTIVE DATE"]
    prac.initial = r[i]["INITIAL"]
    prac.keyDelegationLevel = r[i]["KEY DELEGATION LEVEL"]
    prac.keystrokesFromLmWp = r[i]["KEYSTROKES FROM LM WP"]
    prac.language = r[i]["LANGUAGE"]
    prac.lastOptionAccessed = r[i]["LAST OPTION ACCESSED"]
    prac.lastOptionMainMenu = r[i]["LAST OPTION MAIN MENU"]
    prac.lastSignOnDatetime = r[i]["LAST SIGN-ON DATE/TIME"]
    prac.lastTrainingMonthYear = r[i]["LAST TRAINING MONTH & YEAR"]
    prac.lockoutUserUntil = r[i]["LOCKOUT USER UNTIL"]
    prac.mailCode = r[i]["MAIL CODE"]
    prac.multiDeviceDespooling = r[i]["MULTI-DEVICE DESPOOLING"]
    prac.multipleSignOn = r[i]["MULTIPLE SIGN-ON"]
    prac.multipleSignOnLimit = r[i]["MULTIPLE SIGN-ON LIMIT"]
    prac.name = r[i]["NAME"]
    prac.nameComponents = r[i]["NAME COMPONENTS"]
    prac.networkUsername = r[i]["NETWORK USERNAME"]
    prac.nickName = r[i]["NICK NAME"]
    prac.nonVaPrescriber = r[i]["NON-VA PRESCRIBER"]
    prac.npi = r[i]["NPI"]
    prac.npiEntryStatus = r[i]["NPI ENTRY STATUS"]
    prac.officePhone = r[i]["OFFICE PHONE"]
    prac.onExclusionaryList = r[i]["ON EXCLUSIONARY LIST"]
    prac.pac = r[i]["PAC"]
    prac.paidEmployee = r[i]["PAID EMPLOYEE"]
    prac.patientSelectionList = r[i]["PATIENT SELECTION LIST"]
    prac.personFilePointer = r[i]["PERSON FILE POINTER"]
    prac.pgyAssigned = r[i]["PGY assigned"]
    prac.pgyear = r[i]["PGYear"]
    prac.phone_3 = r[i]["PHONE #3"]
    prac.phone_4 = r[i]["PHONE #4"]
    prac.phoneHome = r[i]["PHONE (HOME)"]
    prac.positionTitle = r[i]["POSITION/TITLE"]
    prac.preferredEditor = r[i]["PREFERRED EDITOR"]
    prac.primaryMenuOption = r[i]["PRIMARY MENU OPTION"]
    prac.primaryWindow = r[i]["PRIMARY WINDOW"]
    prac.problemListPrimaryView = r[i]["PROBLEM LIST PRIMARY VIEW"]
    prac.problemSelectionList = r[i]["PROBLEM SELECTION LIST"]
    prac.programOfStudy = r[i]["PROGRAM OF STUDY"]
    prac.prohibitedTimesForSignOn = r[i]["PROHIBITED TIMES FOR SIGN-ON"]
    prac.providerClass = r[i]["PROVIDER CLASS"]
    prac.providerKey = r[i]["PROVIDER KEY"]
    prac.providerType = r[i]["PROVIDER TYPE"]
    prac.radNucMedInactiveDate = r[i]["RAD/NUC MED INACTIVE DATE"]
    prac.rank = r[i]["RANK"]
    prac.remarks = r[i]["REMARKS"]
    prac.renewDate = r[i]["RENEW DATE"]
    prac.requiresCosigner = r[i]["REQUIRES COSIGNER"]
    prac.reserved = r[i]["RESERVED"]
    prac.restrictPatientSelection = r[i]["RESTRICT PATIENT SELECTION"]
    prac.room = r[i]["ROOM"]
    prac.requiresTransmission = r[i]["Requires Transmission"]
    prac.scheduleIiNarcotic = r[i]["SCHEDULE II NARCOTIC"]
    prac.scheduleIiNonNarcotic = r[i]["SCHEDULE II NON-NARCOTIC"]
    prac.scheduleIiiNarcotic = r[i]["SCHEDULE III NARCOTIC"]
    prac.scheduleIiiNonNarcotic = r[i]["SCHEDULE III NON-NARCOTIC"]
    prac.scheduleIv = r[i]["SCHEDULE IV"]
    prac.scheduleV = r[i]["SCHEDULE V"]
    prac.secid = r[i]["SECID"]
    prac.serviceSection = r[i]["SERVICE/SECTION"]
    prac.sex = r[i]["SEX"]
    prac.signatureBlockPrintedName = r[i]["SIGNATURE BLOCK PRINTED NAME"]
    prac.signatureBlockTitle = r[i]["SIGNATURE BLOCK TITLE"]
    prac.socialWorker = r[i]["SOCIAL WORKER ?"]
    prac.socialWorkerNumber = r[i]["SOCIAL WORKER'S NUMBER"]
    prac.ssn = r[i]["SSN"]
    prac.staffReviewRequired = r[i]["STAFF REVIEW REQUIRED"]
    prac.startDateOfTempAddres = r[i]["START DATE OF TEMP ADDRES"]
    prac.startOfTraining = r[i]["START OF TRAINING"]
    prac.startupDefault = r[i]["STARTUP DEFAULT"]
    prac.state = r[i]["STATE"]
    prac.streetAddress_1 = r[i]["STREET ADDRESS 1"]
    prac.streetAddress_2 = r[i]["STREET ADDRESS 2"]
    prac.streetAddress_3 = r[i]["STREET ADDRESS 3"]
    prac.subjectAlternativeName = r[i]["SUBJECT ALTERNATIVE NAME"]
    prac.subjectOrganization = r[i]["SUBJECT ORGANIZATION"]
    prac.subjectOrganizationId = r[i]["SUBJECT ORGANIZATION ID"]
    prac.supplyEmployee = r[i]["SUPPLY EMPLOYEE"]
    prac.surrogate = r[i]["SURROGATE"]
    prac.taxId = r[i]["TAX ID"]
    prac.temporaryAddress_1 = r[i]["TEMPORARY ADDRESS 1"]
    prac.temporaryAddress_2 = r[i]["TEMPORARY ADDRESS 2"]
    prac.temporaryAddress_3 = r[i]["TEMPORARY ADDRESS 3"]
    prac.temporaryCity = r[i]["TEMPORARY CITY"]
    prac.temporaryState = r[i]["TEMPORARY STATE"]
    prac.temporaryZipCode = r[i]["TEMPORARY ZIP CODE"]
    prac.terminalTypeLastUsed = r[i]["TERMINAL TYPE LAST USED"]
    prac.terminationDate = r[i]["TERMINATION DATE"]
    prac.textTerminator = r[i]["TEXT TERMINATOR"]
    prac.timedReadOfSeconds = r[i]["TIMED READ (# OF SECONDS)"]
    prac.timestamp = r[i]["TIMESTAMP"]
    prac.title = r[i]["TITLE"]
    prac.totalMinutesUsingLmWp = r[i]["TOTAL MINUTES USING LM WP"]
    prac.typeAhead = r[i]["TYPE-AHEAD"]
    prac.terminationReason = r[i]["Termination Reason"]
    prac.uniqueUserId = r[i]["UNIQUE USER ID"]
    prac.usualCosigner = r[i]["USUAL COSIGNER"]
    prac.va = r[i]["VA#"]
    prac.verifyCode = r[i]["VERIFY CODE"]
    prac.verifyCodeNeverExpires = r[i]["VERIFY CODE never expires"]
    prac.vhaTrainingFacility = r[i]["VHA TRAINING FACILITY"]
    prac.voicePager = r[i]["VOICE PAGER"]
    prac.vpid = r[i]["VPID"]
    prac.wantToEditAccessCode = r[i]["Want to edit ACCESS CODE (Y/N)"]
    prac.wantToEditVerifyCode = r[i]["Want to edit VERIFY CODE (Y/N)"]
    prac.xusActiveUser = r[i]["XUS Active User"]
    prac.xusLogonAttemptCount = r[i]["XUS Logon Attempt Count"]
    prac.zipCode = r[i]["ZIP CODE"]
    prac.cprsTabCprsTab = r[i]["CPRS TAB.CPRS TAB[1]"]
    prac.cprsTabEffectiveDate = r[i]["CPRS TAB.EFFECTIVE DATE[1]"]
    prac.cprsTabExpirationDate = r[i]["CPRS TAB.EXPIRATION DATE[1]"]
    prac.divisionDefault = r[i]["DIVISION.DEFAULT[2957]"]
    prac.divisionDivision = r[i]["DIVISION.DIVISION[2957]"]
    prac.secondaryMenuOptionsSecondaryMenuOptions_1 = r[i]["SECONDARY MENU OPTIONS.SECONDARY MENU OPTIONS[1]"]
    prac.secondaryMenuOptionsSynonym_1 = r[i]["SECONDARY MENU OPTIONS.SYNONYM[1]"]
    prac.secondaryMenuOptionsSecondaryMenuOptions_2 = r[i]["SECONDARY MENU OPTIONS.SECONDARY MENU OPTIONS[2]"]
    prac.secondaryMenuOptionsSynonym_2 = r[i]["SECONDARY MENU OPTIONS.SYNONYM[2]"]
    prac.secondaryMenuOptionsSecondaryMenuOptions_3 = r[i]["SECONDARY MENU OPTIONS.SECONDARY MENU OPTIONS[3]"]
    prac.secondaryMenuOptionsSynonym_3 = r[i]["SECONDARY MENU OPTIONS.SYNONYM[3]"]
    prac.aliasAlias_1 = r[i]["ALIAS.ALIAS[1]"]
    prac.aliasAlias_2 = r[i]["ALIAS.ALIAS[2]"]
    prac.aliasAlias_3 = r[i]["ALIAS.ALIAS[3]"]
    prac.personClassEffectiveDate = r[i]["PERSON CLASS.Effective Date[1]"]
    prac.personClassExpirationDate = r[i]["PERSON CLASS.Expiration Date[1]"]
    prac.personClassPersonClass = r[i]["PERSON CLASS.Person Class[1]"]
    prac.keysDateGiven1 = r[i]["KEYS.DATE GIVEN[27]"]
    prac.keysGivenBy1 = r[i]["KEYS.GIVEN BY[27]"]
    prac.keysKey1 = r[i]["KEYS.KEY[27]"]
    prac.keysReviewDate1 = r[i]["KEYS.REVIEW DATE[27]"]
    prac.keysDateGiven2 = r[i]["KEYS.DATE GIVEN[28]"]
    prac.keysGivenBy2 = r[i]["KEYS.GIVEN BY[28]"]
    prac.keysKey2 = r[i]["KEYS.KEY[28]"]
    prac.keysReviewDate2 = r[i]["KEYS.REVIEW DATE[28]"]
    prac.keysDateGiven3 = r[i]["KEYS.DATE GIVEN[370]"]
    prac.keysGivenBy3 = r[i]["KEYS.GIVEN BY[370]"]
    prac.keysKey3 = r[i]["KEYS.KEY[370]"]
    prac.keysReviewDate3 = r[i]["KEYS.REVIEW DATE[370]"]
    prac.keysDateGiven4 = r[i]["KEYS.DATE GIVEN[371]"]
    prac.keysGivenBy4 = r[i]["KEYS.GIVEN BY[371]"]
    prac.keysKey4 = r[i]["KEYS.KEY[371]"]
    prac.keysReviewDate4 = r[i]["KEYS.REVIEW DATE[371]"]
    prac.keysDateGiven5 = r[i]["KEYS.DATE GIVEN[372]"]
    prac.keysGivenBy5 = r[i]["KEYS.GIVEN BY[372]"]
    prac.keysKey5 = r[i]["KEYS.KEY[372]"]
    prac.keysReviewDate5 = r[i]["KEYS.REVIEW DATE[372]"]
    prac.keysDateGiven6 = r[i]["KEYS.DATE GIVEN[58]"]
    prac.keysGivenBy6 = r[i]["KEYS.GIVEN BY[58]"]
    prac.keysKey6 = r[i]["KEYS.KEY[58]"]
    prac.keysReviewDate6 = r[i]["KEYS.REVIEW DATE[58]"]
    prac.keysDateGiven7 = r[i]["KEYS.DATE GIVEN[7]"]
    prac.keysGivenBy7 = r[i]["KEYS.GIVEN BY[7]"]
    prac.keysKey7 = r[i]["KEYS.KEY[7]"]
    prac.keysReviewDate7 = r[i]["KEYS.REVIEW DATE[7]"]
    prac.keysDateGiven8 = r[i]["KEYS.DATE GIVEN[88]"]
    prac.keysGivenBy8 = r[i]["KEYS.GIVEN BY[88]"]
    prac.keysKey8 = r[i]["KEYS.KEY[88]"]
    prac.keysReviewDate8 = r[i]["KEYS.REVIEW DATE[88]"]

    resource.practitioner.push(prac)
  }

  return resource
}


function mapPatient(response: Response): Resource {
  let r = response.json()
  let resource: Resource = new Resource()

  if (r.hasOwnProperty('401')) {
    resource.returnCode = '401'
    return resource
  }

  for (let i = 0; i < r.length; i++) {

    let pat = new FhirPatient()

    pat.amountOfMilitaryRetirement = r[i]["*AMOUNT OF MILITARY RETIREMENT"]
    pat.amountOfSocialSecurity = r[i]["*AMOUNT OF SOCIAL SECURITY"]
    pat.categoryOfBeneficiary = r[i]["*CATEGORY OF BENEFICIARY"]
    pat.claimFolderLocation = r[i]["*CLAIM FOLDER LOCATION"]
    pat.currentPcPractitioner = r[i]["*CURRENT PC PRACTITIONER"]
    pat.currentPcTeam = r[i]["*CURRENT PC TEAM"]
    pat.heightCm = r[i]["*HEIGHT(cm)"]
    pat.reactions = r[i]["*REACTIONS"]
    pat.weightKg = r[i]["*WEIGHT(kg)"]
    pat.oneU4n = r[i]["1U4N"]
    pat.secondMostRecentDateOfCare = r[i]["2ND MOST RECENT DATE OF CARE"]
    pat.secondMostRecentLocation = r[i]["2ND MOST RECENT LOCATION"]
    pat.absenceDivision = r[i]["ABSENCE DIVISION"]
    pat.addressChangeDttm = r[i]["ADDRESS CHANGE DT/TM"]
    pat.addressChangeSite = r[i]["ADDRESS CHANGE SITE"]
    pat.addressChangeSource = r[i]["ADDRESS CHANGE SOURCE"]
    pat.addressChangeUser = r[i]["ADDRESS CHANGE USER"]
    pat.age = r[i]["AGE"]
    pat.agencyAlliedCountry = r[i]["AGENCY/ALLIED COUNTRY"]
    pat.agentOrangeExamDate = r[i]["AGENT ORANGE EXAM DATE"]
    pat.agentOrangeExposIndicated = r[i]["AGENT ORANGE EXPOS. INDICATED?"]
    pat.agentOrangeExposureLocation = r[i]["AGENT ORANGE EXPOSURE LOCATION"]
    pat.agentOrangeRegistrationNo = r[i]["AGENT ORANGE REGISTRATION #"]
    pat.agentOrangeRegistrationDate = r[i]["AGENT ORANGE REGISTRATION DATE"]
    pat.agentOrangeReportedToCO = r[i]["AGENT ORANGE REPORTED TO C.O."]
    pat.amountOfAidAttendance = r[i]["AMOUNT OF AID & ATTENDANCE"]
    pat.amountOfGiInsurance = r[i]["AMOUNT OF GI INSURANCE"]
    pat.amountOfHousebound = r[i]["AMOUNT OF HOUSEBOUND"]
    pat.amountOfOtherIncome = r[i]["AMOUNT OF OTHER INCOME"]
    pat.amountOfOtherRetirement = r[i]["AMOUNT OF OTHER RETIREMENT"]
    pat.amountOfSsi = r[i]["AMOUNT OF SSI"]
    pat.amountOfVaDisability = r[i]["AMOUNT OF VA DISABILITY"]
    pat.amountOfVaPension = r[i]["AMOUNT OF VA PENSION"]
    pat.appointmentRequestComment = r[i]["APPOINTMENT REQUEST COMMENT"]
    pat.appointmentRequestDate = r[i]["APPOINTMENT REQUEST DATE"]
    pat.appointmentRequestOn_1010ez = r[i]["APPOINTMENT REQUEST ON 1010EZ"]
    pat.appointmentRequestStatus = r[i]["APPOINTMENT REQUEST STATUS"]
    pat.attendingPhysician = r[i]["ATTENDING PHYSICIAN"]
    pat.badAddressIndicator = r[i]["BAD ADDRESS INDICATOR"]
    pat.campLejeune = r[i]["CAMP LEJEUNE"]
    pat.campLejeuneChangeSite = r[i]["CAMP LEJEUNE CHANGE SITE"]
    pat.campLejeuneDataEntryLocked = r[i]["CAMP LEJEUNE DATA ENTRY LOCKED"]
    pat.campLejeuneDate = r[i]["CAMP LEJEUNE DATE"]
    pat.campLejeuneSource = r[i]["CAMP LEJEUNE SOURCE"]
    pat.cellularNumberChangeDtTm = r[i]["CELLULAR NUMBER CHANGE DT/TM"]
    pat.cellularNumberChangeSite = r[i]["CELLULAR NUMBER CHANGE SITE"]
    pat.cellularNumberChangeSource = r[i]["CELLULAR NUMBER CHANGE SOURCE"]
    pat.checkForDuplicate = r[i]["CHECK FOR DUPLICATE"]
    pat.city = r[i]["CITY"]
    pat.cityCivil = r[i]["CITY (CIVIL)"]
    pat.cityVa = r[i]["CITY (VA)"]
    pat.claimNumber = r[i]["CLAIM NUMBER"]
    pat.classIiDentalIndicator = r[i]["CLASS II DENTAL INDICATOR"]
    pat.cmorActivityScore = r[i]["CMOR ACTIVITY SCORE"]
    pat.cnhCurrent = r[i]["CNH CURRENT"]
    pat.collateralSponsorsName = r[i]["COLLATERAL SPONSOR'S NAME"]
    pat.combatFromDate = r[i]["COMBAT FROM DATE"]
    pat.combatIndicatedOn_1010ez = r[i]["COMBAT INDICATED ON 1010EZ"]
    pat.combatServiceIndicated = r[i]["COMBAT SERVICE INDICATED?"]
    pat.combatServiceLocation = r[i]["COMBAT SERVICE LOCATION"]
    pat.combatToDate = r[i]["COMBAT TO DATE"]
    pat.combatVeteranEndDate = r[i]["COMBAT VETERAN END DATE"]
    pat.condition = r[i]["CONDITION"]
    pat.confidentialAddrChangeDtTm = r[i]["CONFIDENTIAL ADDR CHANGE DT/TM"]
    pat.confidentialAddrChangeSite = r[i]["CONFIDENTIAL ADDR CHANGE SITE"]
    pat.confidentialAddrCountry = r[i]["CONFIDENTIAL ADDR COUNTRY"]
    pat.confidentialAddrPostalCode = r[i]["CONFIDENTIAL ADDR POSTAL CODE"]
    pat.confidentialAddrProvince = r[i]["CONFIDENTIAL ADDR PROVINCE"]
    pat.confidentialAddressActive = r[i]["CONFIDENTIAL ADDRESS ACTIVE?"]
    pat.confidentialAddressCity = r[i]["CONFIDENTIAL ADDRESS CITY"]
    pat.confidentialAddressCounty = r[i]["CONFIDENTIAL ADDRESS COUNTY"]
    pat.confidentialAddressState = r[i]["CONFIDENTIAL ADDRESS STATE"]
    pat.confidentialAddressZipCode = r[i]["CONFIDENTIAL ADDRESS ZIP CODE"]
    pat.confidentialEndDate = r[i]["CONFIDENTIAL END DATE"]
    pat.confidentialPhoneNumber = r[i]["CONFIDENTIAL PHONE NUMBER"]
    pat.confidentialStartDate = r[i]["CONFIDENTIAL START DATE"]
    pat.confidentialStreetLine_1 = r[i]["CONFIDENTIAL STREET [LINE 1]"]
    pat.confidentialStreetLine_2 = r[i]["CONFIDENTIAL STREET [LINE 2]"]
    pat.confidentialStreetLine_3 = r[i]["CONFIDENTIAL STREET [LINE 3]"]
    pat.coordinatingMasterOfRecord = r[i]["COORDINATING MASTER OF RECORD"]
    pat.country = r[i]["COUNTRY"]
    pat.county = r[i]["COUNTY"]
    pat.coveredByHealthInsurance = r[i]["COVERED BY HEALTH INSURANCE?"]
    pat.currentAdmission = r[i]["CURRENT ADMISSION"]
    pat.currentEnrollment = r[i]["CURRENT ENROLLMENT"]
    pat.currentMeansTestStatus = r[i]["CURRENT MEANS TEST STATUS"]
    pat.currentMohIndicator = r[i]["CURRENT MOH INDICATOR"]
    pat.currentMovement = r[i]["CURRENT MOVEMENT"]
    pat.currentPhIndicator = r[i]["CURRENT PH INDICATOR"]
    pat.currentPurpleHeartRemarks = r[i]["CURRENT PURPLE HEART REMARKS"]
    pat.currentPurpleHeartStatus = r[i]["CURRENT PURPLE HEART STATUS"]
    pat.currentRoom = r[i]["CURRENT ROOM"]
    pat.cvDateEdited = r[i]["CV DATE EDITED"]
    pat.dCity = r[i]["D-CITY"]
    pat.dDesigneeSameAsNok = r[i]["D-DESIGNEE SAME AS NOK?"]
    pat.dNameComponents = r[i]["D-NAME COMPONENTS"]
    pat.dNameOfDesignee = r[i]["D-NAME OF DESIGNEE"]
    pat.dPhoneNumber = r[i]["D-PHONE NUMBER"]
    pat.dRelationshipToPatient = r[i]["D-RELATIONSHIP TO PATIENT"]
    pat.dState = r[i]["D-STATE"]
    pat.dStreetAddressLine_1 = r[i]["D-STREET ADDRESS [LINE 1]"]
    pat.dStreetAddressLine_2 = r[i]["D-STREET ADDRESS [LINE 2]"]
    pat.dStreetAddressLine_3 = r[i]["D-STREET ADDRESS [LINE 3]"]
    pat.dWorkPhoneNumber = r[i]["D-WORK PHONE NUMBER"]
    pat.dZipCode = r[i]["D-ZIP CODE"]
    pat.dZipPlus4 = r[i]["D-ZIP+4"]
    pat.dateCommentLastEdited = r[i]["DATE COMMENT LAST EDITED"]
    pat.dateEnteredIntoFile = r[i]["DATE ENTERED INTO FILE"]
    pat.dateEnteredOnSiList = r[i]["DATE ENTERED ON SI LIST"]
    pat.dateFacilityInitiatedReview = r[i]["DATE FACILITY INITIATED REVIEW"]
    pat.dateMedicaidLastAsked = r[i]["DATE MEDICAID LAST ASKED"]
    pat.dateOfBirth = r[i]["DATE OF BIRTH"]
    pat.dateOfDeath = r[i]["DATE OF DEATH"]
    pat.dateOfDeathLastUpdated = r[i]["DATE OF DEATH LAST UPDATED"]
    pat.dateOfDecision = r[i]["DATE OF DECISION"]
    pat.dateOfRetirement = r[i]["DATE OF RETIREMENT"]
    pat.dateRuledIncompetentCivil = r[i]["DATE RULED INCOMPETENT (CIVIL)"]
    pat.dateRuledIncompetentVa = r[i]["DATE RULED INCOMPETENT (VA)"]
    pat.dateStatusLastEdited = r[i]["DATE STATUS LAST EDITED"]
    pat.dateVeteranRequestedCdEval = r[i]["DATE VETERAN REQUESTED CD EVAL"]
    pat.dateVeteranWasNotified = r[i]["DATE VETERAN WAS NOTIFIED"]
    pat.deathEnteredBy = r[i]["DEATH ENTERED BY"]
    pat.decidedBy = r[i]["DECIDED BY"]
    pat.dentalApplDueBeforeDate = r[i]["DENTAL APPL DUE BEFORE DATE"]
    pat.dentalClassification = r[i]["DENTAL CLASSIFICATION"]
    pat.dentalEligibilityExpiration = r[i]["DENTAL ELIGIBILITY EXPIRATION"]
    pat.designeeChangeDatetime = r[i]["DESIGNEE CHANGE DATE/TIME"]
    pat.disabilityDischargeOn_1010ez = r[i]["DISABILITY DISCHARGE ON 1010EZ"]
    pat.disabilityRetFromMilitary = r[i]["DISABILITY RET. FROM MILITARY?"]
    pat.dischargeDueToDisability = r[i]["DISCHARGE DUE TO DISABILITY"]
    pat.division = r[i]["DIVISION"]
    pat.duplicateStatus = r[i]["DUPLICATE STATUS"]
    pat.eCity = r[i]["E-CITY"]
    pat.eContactChangeDatetime = r[i]["E-CONTACT CHANGE DATE/TIME"]
    pat.eEmerContactSameAsNok = r[i]["E-EMER. CONTACT SAME AS NOK?"]
    pat.eName = r[i]["E-NAME"]
    pat.eNameComponents = r[i]["E-NAME COMPONENTS"]
    pat.ePhoneNumber = r[i]["E-PHONE NUMBER"]
    pat.eRelationshipToPatient = r[i]["E-RELATIONSHIP TO PATIENT"]
    pat.eState = r[i]["E-STATE"]
    pat.eStreetAddressLine_1 = r[i]["E-STREET ADDRESS [LINE 1]"]
    pat.eStreetAddressLine_2 = r[i]["E-STREET ADDRESS [LINE 2]"]
    pat.eStreetAddressLine_3 = r[i]["E-STREET ADDRESS [LINE 3]"]
    pat.eWorkPhoneNumber = r[i]["E-WORK PHONE NUMBER"]
    pat.eZipCode = r[i]["E-ZIP CODE"]
    pat.eZipPlus4 = r[i]["E-ZIP+4"]
    pat.e2City = r[i]["E2-CITY"]
    pat.e2ContactChangeDatetime = r[i]["E2-CONTACT CHANGE DATE/TIME"]
    pat.e2NameComponents = r[i]["E2-NAME COMPONENTS"]
    pat.e2NameOfSecondaryContact = r[i]["E2-NAME OF SECONDARY CONTACT"]
    pat.e2PhoneNumber = r[i]["E2-PHONE NUMBER"]
    pat.e2RelationshipToPatient = r[i]["E2-RELATIONSHIP TO PATIENT"]
    pat.e2State = r[i]["E2-STATE"]
    pat.e2StreetAddressLine_1 = r[i]["E2-STREET ADDRESS [LINE 1]"]
    pat.e2StreetAddressLine_2 = r[i]["E2-STREET ADDRESS [LINE 2]"]
    pat.e2StreetAddressLine_3 = r[i]["E2-STREET ADDRESS [LINE 3]"]
    pat.e2WorkPhoneNumber = r[i]["E2-WORK PHONE NUMBER"]
    pat.e2ZipCode = r[i]["E2-ZIP CODE"]
    pat.e2ZipPlus4 = r[i]["E2-ZIP+4"]
    pat.effDateCombinedScEval = r[i]["EFF. DATE COMBINED SC% EVAL."]
    pat.eligibilityInterimResponse = r[i]["ELIGIBILITY INTERIM RESPONSE"]
    pat.eligibilityStatus = r[i]["ELIGIBILITY STATUS"]
    pat.eligibilityStatusDate = r[i]["ELIGIBILITY STATUS DATE"]
    pat.eligibilityStatusEnteredBy = r[i]["ELIGIBILITY STATUS ENTERED BY"]
    pat.eligibilityVerifMethod = r[i]["ELIGIBILITY VERIF. METHOD"]
    pat.eligibilityVerifSource = r[i]["ELIGIBILITY VERIF. SOURCE"]
    pat.eligibleForMedicaid = r[i]["ELIGIBLE FOR MEDICAID?"]
    pat.emailAddress = r[i]["EMAIL ADDRESS"]
    pat.emailAddressChangeDtTm = r[i]["EMAIL ADDRESS CHANGE DT/TM"]
    pat.emailAddressChangeSite = r[i]["EMAIL ADDRESS CHANGE SITE"]
    pat.emailAddressChangeSource = r[i]["EMAIL ADDRESS CHANGE SOURCE"]
    pat.emailAddressIndicator = r[i]["EMAIL ADDRESS INDICATOR"]
    pat.emailAddressIndicatorDtTm = r[i]["EMAIL ADDRESS INDICATOR DT/TM"]
    pat.emergencyResponseIndicator = r[i]["EMERGENCY RESPONSE INDICATOR"]
    pat.employerCity = r[i]["EMPLOYER CITY"]
    pat.employerName = r[i]["EMPLOYER NAME"]
    pat.employerPhoneNumber = r[i]["EMPLOYER PHONE NUMBER"]
    pat.employerState = r[i]["EMPLOYER STATE"]
    pat.employerStreetLine_1 = r[i]["EMPLOYER STREET [LINE 1]"]
    pat.employerStreetLine_2 = r[i]["EMPLOYER STREET [LINE 2]"]
    pat.employerStreetLine_3 = r[i]["EMPLOYER STREET [LINE 3]"]
    pat.employerZipCode = r[i]["EMPLOYER ZIP CODE"]
    pat.employerZipPlus4 = r[i]["EMPLOYER ZIP+4"]
    pat.employmentStatus = r[i]["EMPLOYMENT STATUS"]
    pat.excludeFromFacilityDir = r[i]["EXCLUDE FROM FACILITY DIR"]
    pat.facilityMakingDetermination = r[i]["FACILITY MAKING DETERMINATION"]
    pat.fathersName = r[i]["FATHER'S NAME"]
    pat.fathersNameComponents = r[i]["FATHER'S NAME COMPONENTS"]
    pat.feeHospitalID = r[i]["FEE HOSPITAL I.D."]
    pat.fffDateEntered = r[i]["FFF DATE ENTERED"]
    pat.fffDateRemoved = r[i]["FFF DATE REMOVED"]
    pat.fffEnteredBy = r[i]["FFF ENTERED BY"]
    pat.fffRemovalRemarks = r[i]["FFF REMOVAL REMARKS"]
    pat.fffRemovedBy = r[i]["FFF REMOVED BY"]
    pat.filipinoVeteranProof = r[i]["FILIPINO VETERAN PROOF"]
    pat.foreignIdNumber = r[i]["FOREIGN ID NUMBER"]
    pat.fugitiveFelonFlag = r[i]["FUGITIVE FELON FLAG"]
    pat.fullIcn = r[i]["FULL ICN"]
    pat.giInsurancePolicy = r[i]["GI INSURANCE POLICY?"]
    pat.governmentAgency = r[i]["GOVERNMENT AGENCY"]
    pat.grenadaFromDate = r[i]["GRENADA FROM DATE"]
    pat.grenadaServiceIndicated = r[i]["GRENADA SERVICE INDICATED?"]
    pat.grenadaToDate = r[i]["GRENADA TO DATE"]
    pat.guardianCivil = r[i]["GUARDIAN (CIVIL)"]
    pat.guardianVa = r[i]["GUARDIAN (VA)"]
    pat.howWasPatientEntered = r[i]["HOW WAS PATIENT ENTERED?"]
    pat.icnChecksum = r[i]["ICN CHECKSUM"]
    pat.ineligibleDate = r[i]["INELIGIBLE DATE"]
    pat.ineligibleReason = r[i]["INELIGIBLE REASON"]
    pat.ineligibleTwxCity = r[i]["INELIGIBLE TWX CITY"]
    pat.ineligibleTwxSource = r[i]["INELIGIBLE TWX SOURCE"]
    pat.ineligibleTwxState = r[i]["INELIGIBLE TWX STATE"]
    pat.ineligibleVaroDecision = r[i]["INELIGIBLE VARO DECISION"]
    pat.initialOdsTreatmentReceived = r[i]["INITIAL ODS TREATMENT RECEIVED"]
    pat.inpatientWard = r[i]["INPATIENT WARD"]
    pat.institutionCivil = r[i]["INSTITUTION (CIVIL)"]
    pat.institutionVa = r[i]["INSTITUTION (VA)"]
    pat.integrationControlNumber = r[i]["INTEGRATION CONTROL NUMBER"]
    pat.kAddressSameAsPatients = r[i]["K-ADDRESS SAME AS PATIENT'S?"]
    pat.kCity = r[i]["K-CITY"]
    pat.kNameComponents = r[i]["K-NAME COMPONENTS"]
    pat.kNameOfPrimaryNok = r[i]["K-NAME OF PRIMARY NOK"]
    pat.kPhoneNumber = r[i]["K-PHONE NUMBER"]
    pat.kRelationshipToPatient = r[i]["K-RELATIONSHIP TO PATIENT"]
    pat.kState = r[i]["K-STATE"]
    pat.kStreetAddressLine_1 = r[i]["K-STREET ADDRESS [LINE 1]"]
    pat.kStreetAddressLine_2 = r[i]["K-STREET ADDRESS [LINE 2]"]
    pat.kStreetAddressLine_3 = r[i]["K-STREET ADDRESS [LINE 3]"]
    pat.kWorkPhoneNumber = r[i]["K-WORK PHONE NUMBER"]
    pat.kZipCode = r[i]["K-ZIP CODE"]
    pat.kZipPlus4 = r[i]["K-ZIP+4"]
    pat.k2AddressSameAsPatients = r[i]["K2-ADDRESS SAME AS PATIENT'S?"]
    pat.k2City = r[i]["K2-CITY"]
    pat.k2NameComponents = r[i]["K2-NAME COMPONENTS"]
    pat.k2NameOfSecondaryNok = r[i]["K2-NAME OF SECONDARY NOK"]
    pat.k2PhoneNumber = r[i]["K2-PHONE NUMBER"]
    pat.k2RelationshipToPatient = r[i]["K2-RELATIONSHIP TO PATIENT"]
    pat.k2State = r[i]["K2-STATE"]
    pat.k2StreetAddressLine_1 = r[i]["K2-STREET ADDRESS [LINE 1]"]
    pat.k2StreetAddressLine_2 = r[i]["K2-STREET ADDRESS [LINE 2]"]
    pat.k2StreetAddressLine_3 = r[i]["K2-STREET ADDRESS [LINE 3]"]
    pat.k2WorkPhoneNumber = r[i]["K2-WORK PHONE NUMBER"]
    pat.k2ZipCode = r[i]["K2-ZIP CODE"]
    pat.k2ZipPlus4 = r[i]["K2-ZIP+4"]
    pat.labReferralRef = r[i]["LAB REFERRAL REF"]
    pat.laboratoryReference = r[i]["LABORATORY REFERENCE"]
    pat.lastDmmsEpisodeNumber = r[i]["LAST DMMS EPISODE NUMBER"]
    pat.lastEditedBy = r[i]["LAST EDITED BY"]
    pat.lastMeansTest = r[i]["LAST MEANS TEST"]
    pat.lebanonFromDate = r[i]["LEBANON FROM DATE"]
    pat.lebanonServiceIndicated = r[i]["LEBANON SERVICE INDICATED?"]
    pat.lebanonToDate = r[i]["LEBANON TO DATE"]
    pat.locallyAssignedIcn = r[i]["LOCALLY ASSIGNED ICN"]
    pat.lodgerWardLocation = r[i]["LODGER WARD LOCATION"]
    pat.maritalStatus = r[i]["MARITAL STATUS"]
    pat.medicaidNumber = r[i]["MEDICAID NUMBER"]
    pat.methodOfDetermination = r[i]["METHOD OF DETERMINATION"]
    pat.militaryDisabilityRetirement = r[i]["MILITARY DISABILITY RETIREMENT"]
    pat.missingPersonDate = r[i]["MISSING PERSON DATE"]
    pat.missingPersonTwxCity = r[i]["MISSING PERSON TWX CITY"]
    pat.missingPersonTwxSource = r[i]["MISSING PERSON TWX SOURCE"]
    pat.missingPersonTwxState = r[i]["MISSING PERSON TWX STATE"]
    pat.monetaryBenVerifyDate = r[i]["MONETARY BEN. VERIFY DATE"]
    pat.mostRecent_1010ez = r[i]["MOST RECENT 1010EZ"]
    pat.mostRecentDateOfCare = r[i]["MOST RECENT DATE OF CARE"]
    pat.mostRecentLocationOfCare = r[i]["MOST RECENT LOCATION OF CARE"]
    pat.mothersMaidenName = r[i]["MOTHER'S MAIDEN NAME"]
    pat.mothersName = r[i]["MOTHER'S NAME"]
    pat.mothersNameComponents = r[i]["MOTHER'S NAME COMPONENTS"]
    pat.mothersMaidenNameComponents = r[i]["MOTHERS MAIDEN NAME COMPONENTS"]
    pat.multipleBirthIndicator = r[i]["MULTIPLE BIRTH INDICATOR"]
    pat.name = r[i]["NAME"]
    pat.nameComponents = r[i]["NAME COMPONENTS"]
    pat.networkIdentifier = r[i]["NETWORK IDENTIFIER"]
    pat.occupation = r[i]["OCCUPATION"]
    pat.pT = r[i]["P&T"]
    pat.pTEffectiveDate = r[i]["P&T EFFECTIVE DATE"]
    pat.pagerNumber = r[i]["PAGER NUMBER"]
    pat.pagerNumberChangeDtTm = r[i]["PAGER NUMBER CHANGE DT/TM"]
    pat.pagerNumberChangeSite = r[i]["PAGER NUMBER CHANGE SITE"]
    pat.pagerNumberChangeSource = r[i]["PAGER NUMBER CHANGE SOURCE"]
    pat.panamaFromDate = r[i]["PANAMA FROM DATE"]
    pat.panamaServiceIndicated = r[i]["PANAMA SERVICE INDICATED?"]
    pat.panamaToDate = r[i]["PANAMA TO DATE"]
    pat.patientMergedTo = r[i]["PATIENT MERGED TO"]
    pat.pensionAwardEffectiveDate = r[i]["PENSION AWARD EFFECTIVE DATE"]
    pat.pensionAwardLock = r[i]["PENSION AWARD LOCK"]
    pat.pensionAwardReason = r[i]["PENSION AWARD REASON"]
    pat.pensionIndicatorLock = r[i]["PENSION INDICATOR LOCK"]
    pat.pensionTerminatedDate = r[i]["PENSION TERMINATED DATE"]
    pat.pensionTerminatedReason_1 = r[i]["PENSION TERMINATED REASON 1"]
    pat.pensionTerminatedReason_2 = r[i]["PENSION TERMINATED REASON 2"]
    pat.pensionTerminatedReason_3 = r[i]["PENSION TERMINATED REASON 3"]
    pat.pensionTerminatedReason_4 = r[i]["PENSION TERMINATED REASON 4"]
    pat.periodOfService = r[i]["PERIOD OF SERVICE"]
    pat.persianGulfFromDate = r[i]["PERSIAN GULF FROM DATE"]
    pat.persianGulfService = r[i]["PERSIAN GULF SERVICE?"]
    pat.persianGulfToDate = r[i]["PERSIAN GULF TO DATE"]
    pat.phDivision = r[i]["PH DIVISION"]
    pat.phoneCivil = r[i]["PHONE (CIVIL)"]
    pat.phoneVa = r[i]["PHONE (VA)"]
    pat.phoneNumberCellular = r[i]["PHONE NUMBER [CELLULAR]"]
    pat.phoneNumberResidence = r[i]["PHONE NUMBER [RESIDENCE]"]
    pat.phoneNumberWork = r[i]["PHONE NUMBER [WORK]"]
    pat.placeOfBirthCity = r[i]["PLACE OF BIRTH [CITY]"]
    pat.placeOfBirthState = r[i]["PLACE OF BIRTH [STATE]"]
    pat.postalCode = r[i]["POSTAL CODE"]
    pat.powConfinementLocation = r[i]["POW CONFINEMENT LOCATION"]
    pat.powFromDate = r[i]["POW FROM DATE"]
    pat.powStatusIndicated = r[i]["POW STATUS INDICATED?"]
    pat.powStatusVerified = r[i]["POW STATUS VERIFIED"]
    pat.powToDate = r[i]["POW TO DATE"]
    pat.preferredFacility = r[i]["PREFERRED FACILITY"]
    pat.primaryEligibilityCode = r[i]["PRIMARY ELIGIBILITY CODE"]
    pat.primaryLongId = r[i]["PRIMARY LONG ID"]
    pat.primaryNokChangeDatetime = r[i]["PRIMARY NOK CHANGE DATE/TIME"]
    pat.primaryShortId = r[i]["PRIMARY SHORT ID"]
    pat.proj_112shad = r[i]["PROJ 112/SHAD"]
    pat.provider = r[i]["PROVIDER"]
    pat.province = r[i]["PROVINCE"]
    pat.pseudoSsnReason = r[i]["PSEUDO SSN REASON"]
    pat.race = r[i]["RACE"]
    pat.radiationExposureIndicated = r[i]["RADIATION EXPOSURE INDICATED?"]
    pat.radiationExposureMethod = r[i]["RADIATION EXPOSURE METHOD"]
    pat.radiationRegistrationDate = r[i]["RADIATION REGISTRATION DATE"]
    pat.rank = r[i]["RANK"]
    pat.ratedIncompetent = r[i]["RATED INCOMPETENT?"]
    pat.recalledToActiveDuty = r[i]["RECALLED TO ACTIVE DUTY"]
    pat.receivedVaCarePreviously = r[i]["RECEIVED VA CARE PREVIOUSLY?"]
    pat.receivingAVaPension = r[i]["RECEIVING A VA PENSION?"]
    pat.receivingAaBenefits = r[i]["RECEIVING A&A BENEFITS?"]
    pat.receivingHouseboundBenefits = r[i]["RECEIVING HOUSEBOUND BENEFITS?"]
    pat.receivingMilitaryRetirement = r[i]["RECEIVING MILITARY RETIREMENT?"]
    pat.receivingSocialSecurity = r[i]["RECEIVING SOCIAL SECURITY?"]
    pat.receivingSupSecuritySsi = r[i]["RECEIVING SUP. SECURITY (SSI)?"]
    pat.receivingVaDisability = r[i]["RECEIVING VA DISABILITY?"]
    pat.relationshipCivil = r[i]["RELATIONSHIP (CIVIL)"]
    pat.relationshipVa = r[i]["RELATIONSHIP (VA)"]
    pat.religiousPreference = r[i]["RELIGIOUS PREFERENCE"]
    pat.remarks = r[i]["REMARKS"]
    pat.residenceNumberChangeDttm = r[i]["RESIDENCE NUMBER CHANGE DT/TM"]
    pat.residenceNumberChangeSite = r[i]["RESIDENCE NUMBER CHANGE SITE"]
    pat.residenceNumberChangeSource = r[i]["RESIDENCE NUMBER CHANGE SOURCE"]
    pat.reviewDate = r[i]["REVIEW DATE"]
    pat.roomBed = r[i]["ROOM-BED"]
    pat.scAwardDate = r[i]["SC AWARD DATE"]
    pat.scoreCalculationDate = r[i]["SCORE CALCULATION DATE"]
    pat.secondaryNokChangeDatetime = r[i]["SECONDARY NOK CHANGE DATE/TIME"]
    pat.selfIdentifiedGender = r[i]["SELF IDENTIFIED GENDER"]
    pat.serviceBranchLast = r[i]["SERVICE BRANCH [LAST]"]
    pat.serviceBranchNntl = r[i]["SERVICE BRANCH [NNTL]"]
    pat.serviceBranchNtl = r[i]["SERVICE BRANCH [NTL]"]
    pat.serviceComponentLast = r[i]["SERVICE COMPONENT [LAST]"]
    pat.serviceComponentNntl = r[i]["SERVICE COMPONENT [NNTL]"]
    pat.serviceComponentNtl = r[i]["SERVICE COMPONENT [NTL]"]
    pat.serviceConnectedPercentage = r[i]["SERVICE CONNECTED PERCENTAGE"]
    pat.serviceConnected = r[i]["SERVICE CONNECTED?"]
    pat.serviceDentalInjury = r[i]["SERVICE DENTAL INJURY?"]
    pat.serviceDischargeTypeLast = r[i]["SERVICE DISCHARGE TYPE [LAST]"]
    pat.serviceDischargeTypeNntl = r[i]["SERVICE DISCHARGE TYPE [NNTL]"]
    pat.serviceDischargeTypeNtl = r[i]["SERVICE DISCHARGE TYPE [NTL]"]
    pat.serviceEntryDateLast = r[i]["SERVICE ENTRY DATE [LAST]"]
    pat.serviceEntryDateNntl = r[i]["SERVICE ENTRY DATE [NNTL]"]
    pat.serviceEntryDateNtl = r[i]["SERVICE ENTRY DATE [NTL]"]
    pat.serviceNumberLast = r[i]["SERVICE NUMBER [LAST]"]
    pat.serviceNumberNntl = r[i]["SERVICE NUMBER [NNTL]"]
    pat.serviceNumberNtl = r[i]["SERVICE NUMBER [NTL]"]
    pat.serviceSecondEpisode = r[i]["SERVICE SECOND EPISODE?"]
    pat.serviceSeparationDateLast = r[i]["SERVICE SEPARATION DATE [LAST]"]
    pat.serviceSeparationDateNntl = r[i]["SERVICE SEPARATION DATE [NNTL]"]
    pat.serviceSeparationDateNtl = r[i]["SERVICE SEPARATION DATE [NTL]"]
    pat.serviceTeethExtracted = r[i]["SERVICE TEETH EXTRACTED?"]
    pat.serviceThirdEpisode = r[i]["SERVICE THIRD EPISODE?"]
    pat.serviceVerificationDate = r[i]["SERVICE VERIFICATION DATE"]
    pat.sex = r[i]["SEX"]
    pat.socialSecurityNumber = r[i]["SOCIAL SECURITY NUMBER"]
    pat.somaliaFromDate = r[i]["SOMALIA FROM DATE"]
    pat.somaliaServiceIndicated = r[i]["SOMALIA SERVICE INDICATED?"]
    pat.somaliaToDate = r[i]["SOMALIA TO DATE"]
    pat.sourceDesignation = r[i]["SOURCE DESIGNATION"]
    pat.sourceOfNotification = r[i]["SOURCE OF NOTIFICATION"]
    pat.southwestAsiaConditions = r[i]["SOUTHWEST ASIA CONDITIONS?"]
    pat.spinalCordInjury = r[i]["SPINAL CORD INJURY"]
    pat.spousesEmpPhoneNumber = r[i]["SPOUSE'S EMP PHONE NUMBER"]
    pat.spousesEmpStreetLine_1 = r[i]["SPOUSE'S EMP STREET [LINE 1]"]
    pat.spousesEmpStreetLine_2 = r[i]["SPOUSE'S EMP STREET [LINE 2]"]
    pat.spousesEmpStreetLine_3 = r[i]["SPOUSE'S EMP STREET [LINE 3]"]
    pat.spousesEmpZipCode = r[i]["SPOUSE'S EMP ZIP CODE"]
    pat.spousesEmpZipPlus4 = r[i]["SPOUSE'S EMP ZIP+4"]
    pat.spousesEmployerName = r[i]["SPOUSE'S EMPLOYER NAME"]
    pat.spousesEmployersCity = r[i]["SPOUSE'S EMPLOYER'S CITY"]
    pat.spousesEmployersState = r[i]["SPOUSE'S EMPLOYER'S STATE"]
    pat.spousesEmploymentStatus = r[i]["SPOUSE'S EMPLOYMENT STATUS"]
    pat.spousesOccupation = r[i]["SPOUSE'S OCCUPATION"]
    pat.spousesRetirementDate = r[i]["SPOUSE'S RETIREMENT DATE"]
    pat.ssnVerificationStatus = r[i]["SSN VERIFICATION STATUS"]
    pat.state = r[i]["STATE"]
    pat.stateCivil = r[i]["STATE (CIVIL)"]
    pat.stateVa = r[i]["STATE (VA)"]
    pat.streetAddress_1Civil = r[i]["STREET ADDRESS 1 (CIVIL)"]
    pat.streetAddress_1Va = r[i]["STREET ADDRESS 1 (VA)"]
    pat.streetAddress_2Civil = r[i]["STREET ADDRESS 2 (CIVIL)"]
    pat.streetAddress_2Va = r[i]["STREET ADDRESS 2 (VA)"]
    pat.streetAddressLine_1 = r[i]["STREET ADDRESS [LINE 1]"]
    pat.streetAddressLine_2 = r[i]["STREET ADDRESS [LINE 2]"]
    pat.streetAddressLine_3 = r[i]["STREET ADDRESS [LINE 3]"]
    pat.subscriptionControlNumber = r[i]["SUBSCRIPTION CONTROL NUMBER"]
    pat.swAsiaCondExamDate = r[i]["SW ASIA COND EXAM DATE"]
    pat.swAsiaCondRegistrationDate = r[i]["SW ASIA COND REGISTRATION DATE"]
    pat.temporaryAddressActive = r[i]["TEMPORARY ADDRESS ACTIVE?"]
    pat.temporaryAddressChangeDttm = r[i]["TEMPORARY ADDRESS CHANGE DT/TM"]
    pat.temporaryAddressChangeSite = r[i]["TEMPORARY ADDRESS CHANGE SITE"]
    pat.temporaryAddressCountry = r[i]["TEMPORARY ADDRESS COUNTRY"]
    pat.temporaryAddressCounty = r[i]["TEMPORARY ADDRESS COUNTY"]
    pat.temporaryAddressEndDate = r[i]["TEMPORARY ADDRESS END DATE"]
    pat.temporaryAddressPostalCode = r[i]["TEMPORARY ADDRESS POSTAL CODE"]
    pat.temporaryAddressProvince = r[i]["TEMPORARY ADDRESS PROVINCE"]
    pat.temporaryAddressStartDate = r[i]["TEMPORARY ADDRESS START DATE"]
    pat.temporaryCity = r[i]["TEMPORARY CITY"]
    pat.temporaryIdNumber = r[i]["TEMPORARY ID NUMBER"]
    pat.temporaryPhoneNumber = r[i]["TEMPORARY PHONE NUMBER"]
    pat.temporaryState = r[i]["TEMPORARY STATE"]
    pat.temporaryStreetLine_1 = r[i]["TEMPORARY STREET [LINE 1]"]
    pat.temporaryStreetLine_2 = r[i]["TEMPORARY STREET [LINE 2]"]
    pat.temporaryStreetLine_3 = r[i]["TEMPORARY STREET [LINE 3]"]
    pat.temporaryZipCode = r[i]["TEMPORARY ZIP CODE"]
    pat.temporaryZipPlus4 = r[i]["TEMPORARY ZIP+4"]
    pat.terminalDigitOfSsn = r[i]["TERMINAL DIGIT OF SSN"]
    pat.testPatientIndicator = r[i]["TEST PATIENT INDICATOR"]
    pat.totalAnnualVaCheckAmount = r[i]["TOTAL ANNUAL VA CHECK AMOUNT"]
    pat.treatingSpecialty = r[i]["TREATING SPECIALTY"]
    pat.type = r[i]["TYPE"]
    pat.typeOfOtherRetirement = r[i]["TYPE OF OTHER RETIREMENT"]
    pat.unemployable = r[i]["UNEMPLOYABLE"]
    pat.userEnrolleeSite = r[i]["USER ENROLLEE SITE"]
    pat.userEnrolleeValidThrough = r[i]["USER ENROLLEE VALID THROUGH"]
    pat.veteranYn = r[i]["VETERAN (Y/N)?"]
    pat.veteranCatastrophicallyDisabled = r[i]["VETERAN CATASTROPHICALLY DISABLED?"]
    pat.vietnamFromDate = r[i]["VIETNAM FROM DATE"]
    pat.vietnamServiceIndicated = r[i]["VIETNAM SERVICE INDICATED?"]
    pat.vietnamToDate = r[i]["VIETNAM TO DATE"]
    pat.vtsPatientFlag = r[i]["VTS PATIENT FLAG"]
    pat.wardLocation = r[i]["WARD LOCATION"]
    pat.whoEnteredPatient = r[i]["WHO ENTERED PATIENT"]
    pat.yugoslaviaFromDate = r[i]["YUGOSLAVIA FROM DATE"]
    pat.yugoslaviaServiceIndicated = r[i]["YUGOSLAVIA SERVICE INDICATED?"]
    pat.yugoslaviaToDate = r[i]["YUGOSLAVIA TO DATE"]
    pat.zipCivil = r[i]["ZIP (CIVIL)"]
    pat.zipVa = r[i]["ZIP (VA)"]
    pat.zipCode = r[i]["ZIP CODE"]
    pat.zipPlus4 = r[i]["ZIP+4"]
    pat.zipPlus4Civil = r[i]["ZIP+4 (CIVIL)"]
    pat.zipPlus4Va = r[i]["ZIP+4 (VA)"]
    pat.aliasAlias = r[i]["ALIAS.ALIAS[1]"]
    pat.aliasAliasComponents = r[i]["ALIAS.ALIAS COMPONENTS[1]"]
    pat.aliasAliasSsn = r[i]["ALIAS.ALIAS SSN[1]"]
    pat.aliasAlias2 = r[i]["ALIAS.ALIAS[2]"]
    pat.aliasAliasComponents2 = r[i]["ALIAS.ALIAS COMPONENTS[2]"]
    pat.aliasAliasSsn2 = r[i]["ALIAS.ALIAS SSN[2]"]
    pat.insuranceTypeCommentPatientPolicy = r[i]["INSURANCE TYPE.*COMMENT - PATIENT POLICY[1]"]
    pat.insuranceTypeGroupName = r[i]["INSURANCE TYPE.*GROUP NAME[1]"]
    pat.insuranceTypeGroupNumber = r[i]["INSURANCE TYPE.*GROUP NUMBER[1]"]
    pat.insuranceTypeNameOfInsured = r[i]["INSURANCE TYPE.*NAME OF INSURED[1]"]
    pat.insuranceTypeSubscriberId = r[i]["INSURANCE TYPE.*SUBSCRIBER ID[1]"]
    pat.insuranceTypeCoordinationOfBenefits = r[i]["INSURANCE TYPE.COORDINATION OF BENEFITS[1]"]
    pat.insuranceTypeDateEntered = r[i]["INSURANCE TYPE.DATE ENTERED[1]"]
    pat.insuranceTypeDateLastEdited = r[i]["INSURANCE TYPE.DATE LAST EDITED[1]"]
    pat.insuranceTypeDateLastVerified = r[i]["INSURANCE TYPE.DATE LAST VERIFIED[1]"]
    pat.insuranceTypeDateOfSourceOfInformation = r[i]["INSURANCE TYPE.DATE OF SOURCE OF INFORMATION[1]"]
    pat.insuranceTypeDatetimePeriod = r[i]["INSURANCE TYPE.DATE TIME PERIOD[1]"]
    pat.insuranceTypeDatetimePeriodFormatQual = r[i]["INSURANCE TYPE.DATE TIME PERIOD FORMAT QUAL[1]"]
    pat.insuranceTypeEbDisplayEntry = r[i]["INSURANCE TYPE.EB DISPLAY ENTRY[1]"]
    pat.insuranceTypeEffectiveDateOfPolicy = r[i]["INSURANCE TYPE.EFFECTIVE DATE OF POLICY[1]"]
    pat.insuranceTypeEivAutoUpdate = r[i]["INSURANCE TYPE.EIV AUTO-UPDATE[1]"]
    pat.insuranceTypeEmployClaimStAddressLine_2 = r[i]["INSURANCE TYPE.EMPLOY CLAIM ST ADDRESS LINE 2[1]"]
    pat.insuranceTypeEmployClaimStAddressLine_3 = r[i]["INSURANCE TYPE.EMPLOY CLAIM ST ADDRESS LINE 3[1]"]
    pat.insuranceTypeEmployerClaimsCity = r[i]["INSURANCE TYPE.EMPLOYER CLAIMS CITY[1]"]
    pat.insuranceTypeEmployerClaimsPhone = r[i]["INSURANCE TYPE.EMPLOYER CLAIMS PHONE[1]"]
    pat.insuranceTypeEmployerClaimsState = r[i]["INSURANCE TYPE.EMPLOYER CLAIMS STATE[1]"]
    pat.insuranceTypeEmployerClaimsStreetAddress = r[i]["INSURANCE TYPE.EMPLOYER CLAIMS STREET ADDRESS[1]"]
    pat.insuranceTypeEmployerClaimsZipCode = r[i]["INSURANCE TYPE.EMPLOYER CLAIMS ZIP CODE[1]"]
    pat.insuranceTypeEmploymentStatus = r[i]["INSURANCE TYPE.EMPLOYMENT STATUS[1]"]
    pat.insuranceTypeEnteredBy = r[i]["INSURANCE TYPE.ENTERED BY[1]"]
    pat.insuranceTypeEsghp = r[i]["INSURANCE TYPE.ESGHP[1]"]
    pat.insuranceTypeGroupPlan = r[i]["INSURANCE TYPE.GROUP PLAN[1]"]
    pat.insuranceTypeInsuranceExpirationDate = r[i]["INSURANCE TYPE.INSURANCE EXPIRATION DATE[1]"]
    pat.insuranceTypeInsuranceType = r[i]["INSURANCE TYPE.INSURANCE TYPE[1]"]
    pat.insuranceTypeInsuredsBranch = r[i]["INSURANCE TYPE.INSURED'S BRANCH[1]"]
    pat.insuranceTypeInsuredsCity = r[i]["INSURANCE TYPE.INSURED'S CITY[1]"]
    pat.insuranceTypeInsuredsCountry = r[i]["INSURANCE TYPE.INSURED'S COUNTRY[1]"]
    pat.insuranceTypeInsuredsCountrySubdivision = r[i]["INSURANCE TYPE.INSURED'S COUNTRY SUBDIVISION[1]"]
    pat.insuranceTypeInsuredsDob = r[i]["INSURANCE TYPE.INSURED'S DOB[1]"]
    pat.insuranceTypeInsuredsPhone = r[i]["INSURANCE TYPE.INSURED'S PHONE[1]"]
    pat.insuranceTypeInsuredsRank = r[i]["INSURANCE TYPE.INSURED'S RANK[1]"]
    pat.insuranceTypeInsuredsSex = r[i]["INSURANCE TYPE.INSURED'S SEX[1]"]
    pat.insuranceTypeInsuredsSsn = r[i]["INSURANCE TYPE.INSURED'S SSN[1]"]
    pat.insuranceTypeInsuredsState = r[i]["INSURANCE TYPE.INSURED'S STATE[1]"]
    pat.insuranceTypeInsuredsStreet_1 = r[i]["INSURANCE TYPE.INSURED'S STREET 1[1]"]
    pat.insuranceTypeInsuredsStreet_2 = r[i]["INSURANCE TYPE.INSURED'S STREET 2[1]"]
    pat.insuranceTypeInsuredsZip = r[i]["INSURANCE TYPE.INSURED'S ZIP[1]"]
    pat.insuranceTypeLastEditedBy = r[i]["INSURANCE TYPE.LAST EDITED BY[1]"]
    pat.insuranceTypeMilitaryEmploymentStatus = r[i]["INSURANCE TYPE.MILITARY EMPLOYMENT STATUS[1]"]
    pat.insuranceTypeMilitaryGovtAffiliationCode = r[i]["INSURANCE TYPE.MILITARY GOVT AFFILIATION CODE[1]"]
    pat.insuranceTypeMilitaryInfoStatusCode = r[i]["INSURANCE TYPE.MILITARY INFO STATUS CODE[1]"]
    pat.insuranceTypeMilitaryPersonnelDescription = r[i]["INSURANCE TYPE.MILITARY PERSONNEL DESCRIPTION[1]"]
    pat.insuranceTypeMilitaryServiceRankCode = r[i]["INSURANCE TYPE.MILITARY SERVICE RANK CODE[1]"]
    pat.insuranceTypeNewGroupName = r[i]["INSURANCE TYPE.NEW GROUP NAME[1]"]
    pat.insuranceTypeNewGroupNumber = r[i]["INSURANCE TYPE.NEW GROUP NUMBER[1]"]
    pat.insuranceTypePatientId = r[i]["INSURANCE TYPE.PATIENT ID[1]"]
    pat.insuranceTypePatientsSecQualifier_1 = r[i]["INSURANCE TYPE.PATIENT'S SEC QUALIFIER(1)[1]"]
    pat.insuranceTypePatientsSecQualifier_2 = r[i]["INSURANCE TYPE.PATIENT'S SEC QUALIFIER(2)[1]"]
    pat.insuranceTypePatientsSecQualifier_3 = r[i]["INSURANCE TYPE.PATIENT'S SEC QUALIFIER(3)[1]"]
    pat.insuranceTypePatientsSecondaryId_1 = r[i]["INSURANCE TYPE.PATIENT'S SECONDARY ID(1)[1]"]
    pat.insuranceTypePatientsSecondaryId_2 = r[i]["INSURANCE TYPE.PATIENT'S SECONDARY ID(2)[1]"]
    pat.insuranceTypePatientsSecondaryId_3 = r[i]["INSURANCE TYPE.PATIENT'S SECONDARY ID(3)[1]"]
    pat.insuranceTypePharmacyPersonCode = r[i]["INSURANCE TYPE.PHARMACY PERSON CODE[1]"]
    pat.insuranceTypePharmacyRelationshipCode = r[i]["INSURANCE TYPE.PHARMACY RELATIONSHIP CODE[1]"]
    pat.insuranceTypePrimaryCareProvider = r[i]["INSURANCE TYPE.PRIMARY CARE PROVIDER[1]"]
    pat.insuranceTypePrimaryProviderPhone = r[i]["INSURANCE TYPE.PRIMARY PROVIDER PHONE[1]"]
    pat.insuranceTypePtRelationshipHipaa = r[i]["INSURANCE TYPE.PT. RELATIONSHIP - HIPAA[1]"]
    pat.insuranceTypePtRelationshipToInsured = r[i]["INSURANCE TYPE.PT. RELATIONSHIP TO INSURED[1]"]
    pat.insuranceTypeRequestedServiceDate = r[i]["INSURANCE TYPE.REQUESTED SERVICE DATE[1]"]
    pat.insuranceTypeRequestedServiceType = r[i]["INSURANCE TYPE.REQUESTED SERVICE TYPE[1]"]
    pat.insuranceTypeRetirementDate = r[i]["INSURANCE TYPE.RETIREMENT DATE[1]"]
    pat.insuranceTypeSendBillToEmployer = r[i]["INSURANCE TYPE.SEND BILL TO EMPLOYER[1]"]
    pat.insuranceTypeSourceOfInformation = r[i]["INSURANCE TYPE.SOURCE OF INFORMATION[1]"]
    pat.insuranceTypeStopPolicyFromBilling = r[i]["INSURANCE TYPE.STOP POLICY FROM BILLING[1]"]
    pat.insuranceTypeSubscribersEmployerName = r[i]["INSURANCE TYPE.SUBSCRIBER'S EMPLOYER NAME[1]"]
    pat.insuranceTypeSubscribersSecId_1 = r[i]["INSURANCE TYPE.SUBSCRIBER'S SEC ID(1)[1]"]
    pat.insuranceTypeSubscribersSecId_2 = r[i]["INSURANCE TYPE.SUBSCRIBER'S SEC ID(2)[1]"]
    pat.insuranceTypeSubscribersSecId_3 = r[i]["INSURANCE TYPE.SUBSCRIBER'S SEC ID(3)[1]"]
    pat.insuranceTypeSubscribersSecQualifier_1 = r[i]["INSURANCE TYPE.SUBSCRIBER'S SEC QUALIFIER(1)[1]"]
    pat.insuranceTypeSubscribersSecQualifier_2 = r[i]["INSURANCE TYPE.SUBSCRIBER'S SEC QUALIFIER(2)[1]"]
    pat.insuranceTypeSubscribersSecQualifier_3 = r[i]["INSURANCE TYPE.SUBSCRIBER'S SEC QUALIFIER(3)[1]"]
    pat.insuranceTypeVerifiedBy = r[i]["INSURANCE TYPE.VERIFIED BY[1]"]
    pat.insuranceTypeWhoseInsurance = r[i]["INSURANCE TYPE.WHOSE INSURANCE[1]"]

    resource.patient.push(pat)

  }



  return resource
}


function mapCondition(response: Response): Resource {
  let r = response.json()
  let resource: Resource = new Resource()

  if (r.hasOwnProperty('401')) {
    resource.returnCode = '401'
    return resource
  }

  for (let i = 0; i < r.length; i++) {

    let cond = new FhirCondition()

    cond.ien = r[i]["IEN##"]
    cond.agentOrangeExposure = r[i]["AGENT ORANGE EXPOSURE"]
    cond.class = r[i]["CLASS"]
    cond.clinic = r[i]["CLINIC"]
    cond.codingSystem = r[i]["CODING SYSTEM"]
    cond.combatVeteran = r[i]["COMBAT VETERAN"]
    cond.condition = r[i]["CONDITION"]
    cond.dateEntered = r[i]["DATE ENTERED"]
    cond.dateLastModified = r[i]["DATE LAST MODIFIED"]
    cond.dateOfInterest = r[i]["DATE OF INTEREST"]
    cond.dateOfOnset = r[i]["DATE OF ONSET"]
    cond.dateRecorded = r[i]["DATE RECORDED"]
    cond.dateResolved = r[i]["DATE RESOLVED"]
    cond.diagnosis = r[i]["DIAGNOSIS"]
    cond.enteredBy = r[i]["ENTERED BY"]
    cond.facility = r[i]["FACILITY"]
    cond.headAndOrNeckCancer = r[i]["HEAD AND/OR NECK CANCER"]
    cond.ionizingRadiationExposure = r[i]["IONIZING RADIATION EXPOSURE"]
    cond.militarySexualTrauma = r[i]["MILITARY SEXUAL TRAUMA"]
    cond.nmbr = r[i]["NMBR"]
    cond.patientName = r[i]["PATIENT NAME"]
    cond.persianGulfExposure = r[i]["PERSIAN GULF EXPOSURE"]
    cond.priority = r[i]["PRIORITY"]
    cond.problem = r[i]["PROBLEM"]
    cond.providerNarrative = r[i]["PROVIDER NARRATIVE"]
    cond.recordingProvider = r[i]["RECORDING PROVIDER"]
    cond.responsibleProvider = r[i]["RESPONSIBLE PROVIDER"]
    cond.service = r[i]["SERVICE"]
    cond.serviceConnected = r[i]["SERVICE CONNECTED"]
    cond.shipboardHazardDefense = r[i]["SHIPBOARD HAZARD & DEFENSE"]
    cond.snomedCtConceptCode = r[i]["SNOMED CT CONCEPT CODE"]
    cond.snomedCtDesignationCode = r[i]["SNOMED CT DESIGNATION CODE"]
    cond.snomedCtToIcdMapStatus = r[i]["SNOMED CT-TO-ICD MAP STATUS"]
    cond.status = r[i]["STATUS"]
    cond.uniqueNewTermRequested = r[i]["UNIQUE NEW TERM REQUESTED"]
    cond.uniqueTermRequestComment = r[i]["UNIQUE TERM REQUEST COMMENT"]
    cond.vhatConceptVuidFuture = r[i]["VHAT CONCEPT VUID (FUTURE)"]
    cond.vhatDesignationVuidFuture = r[i]["VHAT DESIGNATION VUID (FUTURE)"]
    cond.noteFacilityNoteFacility1 = r[i]["NOTE FACILITY.NOTE FACILITY[1]"]
    cond.noteAuthor1 = r[i]["NOTE.AUTHOR[1]"]
    cond.noteDateNoteAdded1 = r[i]["NOTE.DATE NOTE ADDED[1]"]
    cond.noteNoteNarrative1 = r[i]["NOTE.NOTE NARRATIVE[1]"]
    cond.noteNnoteNmbr1 = r[i]["NOTE.NOTE NMBR[1]"]
    cond.noteStatus1 = r[i]["NOTE.STATUS[1]"]
    cond.noteAuthor2 = r[i]["NOTE.AUTHOR[2]"]
    cond.noteDateNoteAdded2 = r[i]["NOTE.DATE NOTE ADDED[2]"]
    cond.noteNoteNarrative2 = r[i]["NOTE.NOTE NARRATIVE[2]"]
    cond.noteNoteNmbr2 = r[i]["NOTE.NOTE NMBR[2]"]
    cond.noteStatus2 = r[i]["NOTE.STATUS[2]"]

    resource.condition.push(cond)
  }

  return resource
}


function mapFind(response: Response): Resource {
  let r = response.json()
  // console.log(" RESPOSNE ====== "+JSON.stringify(r))
  let resource: Resource = new Resource()

  if (r.hasOwnProperty('401')) {
    resource.returnCode = '401'
    return resource
  }

  for (let i = 0; i < r.length; i++) {
    let fhirFind: FhirFind = new FhirFind()
    fhirFind.ien = r[i]["IEN"]
    for (let j = 0; j < r[i]["Identifiers"].length; j++) {
      fhirFind.identifier.push(r[i]["Identifiers"][j])
    }
    resource.find.push(fhirFind)
  }

  return resource
}
