import { TestBed, inject } from '@angular/core/testing'
import { MockBackend, MockConnection } from '@angular/http/testing';
import { Http, BaseRequestOptions, Response, ResponseOptions, RequestMethod } from '@angular/http';
import { ErrorService } from '../../../core/error-messages/error.service'

import { FhirService, ResourceInt } from './fhir.service'
import { Resource } from './resource'
import { FhirOrganization } from './fhir-organization'
import { FhirPatient } from './fhir-patient'
import { FhirPractitioner } from './fhir-practitioner'
import { FhirCondition } from './fhir-condition'

describe('FhirService', () => {
  let service: FhirService;
  let backend: MockBackend;

  beforeEach(() => {
    TestBed.configureTestingModule({
      providers: [FhirService,
                  MockBackend,
                  ErrorService,
                  BaseRequestOptions,
                  {
                    provide: Http,
                    useFactory: function(mockBackend, defaultOptions) {
                      return new Http(mockBackend, defaultOptions);
                    },
                    deps: [MockBackend, BaseRequestOptions]
                  }
                ]
    })
  })

  beforeEach(inject([FhirService, MockBackend], function(fhirService, mockBackend) {
    service = fhirService;
    backend = mockBackend;
  }));

  afterEach(function() {
    TestBed.resetTestingModule();
  });

  it('organization should get data', function() {

    backend.connections.subscribe(function(connection: MockConnection) {

      let options = new ResponseOptions(
        { body: [{
                  organization: {
                    'name':'my name 123'
                  }
                }]
        }
      )
      connection.mockRespond(new Response(options));
    })


    let val:ResourceInt = {id:"1214",type:"ORGANIZATION",util:"RECORD"}

    service.getResource(val).then(res => {
      expect(res['organization'].length).toEqual(1)
      expect(res['patient'].length).toEqual(0)
      expect(res['practitioner'].length).toEqual(0)
      expect(res['condition'].length).toEqual(0)
    })
  })

  it('patient should get data', function() {

    backend.connections.subscribe(function(connection: MockConnection) {

      let options = new ResponseOptions(
        { body: [{
                  patient: {
                    'age':'33'
                  }
                }]
        }
      )
      connection.mockRespond(new Response(options));
    })

    let val:ResourceInt = {id:"1214",type:"PATIENT",util:"RECORD"}

    service.getResource(val).then(res => {
      expect(res['organization'].length).toEqual(0)
      expect(res['patient'].length).toEqual(1)
      expect(res['practitioner'].length).toEqual(0)
      expect(res['condition'].length).toEqual(0)
    })
  })

  it('condition should get data', function() {

    backend.connections.subscribe(function(connection: MockConnection) {

      let options = new ResponseOptions(
        { body: [{
                  condition: {
                    'class':'myclass'
                  }
                }]
        }
      )
      connection.mockRespond(new Response(options));
    })

    let val:ResourceInt = {id:"1214",type:"CONDITION",util:"RECORD"}

    service.getResource(val).then(res => {
      expect(res['organization'].length).toEqual(0)
      expect(res['patient'].length).toEqual(0)
      expect(res['practitioner'].length).toEqual(0)
      expect(res['condition'].length).toEqual(1)
    })
  })

  it('practitioner should get data', function() {

    backend.connections.subscribe(function(connection: MockConnection) {

      let options = new ResponseOptions(
        { body: [{
                  practitioner: {
                    'class':'myclass'
                  }
                }]
        }
      )
      connection.mockRespond(new Response(options));
    })

    let val:ResourceInt = {id:"1214",type:"PRACTITIONER",util:"RECORD"}

    service.getResource(val).then(res => {
      expect(res['organization'].length).toEqual(0)
      expect(res['patient'].length).toEqual(0)
      expect(res['practitioner'].length).toEqual(1)
      expect(res['condition'].length).toEqual(0)
    })
  })

  it('record should get data', function() {

    backend.connections.subscribe(function(connection: MockConnection) {
        let options = new ResponseOptions(
          { body: [{
            records: [{
                      'class':'myclass'
                     }]
                   }]
           }
         )
         connection.mockRespond(new Response(options));
       })

      service.getRecord("4.3","1").then(res => {
         expect(res[0]['records'].length).toEqual(1)
       })

  })

  it('should catch error', function() {

    backend.connections.subscribe(function(connection: MockConnection) {
      let options = new ResponseOptions(
        { body: '', status: 401}
      )
      connection.mockRespond(new Response(options));
    })

    let val:ResourceInt = {id:"wrong",type:"PATIENT",util:"RECORD"}

    service.getResource(val).then(res => {
      // should not be reached
      expect(val).toEqual(null)
    }).catch(err => {
      expect(err).toBeTruthy()
    })
  })

})
