import { Injectable, ReflectiveInjector } from '@angular/core'
import {Http, XHRBackend, RequestOptions, Request, RequestOptionsArgs, Response, Headers} from '@angular/http'
import {Observable} from 'rxjs/Observable'
import { MonitoringService } from '../application-insights/monitoring.service'
/**
 * TAS app wrapper for Http adding application insights monitoring
 */
@Injectable()
export class TasHttp extends Http {
  private monitor: MonitoringService
  starttime: number

  constructor (backend: XHRBackend, options: RequestOptions) {
    super(backend, options)
    // Manually retrieve the monitoring service from the injector
    // so that constructor has no dependencies that must be passed in from child
    const injector = ReflectiveInjector.resolveAndCreate([MonitoringService])
    this.monitor = injector.get(MonitoringService)
  }

  private requestOptions(options?: RequestOptionsArgs): RequestOptionsArgs {
    if (options == null) {
      options = new RequestOptions()
    }
    if (options.headers == null) {
      options.headers = new Headers({})
    }
    return options
  }

  beforeRequest() {
    this.starttime = Date.now()
  }

  afterRequest(res) {
    let roundtrip = Date.now() - this.starttime
    console.log("http roundtrip: " + roundtrip + 'ms' + ' ' + res.url)
    this.monitor.logHttpCall(roundtrip)
  }

  get(url: string, options?: RequestOptionsArgs): Observable<any> {
    this.beforeRequest()
    return super.get(url, this.requestOptions(options))
      .catch(this.onCatch)
      .do((res: Response) => {
        this.onSuccess(res)
      }, (error: any) => {
        this.onError(error)
      })

  }

  onCatch(error: any, caught: Observable<any>): Observable<any> {
    console.error(error)
    return Observable.throw(error)
  }

  onSuccess(res: Response): void {
    this.afterRequest(res)
  }

  onError(error: any): void {
    console.error("err:" + error)
  }

}
