import { Injectable } from '@angular/core'

import { Observable } from 'rxjs/Observable'

import 'rxjs/add/operator/map'
import 'rxjs/add/operator/catch'
import 'rxjs/add/operator/toPromise'
import 'rxjs/add/observable/throw'

import { LocalConfig } from '../config/local-config'

import { MemoryDbVariableMode, MemoryDbLocalStorageMode } from '../database/memorydb'
import { Data } from '../database/data'
import { PagedEntry } from './paged-entry'
import { Aggregate } from './aggregate'

@Injectable()
export class SearchApiServiceStub {
    private _data: Data

    constructor() {
        this._data = new Data(LocalConfig.DataProvider === "LocalStorage" ? new MemoryDbLocalStorageMode() : new MemoryDbVariableMode())
    }

    tokenizationTerms(table: string, property: string): Observable<any> {
        return Observable.create(async observer => {
            await this._data.ensureTokenization(table, _ => _[property])
            const terms = await this._data.tokenizationTerms(table)
            observer.next(terms)
            observer.complete()
        })
    }

    stats(table: string, property: string, aggregate: string): Observable<any> {
        let expression
        switch (aggregate) {
            case Aggregate.FirstLetter:
                expression = _ => _[property][0]
                break
            default:
                return Observable.throw('unregistered expression')
        }
        return Observable.create(async observer => {
            const stats = await this._data.stats(table, expression)
            observer.next(stats)
            observer.complete()
        })
    }

    _dump(table: string): Promise<any> {
        return new Promise((resolve, reject) => {
            this._data.all(table)
                .then(p => {
                    resolve(p)
                })
        })
    }

    sample(options: { [name: string]: any; }): Observable<Array<any>> {
        if (typeof options.datasource === 'undefined') {
            return Observable.throw('datasource is required')
        }
        // + no persistence of sample => no paging
        const count: number = <number>(options.count || 0)
        // console.log(`SearchApiService::sample|options:${JSON.stringify(options)}`)
        return Observable.create(async observer => {
            let data = await this._data.sample(options.datasource.trim(), count)
            observer.next(data.map(p => Object.assign({}, p)))
            observer.complete()
        })
    }

    search(options: { [name: string]: any; }): Observable<PagedEntry> {
        if (typeof options.datasource === 'undefined') {
            return Observable.throw('datasource is required')
        }
        if (typeof options.q === 'undefined') {
            return Observable.throw('q is required')
        }
        if (typeof options.filterProperty === 'undefined') {
            return Observable.throw('filterProperty is required')
        }
        let page: number = <number>(options.page || 1)
        if (page === 0) {
            page = 1
        }
        const pageSize: number = <number>(options.pageSize || 0)
        const text: string = <string>(options.text || '').toLowerCase()
        const datasource: string = <string>(options.datasource || '').toLowerCase()
        let start = (page - 1) * pageSize
        return Observable.create(async observer => {
            await this._data.ensureTokenization(datasource, p => p[options.filterProperty])
            const tokenizationFilterResult = await this._data.tokenizationFilter(datasource, p => p.toLowerCase().includes(options.q.toLowerCase()))
            const ids = await this._data.tokenizationSerializeIDs(tokenizationFilterResult)
            let data = await this._data.getAllByIds(datasource, ids)
            const re = new RegExp('(' + options.q + ')', 'gi')
            const total = data.length
            let paged
            if (pageSize > 0) {
                paged = data.slice(start, start + pageSize)
            } else {
                paged = data.slice(start)
            }
            paged.forEach(p => {
                const replacement = '<span class="search-highlight">{{_}}</span>'.replace('{{_}}', '$1')
                p['original::' + options.filterProperty] = p[options.filterProperty].replace(re, replacement)
            })
            observer.next({
                total,
                items: paged.map(p => p)
            })
            observer.complete()
        })
    }

    filter(options: { [name: string]: any; }): Observable<PagedEntry> {
        if (typeof options.datasource === 'undefined') {
            return Observable.throw('datasource is required')
        }
        if (typeof options.filterProperty === 'undefined') {
            return Observable.throw('filterProperty is required')
        }
        let page: number = <number>(options.page || 1)
        if (page === 0) {
            page = 1
        }
        const filterBy: string = options.filterBy || 'content'
        const pageSize: number = <number>(options.pageSize || 0)
        const text: string = <string>(options.text || '').toLowerCase()
        const datasource: string = <string>(options.datasource || '').toLowerCase()
        let start = (page - 1) * pageSize
        return Observable.create(async observer => {
            let data
            if (filterBy === 'terms') {
                await this._data.ensureTokenization(datasource, p => p[options.filterProperty])
                const tokenizationFilterResult = await this._data.tokenizationFilter(datasource, p => p.toLowerCase().startsWith(text))
                data = Object.keys(tokenizationFilterResult).map(p => {
                    let obj = {}
                    obj[options.filterProperty] = p
                    return obj
                })
            } else {
                data = await this._data.query(datasource, p => typeof p[options.filterProperty] !== 'undefined' && p[options.filterProperty].toLowerCase().startsWith(text))
            }

            // const content = await this.provider.dumpdb()
            // if (content.length === 32) {
            //     console.log(JSON.stringify(content))
            // }
            // console.log(data)
            const total = data.length
            // console.log(data.map(p => p.value))
            let paged
            if (pageSize > 0) {
                paged = data.slice(start, start + pageSize)
            } else {
                paged = data.slice(start)
            }
            // console.log(`start:${start}|pageSize:${pageSize}`)
            // console.log(paged.map(p => p.value))
            observer.next({
                total,
                items: paged.map(p => p)
            })
            observer.complete()
        })
    }
}
