import {Injectable} from '@angular/core'
import {Headers, Http} from '@angular/http'

import {Observable} from 'rxjs/Observable'

import 'rxjs/add/operator/map'
import 'rxjs/add/operator/catch'
import 'rxjs/add/operator/toPromise'

import {LocalConfig} from '../config/local-config'

import {IdGen} from '../database/idgen'
import {MemoryDb, MemoryDbLocalStorageMode, MemoryDbVariableMode} from '../database/memorydb'

export class LogEntryEntity {
  identifier: any // Specific instance of object
  reference: any // Specific instance of resource
  type: any // Type of entity involved
  role: any // What role the entity played
  lifecycle: any // Life-cycle stage for the entity
  securityLabel: Array<any> // Security labels on the entity
  name: string // C? Descriptor for entity
  description: string // Descriptive text
  query: string // C? Query parameters
  detail: Array<any>
}

export class LogEntryAgent {
  role: Array<any>
  reference: any
  userId: string
  altId: string // Alternative User id e.g. authentication
  name: string // Human-meaningful name for the agent
  requestor: boolean // R!  Whether user is initiator
  location: any
  policy: Array<string> // Policy that authorized event
  media: any // Type of media
  network: any
  purposeOfUse: Array<any> // Reason given for this user
}

export class LogEntry {
  id: string
  resourceType: string
  // from Resource: id, meta, implicitRules, and language
  // from DomainResource: text, contained, extension, and modifierExtension
  type: string// R!  Type/identifier of event
  subtype: Array<any> // More specific type/id for the event
  action: number // Type of action performed during the event
  recorded: number // R!  Time when the event occurred on source
  outcome: string // Whether the event succeeded or failed
  outcomeDesc: string // Description of the event outcome
  purposeOfEvent: [{}] // The purposeOfUse of the event
  agent: Array<LogEntryAgent>
  source: any
  entity: Array<LogEntryEntity>
}

export class PagedLogEntry {
  total: number
  items: LogEntry[]
}

@Injectable()
export class LoggingApiService {
  private headers = new Headers({'X-UserId': 'user01', 'X-Key': '12345'})
  private url: string
  private provider: MemoryDb

  constructor(private http: Http) {
    if (LocalConfig.DataProvider !== "http") {
      this.provider = new MemoryDb(LocalConfig.DataProvider === "LocalStorage" ? new MemoryDbLocalStorageMode() : new MemoryDbVariableMode())
    }

    // TODO use getServices() LocalConfig.ApiServer
    this.url = LocalConfig.ApiServer + '/api/v2/logging'
  }

  import(table: Array<any>) {
    this.provider.import(table)
  }

  add(type: string, data: string): Observable<string> {

    // console.log(`LoggingApiService::add|type:${type}|data:${data}`)
    if (LocalConfig.DataProvider !== "http") {
      return Observable.create(observer => {
        const ts = +(new Date())
        let id = IdGen.generate()
        let logEntry = new LogEntry()
        logEntry.id = id
        logEntry.outcomeDesc = data
        logEntry.type = type
        logEntry.recorded = ts
        this.provider.insert("log", id, {ts, value: logEntry, $text: JSON.stringify(logEntry)})
          .then(() => {
            observer.next(id)
            observer.complete()
          })
      })
    } else {
      const ts = +(new Date())
      let id = IdGen.generate()
      let logEntry = new LogEntry()
      logEntry.id = id
      logEntry.outcomeDesc = data
      logEntry.type = type
      logEntry.recorded = ts
      return this.http.post(this.url, logEntry, {headers: this.headers})
        .map(response => {
          return response.text()
        })
        .catch(
          (err, caught) => {
            console.log(err)
            return this.handleError(err)
          }
        )
    }
  }

  get(options: { [name: string]: any; }): Observable<PagedLogEntry> {
    let page: number = <number>(options.page || 1)
    if (page === 0) {
      page = 1
    }
    const pageSize: number = <number>(options.pageSize || 0)
    const text: string = <string>(options.text || '')
    let start = (page - 1) * pageSize
    // console.log(`LoggingApiService::get|start:${start}|pageSize:${pageSize}`)
    if (LocalConfig.DataProvider !== "http") {
      return Observable.create(async observer => {
        // const content = await this.provider.dumpdb()
        // if (content.length === 32) {
        //     console.log(JSON.stringify(content))
        // }
        let data = await this.provider.getAll("log")
        // console.log(data)
        if (typeof options.text !== 'undefined') {
          data = data.filter(p => {
            return p.$text.includes(options.text)
          })
        }
        const total = data.length
        // console.log(data.map(p => p.value))
        let paged
        if (pageSize > 0) {
          paged = data.slice(start, start + pageSize)
        } else {
          paged = data.slice(start)
        }
        // console.log(`start:${start}|pageSize:${pageSize}`)
        // console.log(paged.map(p => p.value))
        observer.next({
          total,
          items: paged.map(p => p.value)
        })
        observer.complete()
      })
    } else {
      let queryCommand = {
        start: start,
        size: options.pageSize,
        query: {
          textSearch: options.text // typeof options.text === 'undefined' ? null : options.text
        }
      }
      return this.http.post(this.url + '/query', queryCommand, {headers: this.headers})
        .map(response => {
            let res = response.json()
            // let pagedLogEntry = <PagedLogEntry> res
            // if (pagedLogEntry instanceof PagedLogEntry) {
            if (res.total && res.items) {
              return res
            } else {
              throw new Error(`POST::${this.url}::` + JSON.stringify(queryCommand) + ' res = ' + JSON.stringify(res))
            }
          }
        )
        .catch((err, caught) => this.handleError(err))
    }
  }

  handleError(err: any) {
    console.log('sever error:', err);  // debug
    if (err instanceof Response) {
      return Observable.throw(this.getErrorMessage(err));
    }
    return Observable.throw(err || 'backend server error');
  }

  getErrorMessage(err) {
    let error = err.json();
    if (typeof error === 'object' && error.resourceType === 'OperationOutcome' && error.issue && error.issue[0]) {
      return {
        errorCode: 504, // VA-FHIR-Server error
        message: error.issue[0].diagnostics || 'VA-FHIR-Server Error - unknown'
      }
    } else {
      return err.error || 'unknown server error'
    }
  }
}
