import {Injectable} from '@angular/core'

import {Observable} from 'rxjs/Rx';
import {Subject} from 'rxjs/Subject'

import {LogLevel} from './log-level'

interface ILogMessage {
  message: string
  level: LogLevel
}

interface IFormatProvider {
  (obj: any): string
}

const JsonFormatProvider: IFormatProvider = (obj: any): string => {
  if (typeof obj === 'undefined') {
    return ''
  }
  return JSON.stringify(obj)
}

const UpperCaseFormatProvider: IFormatProvider = (obj: any): string => {
  if (typeof obj === 'undefined') {
    return ''
  }
  if (typeof obj !== 'string') {
    return obj.toString()
  }
  return obj.toLocaleUpperCase()
}

interface IOutputProvider {
  (data: ILogMessage): void
}

const BrowserConsoleOutputProvider: IOutputProvider = (data: ILogMessage): void => {
  switch (data.level) {
    case LogLevel.Error:
      console.error(data.message)
      break
    case LogLevel.Warn:
      console.warn(data.message)
      break
    case LogLevel.Info:
      console.info(data.message)
      break
    case LogLevel.Debug:
      console.debug(data.message)
      break
  }
}

@Injectable()
class LoggerService {
  level: LogLevel
  private _formatProvider: IFormatProvider

  outputSubject: Subject<ILogMessage>

  get output(): Observable<ILogMessage> {
    return this.outputSubject.asObservable()
  }

  constructor() {
    this.outputSubject = new Subject<ILogMessage>()
  }

  public setformatProvider(formatProvider: IFormatProvider) {
    this._formatProvider = formatProvider
  }

  public setlogLevel(level: LogLevel) {
    this.level = level
  }

  public setOutputProvider(provider: IOutputProvider): void {
    this.output.subscribe(provider)
  }

  public log(message: any) {
    this._log(LogLevel.Info, message)
  }

  public warn(message: any) {
    this._log(LogLevel.Warn, message)
  }

  public error(message: any) {
    this._log(LogLevel.Error, message)
  }

  public info(message: any) {
    this._log(LogLevel.Info, message)
  }

  public debug(message: any) {
    this._log(LogLevel.Debug, message)
  }

  private _log(level: LogLevel, obj: any) {
    // console.log(`LoggerService::_log|level:${level}|this.level:${this.level}`)
    let effectiveLevel = this.level// || LogLevel.Error
    if (typeof this.level === 'undefined') {
      // console.trace('this.level is undefined')
      console.log('this.level is undefined')
    }
    if (level <= effectiveLevel) {
      let message
      if (typeof this._formatProvider !== 'undefined') {
        message = this._formatProvider(obj)
      } else {
        message = obj
      }
      this.outputSubject.next({message, level})
    }
  }
}

export {
  LoggerService,
  LogLevel,
  ILogMessage,
  IOutputProvider,
  IFormatProvider,
  UpperCaseFormatProvider,
  JsonFormatProvider,
  BrowserConsoleOutputProvider
}
