"use strict"

import { IdGen } from './idgen'
import { MemoryDbVariableMode, MemoryDbLocalStorageMode, MemoryDb } from './memorydb'

describe("Database:MemoryDb", function () {
    let provider

    beforeEach(() => {
        provider = new MemoryDb(new MemoryDbVariableMode())
    })

    it("simple add and dumpdb", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")
        const item = { "title": "hello1" }

        await provider.insert(scope, row_key, item)

        const table = await provider.dumpdb()
        expect(table[0].title).toEqual(item.title)
    })

    it("simple add and get", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")
        const item = { "title": "hello1" }

        await provider.insert(scope, row_key, item)

        const item_out = await provider.get(scope, row_key)
        expect(item_out.title).toEqual(item.title)
    })

    it("simple add, change id, and get", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")
        const new_row_key = IdGen.generate("new_row_key")
        const item = { "title": "hello1" }

        await provider.insert(scope, row_key, item)

        const p = await provider.get(scope, row_key)

        await provider.changeId(p, new_row_key)

        const q = await provider.get(scope, new_row_key)
        expect(q.title).toEqual(item.title)
    })

    it("2 adds and getAll", async function () {
        const scope = IdGen.generate("scope")
        const item1 = { "title": "hello1" }
        const item2 = { "title": "hello2" }

        await provider.insert(scope, IdGen.generate("1"), item1)
        await provider.insert(scope, IdGen.generate("2"), item2)

        const items = await provider.getAll(scope)
        expect(items[0].title).toEqual(item1.title)
        expect(items[1].title).toEqual(item2.title)
    })

    it("add, get, and update", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")
        const item = { "title": "hello1" }

        await provider.insert(scope, row_key, item)

        const v1 = await provider.get(scope, row_key)
        v1.title = 'was updated'

        await provider.update(v1)

        const v2 = await provider.get(scope, row_key)
        expect(v2.title).not.toEqual(item.title)
        expect(v2.title).toEqual('was updated')
    })

    it("3 adds, delete, and getAll", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")
        const item1 = { "title": "hello1" }
        const item2 = { "title": "hello2" }
        const item3 = { "title": "hello3" }

        await provider.insert(scope, IdGen.generate("1"), item1)
        await provider.insert(scope, row_key, item2)
        await provider.insert(scope, IdGen.generate("3"), item3)

        await provider.delete(scope, row_key)

        const items = await provider.getAll(scope)
        expect(items[0].title).toEqual(item1.title)
        expect(items[1].title).toEqual(item3.title)
    })

    it('expects stats', async () => {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")
        const item1 = { "title": "hello1" }
        const item2 = { "title": "hello2" }
        const item3 = { "title": "hello3" }

        await provider.insert(scope, IdGen.generate("1"), item1)
        await provider.insert(scope, row_key, item2)
        await provider.insert(scope, IdGen.generate("3"), item3)

        const stats = await provider.stats(scope, _ => _.title[0])
        const keys = Object.keys(stats)

        expect(keys.length).toBeGreaterThan(0)
        expect(stats[keys[0]]).toBeGreaterThan(0)
    })

    it("sample w/o data", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")
        let promises = []
        const sample = await provider.sample(scope, 10)
        expect(sample.length).toEqual(0)
    })

    it("20 adds and sample", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")
        let promises = []
        for (let n = 0; n < 20; n++) {
            await provider.insert(scope, IdGen.generate(n.toString()), { "title": IdGen.generate(n.toString()), order: n })
        }
        const sample = await provider.sample(scope, 10)
        expect(sample.length).toEqual(10)
    })

    it("3 adds and query", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")
        const item1 = { "title": "hello1" }
        const item2 = { "title": "frogs" }
        const item3 = { "title": "hello3" }

        await provider.insert(scope, IdGen.generate("1"), item1)
        await provider.insert(scope, row_key, item2)
        await provider.insert(scope, IdGen.generate("3"), item3)

        const query = _ => _.title.startsWith('h')
        const items = await provider.query(scope, query)
        expect(items.length).toEqual(2)
    })

    it("add, updateall, get", async function () {
        const scope = IdGen.generate("scope")

        const row_key1 = IdGen.generate("1")
        const row_key2 = IdGen.generate("2")
        const row_key3 = IdGen.generate("3")

        const item1 = { "title": "hello1" }
        const item2 = { "title": "frogs" }
        const item3 = { "title": "hello3" }

        await provider.insert(scope, row_key1, item1)
        await provider.insert(scope, row_key2, item2)
        await provider.insert(scope, row_key3, item3)

        let item1Copy = await provider.get(scope, row_key1)
        let item2Copy = await provider.get(scope, row_key2)
        let item3Copy = await provider.get(scope, row_key3)

        item1Copy.text = 'updateda'
        item2Copy.text = 'updatedb'
        item3Copy.text = 'updatedc'

        await provider.updateAll([item1Copy, item2Copy, item3Copy])

        const all = await provider.getAll(scope)
        expect(all[2].text).toEqual(item3Copy.text)
        expect(all[1].text).toEqual(item2Copy.text)
        expect(all[0].text).toEqual(item1Copy.text)
    })

    it("get tokenized data", async function () {
        const scope = IdGen.generate("scope")

        await provider.insert(scope, 'item1', { title: 'n n z c c c d d d dddd' })
        await provider.insert(scope, 'item2', { title: 'n n m c c d dddd doodle' })
        await provider.insert(scope, 'item3', { title: 'n n z m m c d d d donkey' })

        const tokenization = await provider.tokenize(scope, _ => _.title)
        const dump = tokenization.dump()

        let sample_key = tokenization.terms()[0]
        expect(sample_key).not.toBeUndefined()
        expect(sample_key.length).toBeGreaterThan(0)

        const dump_single = tokenization.dump()[sample_key]
        expect(dump_single).not.toBeUndefined()
        expect(dump_single.sum).toBeGreaterThan(0)
        expect(Object.keys(dump_single.spread).length).toBeGreaterThan(0)

        expect(dump['c'].spread['1'].value).toEqual('item3')
        expect(dump['c'].spread['2'].value).toEqual('item2')
        expect(dump['c'].spread['3'].value).toEqual('item1')

        expect(dump['m'].spread['1'].value).toEqual('item2')
        expect(dump['m'].spread['2'].value).toEqual('item3')

        expect(dump['z'].spread['1'].value).toEqual('item1')
        expect(dump['z'].spread['1'].next.value).toEqual('item3')

        expect(dump['d'].spread['1'].value).toEqual('item2')
        expect(dump['d'].spread['3'].value).toEqual('item1')
        expect(dump['d'].spread['3'].next.value).toEqual('item3')

        expect(dump['n'].spread['2'].value).toEqual('item1')
        expect(dump['n'].spread['2'].next.value).toEqual('item2')
        expect(dump['n'].spread['2'].next.next.value).toEqual('item3')
    })

    it("get tokenized data (terms)", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")

        await provider.insert(scope, 'item1', { title: 'n n z c c c d d d dddd' })
        await provider.insert(scope, 'item2', { title: 'n n m c c d dddd doodle' })
        await provider.insert(scope, 'item3', { title: 'n n z m m c d d d donkey' })

        const tokenization = await provider.tokenize(scope, _ => _.title)

        const terms = tokenization.terms()

        expect(terms).not.toBeUndefined()
        expect(terms.length).toBeGreaterThan(0)
    })

    it("get tokenized data (scan)", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")

        await provider.insert(scope, 'item1', { title: 'n n z c c c d d d dddd' })
        await provider.insert(scope, 'item2', { title: 'n n m c c d dddd doodle' })
        await provider.insert(scope, 'item3', { title: 'n n z m m c d d d donkey' })

        const tokenization = await provider.tokenize(scope, _ => _.title)

        let sample_key = 'd'
        const scan = tokenization.scan(sample_key)

        expect(scan[0]).toEqual(3)
        expect(scan[1]).toEqual(1)
    })

    it("get tokenized data (top)", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")

        await provider.insert(scope, 'item1', { title: 'n n z c c c d d d dddd' })
        await provider.insert(scope, 'item2', { title: 'n n m c c d dddd doodle' })
        await provider.insert(scope, 'item3', { title: 'n n z m m c d d d donkey' })

        const tokenization = await provider.tokenize(scope, _ => _.title)

        let sample_key = 'd'
        const top = tokenization.top(sample_key)
        expect(top).toEqual('item1')
    })

    it("get tokenized data (all)", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")

        await provider.insert(scope, 'item1', { title: 'n n z c c c d d d dddd' })
        await provider.insert(scope, 'item2', { title: 'n n m c c d dddd doodle' })
        await provider.insert(scope, 'item3', { title: 'n n z m m c d d d donkey' })

        const tokenization = await provider.tokenize(scope, _ => _.title)

        let sample_key = 'm'
        const all = tokenization.all(sample_key)
        expect(all[0]).toEqual('item2')
        expect(all[1]).toEqual('item3')
    })

    it("get tokenized data (all, chained)", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")

        await provider.insert(scope, 'item1', { title: 'n n z c c c d d d dddd' })
        await provider.insert(scope, 'item2', { title: 'n n m c c d dddd doodle' })
        await provider.insert(scope, 'item3', { title: 'n n z m m c d d d donkey' })

        const tokenization = await provider.tokenize(scope, _ => _.title)

        let sample_key = 'd'
        const all = tokenization.all(sample_key)

        expect(all[0]).toEqual('item2')
        expect(all[1]).toEqual('item1')
        expect(all[2]).toEqual('item3')
    })

    // it("get tokenized data (scan)", function (done) {
    //     const scope = IdGen.generate("scope")
    //     const row_key = IdGen.generate("row_key")
    //     let promises = []
    //     for (let n = 0; n < 20; n++) {
    //         promises.push(provider.insert(scope, IdGen.generate(), { "title": piglet(400), order: n }))
    //     }
    //     Promise.all(promises)
    //         .then(v => provider.tokenize(scope, _ => _.title)
    //             .then(tokenization => {
    //                 const terms = tokenization.terms()
    //                 let sample_key = terms[0]
    //                 const scan = tokenization.scan(sample_key)
    //                 const top = tokenization.top(sample_key)
    //                 const filtered = tokenization.filter(p => p.startsWith(sample_key[0]))
    //                 const filtered_keys = Object.keys(filtered)
    //                 filtered_keys.forEach(p => {
    //                     expect(p[0]).toEqual(sample_key[0])
    //                     expect(terms).to.include(p)
    //                 })
    //                 done()
    //             })
    //         )
    // })

    it("get tokenized data (filter)", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")

        await provider.insert(scope, 'item1', { title: 'n n z c c c d d d dddd' })
        await provider.insert(scope, 'item2', { title: 'n n m c c d dddd doodle' })
        await provider.insert(scope, 'item3', { title: 'n n z m m c d d d donkey' })

        const tokenization = await provider.tokenize(scope, _ => _.title)

        let sample_key = 'd'
        const top = tokenization.top(sample_key)
        const filtered = tokenization.filter(p => p.startsWith(sample_key))
        const filtered_keys = Object.keys(filtered)

        expect(filtered['d'].sum).toEqual(7)
        expect(filtered['dddd'].sum).toEqual(2)
        expect(sample_key.length).toBeGreaterThan(0)
    })

    it("get tokenized data (highlight)", async function () {
        const scope = IdGen.generate("scope")
        const row_key = IdGen.generate("row_key")

        await provider.insert(scope, 'item1', { title: 'n n z c c c d d d dddd' })
        await provider.insert(scope, 'item2', { title: 'n n m c c d dddd doodle' })
        await provider.insert(scope, 'item3', { title: 'n n z m m c d d d donkey' })

        const tokenization = await provider.tokenize(scope, _ => _.title)

        let sample_key = 'd'
        const top = tokenization.top(sample_key)
        const p = await provider.get(scope, top)
        // const re = new RegExp("\\b" + sample_key + "\\b", 'gi')
        // const expected = p.title.replace(re, `/${sample_key}/`)
        const highlighted = tokenization.highlight(sample_key, p.title, '/{{_}}/')
        expect(highlighted).toEqual('n n z c c c /d/ /d/ /d/ dddd')
    })
})
