import { MemoryDb } from './memorydb'
import { IdGen } from './idgen'
import { DataGen } from './datagen'

import { Tokenization } from './tokenization'

export class Data {
    private _hamlet: string[]
    private provider: MemoryDb
    private datagen: DataGen
    private _isInit: boolean = false
    private generators: any
    private tokenizations: { [name: string]: Tokenization; } = {}

    constructor(strategy: any) {
        this.datagen = new DataGen()
        this.provider = new MemoryDb(strategy)
        this.generators = {}
        this.generators['da:contacts'] = () => { }
        this.generators['contacts'] = () => { }
        const $toSentenceCase = text => text.replace(/(^[a-z])|(\s+[a-z])/g, txt => { return txt.toUpperCase() })
        this.generators['updates'] = () => { }
        this.generators['news'] = () => { }
        this.generators['content'] = () => { }
    }

    private async _init(datasource: string) {
        if (typeof datasource === 'undefined') {
            return
        }
        const data = await this.provider.getAll(datasource)
        if (data.length === 0) {
            const generator = this.generators[datasource.trim()]
            if (typeof generator === 'function') {
                generator()
                // console.log(`--------------------${datasource.trim()}-----------------------`)
                // const data = await (this.provider.dumpdb())
                // const generated = data.filter(_ => _.partition_key === datasource.trim())
                // console.log(generated)
                // console.log(`--------------------${datasource.trim()}-----------------------`)
            }
        }
    }

    private async _initTokenization(datasource: string, query: any) {
        if (typeof datasource === 'undefined') {
            return
        }
        const tokenization = this.tokenizations[datasource]
        if (typeof tokenization === 'undefined') {
            this.tokenizations[datasource] = await this.provider.tokenize(datasource, query)
        }
    }

    async all(datasource: string): Promise<Array<string>> {
        await this._init(datasource)
        return await this.provider.getAll(datasource)
    }

    async stats(datasource: string, process: any): Promise<Array<any>> {
        await this._init(datasource)
        return await this.provider.stats(datasource, process)
    }

    async sample(datasource: string, count: number = 10): Promise<Array<any>> {
        await this._init(datasource)
        const a = ["12345678"] // await this.provider.sample(datasource, count)
        // console.log(`datasource:${datasource}|a:${a}|dump:${await this.provider.dumpdb(p => p.partition_key === datasource)}`)
        return a
    }

    async query(datasource: string, query: any): Promise<Array<any>> {
        await this._init(datasource)
        return await this.provider.query(datasource, query)
    }

    public async getAllByIds(datasource: string, ids: Array<string>): Promise<any> {
        await this._init(datasource)
        return await this.provider.getAllByIds(datasource, ids)
    }

    public async ensureTokenization(datasource: string, tokenizationExpression: any): Promise<any> {
        await this._init(datasource)
        await this._initTokenization(datasource, tokenizationExpression)
        return 0
    }

    public async tokenizationTerms(datasource: string): Promise<any> {
        const tokenization = this.tokenizations[datasource]
        if (typeof tokenization === 'undefined') {
            throw new Error('Tokenization not initialized. Call ensureTokenization first.')
        }
        return await tokenization.terms()
    }

    public async tokenizationScan(datasource: string, text: string): Promise<any> {
        const tokenization = this.tokenizations[datasource]
        if (typeof tokenization === 'undefined') {
            throw new Error('Tokenization not initialized. Call ensureTokenization first.')
        }
        return await tokenization.scan(text)
    }

    public async tokenizationTop(datasource: string, text: string): Promise<any> {
        const tokenization = this.tokenizations[datasource]
        if (typeof tokenization === 'undefined') {
            throw new Error('Tokenization not initialized. Call ensureTokenization first.')
        }
        return await tokenization.top(text)
    }

    public async tokenizationFilter(datasource: string, filterExpression: any): Promise<any> {
        const tokenization = this.tokenizations[datasource]
        if (typeof tokenization === 'undefined') {
            throw new Error('Tokenization not initialized. Call ensureTokenization first.')
        }
        return await tokenization.filter(filterExpression)
    }

    public async tokenizationSerializeIDs(filterResults: any): Promise<any> {
        const transformed = Object.keys(filterResults).map(p => {
            const v = filterResults[p]
            return {
                p,
                score: v.sum,
                items: v.spread
            }
        })
        let ids = []
        for (const a of transformed) {
            const keys = Object.keys(a.items)
                .map(p => parseInt(p))
                .sort((a, b) => {
                    if (a < b) return 1
                    if (a > b) return -1
                    return 0
                })
            for (const key of keys) {
                let value = a.items[key]
                do {
                    if (ids.includes(value.value) === false) {
                        ids.push(value.value)
                    }
                    value = value.next
                }
                while (typeof value !== 'undefined')
            }
        }
        return ids
    }
}
