import {Injectable} from '@angular/core'
import {Headers, Http, Response} from '@angular/http'
import {Observable} from 'rxjs/Rx'
import 'rxjs/add/operator/map'
import 'rxjs/add/operator/catch'
import 'rxjs/add/operator/toPromise'
import 'rxjs/add/observable/fromPromise'

import {LocalConfig} from './local-config'
import {MemoryDb, MemoryDbLocalStorageMode, MemoryDbVariableMode} from '../database/memorydb'
import {ErrorService} from "../error-messages/error.service";

@Injectable()
export class ConfigApiService {
  private url: string
  private headers = new Headers({'X-UserId': 'user01', 'X-Key': '12345'})
  private provider: MemoryDb

  constructor(private http: Http, private errorService: ErrorService) {

    this.url = LocalConfig.ApiServer + '/api/v1/config'

    // TODO: Abandon getServices() for now, this will be blocking!
    // this.getServices((service) => service).subscribe(
    //   res => {
    //     this.url = res.coreAPI + '/config';
    //   }
    //   ,
    //   err => {
    //     this.errorService.setErrorCode("", err)
    //   }
    // )

    if (LocalConfig.DataProvider !== "http") {
      this.provider = new MemoryDb(LocalConfig.DataProvider === "LocalStorage" ? new MemoryDbLocalStorageMode() : new MemoryDbVariableMode())
      this.provider.insert('config', 'LogLevel', {value: LocalConfig.Defaults.LogLevel})
      this.provider.insert('config', 'SecondsForExternalRedirect', {value: LocalConfig.Defaults.SecondsForExternalRedirect})
      // this.provider.insert('config', 'ApiServer', { value: LocalConfig.Defaults.ApiServer })
      this.provider.insert('config', 'LocalDomains', {value: LocalConfig.Defaults.LocalDomains})
      this.provider.insert('config', '_TestNumericValue', {value: 42})
      this.provider.insert('config', '_TestArrayValue', {value: [1, 2, 3, 4]})
    }
  }

  update(name: string, value: any): Promise<Response> {
    // console.log(`ConfigApiService::update|name:${name}|value:${value}`)
    if (LocalConfig.DataProvider !== "http") {
      return new Promise((resolve, reject) => {
        this.provider.insert("config", name, {value})
          .then(() => {
            resolve()
          })
      })
    } else {
      return this.http.put(this.url + '/' + name, {value}, {headers: this.headers})
        .toPromise()
    }
  }

  get(name: string): any {
    // console.log(`ConfigApiService::get|name:${name}`)
    if (LocalConfig.DataProvider !== "http") {
      return new Promise((resolve, reject) => {
        this.provider.get("config", name)
          .then(item => {
            resolve(item.value)
          })
      })
    } else {
      return this.http.get(this.url + '/' + name, {headers: this.headers})
        .map(response => response.text())
        .catch((err, caught) => this.handleError(err))
    }
  }

  handleError(err: any) {
    console.log('sever error:', err);  // debug
    if (err instanceof Response) {
      return Observable.throw(this.getErrorMessage(err));
    }
    return Observable.throw(err || 'backend server error');
  }

  getErrorMessage(err) {
    let error = err.json();
    if (typeof error === 'object' && error.resourceType === 'OperationOutcome' && error.issue && error.issue[0]) {
      return {
        errorCode: 504, // VA-FHIR-Server error
        message: error.issue[0].diagnostics || 'VA-FHIR-Server Error - unknown'
      }
    } else {
      return err.error || 'unknown server error'
    }
  }
}
