import { DebugElement } from '@angular/core'
import { inject, TestBed,
          discardPeriodicTasks,
          ComponentFixture, tick,
          getTestBed, async, fakeAsync
        } from '@angular/core/testing'
import { By } from '@angular/platform-browser'
import { HttpModule, Http, BaseRequestOptions, XHRBackend } from '@angular/http'
import { MockBackend } from '@angular/http/testing'
import { FormsModule, ReactiveFormsModule } from '@angular/forms'
import { RouterModule, Router } from '@angular/router'
import { RouterTestingModule } from '@angular/router/testing'

import * as Axe from 'axe-core'
import { KarmaHelper } from '../../karma/karma-helper'
import { GrowlModule } from 'primeng/primeng'
import { AppComponent } from './app.component'
import { MenusService } from './core/menus/menus.service'
import { MainContentService } from './core/maincontent/maincontent.service'
import { LinkComponent } from './mccf-common/link/link.component'
import { ErrorService } from './core/error-messages/error.service'
import { ErrorModalComponent } from './core/error-messages/error-modal/error-modal.component'
import { ErrorMessage } from './core/error-messages/error-message'
import { ClipboardModule } from 'ngx-clipboard'
import { LoggingApiService } from './core/logger/logging-api.service'
import { ConfigService } from './core/config/config.service'
import { ConfigApiService } from './core/config/config-api.service'
import { LoggerService } from './core/logger/logger.service'
import { AppCommService } from './core/app-comm/app-comm.service'
import { MainContent } from './core/maincontent/maincontent'
import { LocalConfig } from './core/config/local-config'
import { SearchComponent } from './core/search/search.component'
import { SearchApiService } from './core/search/search-api.service'
import { SearchApiServiceStub } from './core/search/search-api.service.stub'
import {TooltipsService} from "./mccf-common/tooltips/tooltips.service";
import {MonitoringService} from "./core/application-insights/monitoring.service";
import {MainContentComponent} from "./mccf-system/main-content/main-content.component";
import { MessageService } from 'primeng/components/common/messageservice'
import {SeedService} from "./core/seed/seed.service";

describe('AppComponent', () => {
  let component: AppComponent
  let fixture: ComponentFixture<AppComponent>
  let originalTimeout

  beforeEach(async(() => {
    originalTimeout = jasmine.DEFAULT_TIMEOUT_INTERVAL
    jasmine.DEFAULT_TIMEOUT_INTERVAL = 140000

    LocalConfig.DataProvider = "memory"

    TestBed.configureTestingModule({
      imports: [
        FormsModule,
        ReactiveFormsModule,
        HttpModule,
        RouterModule,
        ClipboardModule,
        RouterTestingModule,
        HttpModule,
        GrowlModule
      ],
      declarations: [
        AppComponent,
        LinkComponent,
        ErrorModalComponent,
        SearchComponent
      ],
      providers: [{
        provide: Http, useFactory: (backend, options) => {
          return new Http(backend, options)
        },
        deps: [MockBackend, BaseRequestOptions]
      }, MockBackend, BaseRequestOptions,
        AppComponent,
        MainContentComponent,
        TooltipsService,
        SeedService,
        MonitoringService,
        MenusService,
        ErrorService,
        ErrorMessage,
        LoggingApiService,
        ConfigService,
        ConfigApiService,
        LoggerService,
        AppCommService,
        MainContentService,
        MessageService,
      { provide: SearchApiService, useClass: SearchApiServiceStub },
      ]
    })
  }))

  beforeEach(async(() => {
    TestBed.compileComponents()
  }))

  beforeEach(() => {
    fixture = TestBed.createComponent(AppComponent)
    component = fixture.componentInstance
    component.ngOnInit()
    fixture.detectChanges()
  })

  afterEach(function () {
    jasmine.DEFAULT_TIMEOUT_INTERVAL = originalTimeout;
  })

  xit('expects search results', inject([SearchApiService], async (searchApiService: SearchApiService) => {
    fixture.detectChanges()

    let de: DebugElement = fixture.debugElement.query(By.css('.usa-nav-submenu > app-search'))
    expect(de).not.toBeNull()
    const searchComponent: SearchComponent = de.componentInstance

    const sample = (await searchApiService._dump('content'))[10]

    searchComponent.filterText = sample.title[0]

    await searchComponent.applyFilter()

    expect(searchComponent.results.length).toBeGreaterThan(0)

    expect(searchComponent.results[0][searchComponent.filterProperty][0]).toEqual(searchComponent.filterText)
  }))

  xit('expects search results via tokenization', (done) => {
    inject([SearchApiService], async (searchApiService: SearchApiService) => {
      fixture.detectChanges()

      let de: DebugElement = fixture.debugElement.query(By.css('.usa-nav-submenu > app-search'))
      expect(de).not.toBeNull()
      const searchComponent: SearchComponent = de.componentInstance
      searchComponent.datasource = 'content'
      searchComponent.filterProperty = 'text'
      searchComponent.filterBy = 'terms'

      searchApiService.tokenizationTerms(searchComponent.datasource, searchComponent.filterProperty).subscribe(async terms => {
        let sample_key = terms[0]

        searchComponent.filterText = sample_key[0]

        await searchComponent.applyFilter()

        expect(searchComponent.results.length).toBeGreaterThan(0)

        expect(searchComponent.results[0][searchComponent.filterProperty][0]).toEqual(searchComponent.filterText)

        done()
      })
    })()
  })

  xit('expects Error (missing filterProperty)', inject([SearchApiService], async (searchApiService: SearchApiService) => {
    fixture.detectChanges()

    let de: DebugElement = fixture.debugElement.query(By.css('.usa-nav-submenu > app-search'))
    expect(de).not.toBeNull()
    const searchComponent: SearchComponent = de.componentInstance

    const sample = (await searchApiService._dump('content'))[10]

    searchComponent.filterProperty = undefined
    searchComponent.filterText = sample.title[0]

    try {
      await searchComponent.filterResults()
      expect(0).toEqual(1)
    } catch (ex) {
      expect(ex).toEqual('filterProperty is required')
      return
    }

    expect(0).toEqual(1)
  }))

  it('select page should have no accessibility issues', (done) => {
    component.maincontent = new MainContent()
    component.maincontent.phone = "777-7777"
    component.maincontent.emailText = "myEmail"
    component.maincontent.emailValue = "myEmail@myemail.us"
    fixture.detectChanges()

    // console.error(" @@@@@@@@@@@@@ html="+document.body.innerHTML)
    Axe.run(document, KarmaHelper.axeOptions()).then((result) => {
      KarmaHelper.axeViolationLog(result)
      expect(result.violations.length).toEqual(0)
      done()
    }).catch((error) => {
      console.error("AXE-CORE ERROR:" + error)

    })
  })

  it('should have three levels of growl popup notifications', fakeAsync(() => {
    // console.error(' @@@@@@@@@@@@@ html=' + document.body.innerHTML)

    component.messageService.addAll([
      {severity: 'success', summary: 'Success', detail: 'Success Message123'},
      {severity: 'warn', summary: 'Warning', detail: 'warning message'},
      {severity: 'error', summary: 'Error', detail: 'There is an error'}
    ])

    tick()
    fixture.detectChanges()

    let element = fixture.nativeElement
    let par = element.querySelectorAll('p')

    let snoticeFound = false
    let wnoticeFound = false
    let enoticeFound = false
    for (let i = 0; i < par.length; i++) {
      // console.error(' @@@@@@@@@@@@@ par=' + par[i].innerHTML)
      if (par[i].innerHTML === 'Success Message123') {
        snoticeFound = true
      }
      if (par[i].innerHTML === 'warning message') {
        wnoticeFound = true
      }
      if (par[i].innerHTML === 'There is an error') {
        enoticeFound = true
      }
    }

    expect(snoticeFound).toBe(true)
    expect(wnoticeFound).toBe(true)
    expect(enoticeFound).toBe(true)

    let de1: DebugElement = fixture.debugElement.query(By.css('div[class=ui-growl-item]'))
    expect(de1).not.toBeNull()
    console.log(" de1= " + de1)

    tick(10000) // wait for popup to close
    discardPeriodicTasks()

  }))

})
