import { inject, TestBed } from '@angular/core/testing'
import { HttpModule, Http, BaseRequestOptions, XHRBackend } from '@angular/http';
import { MockBackend } from '@angular/http/testing';

// import { Config } from '../config/config'
import { ConfigService } from '../config/config.service'
import { SearchApiService } from './search-api.service'
import { SearchApiServiceStub } from './search-api.service.stub'
import { Aggregate } from './aggregate'

import 'rxjs/add/operator/catch';

xdescribe('SearchApiService', function () {
    beforeEach(function () {
        TestBed.configureTestingModule({
            imports: [HttpModule],
            providers: [{
                provide: Http, useFactory: (backend, options) => {
                    return new Http(backend, options);
                },
                deps: [MockBackend, BaseRequestOptions]
            },
            { provide: SearchApiService, useClass: SearchApiServiceStub },
                MockBackend,
                BaseRequestOptions,
                ConfigService
            ]
        })
    })

    it('dumps', inject([SearchApiService], async (searchApiService: SearchApiService) => {
        const all = await searchApiService._dump('contacts')
        expect(all.length).toBeGreaterThan(0)
    }))

    it('gets sample of 10 items', (done) => {
        inject([SearchApiService], async (searchApiService: SearchApiService) => {
            searchApiService.sample({ datasource: 'contacts', count: 10 })
                .subscribe(data => {
                    expect(data.length).toEqual(10)
                    done()
                })
        })()
    })

    it('gets first-letter aggregate stats', (done) => {
        inject([SearchApiService], async (searchApiService: SearchApiService) => {
            return searchApiService.stats('contacts', 'name', Aggregate.FirstLetter)
                .subscribe(stats => {
                    const keys = Object.keys(stats)
                    expect(keys.length).toBeGreaterThan(5)
                    done()
                })
        })()
    })

    it('missing datasource on filter', (done) => {
        inject([SearchApiService], async (searchApiService: SearchApiService) => {
            searchApiService.filter({})
                .subscribe(
                stats => {
                    expect(0).toBeGreaterThan(1)
                },
                err => {
                    expect(err).toEqual('datasource is required')
                    done()
                },
                () => {
                    expect(1).toBeGreaterThan(2)
                })
        })()
    })

    it('expects Error (missing filterProperty)', (done) => {
        inject([SearchApiService], async (searchApiService: SearchApiService) => {
            searchApiService.filter({ datasource: 'asdf' })
                .subscribe(
                stats => {
                    expect(0).toBeGreaterThan(1)
                },
                err => {
                    expect(err).toEqual('filterProperty is required')
                    done()
                },
                () => {
                    expect(1).toBeGreaterThan(2)
                })
        })()
    })

    it('missing datasource on sample', (done) => {
        inject([SearchApiService], async (searchApiService: SearchApiService) => {
            searchApiService.sample({})
                .subscribe(
                stats => {
                    expect(0).toBeGreaterThan(1)
                },
                err => {
                    expect(err).toEqual('datasource is required')
                    done()
                },
                () => {
                    expect(1).toBeGreaterThan(2)
                })
        })()
    })

    it('invalid aggregate', (done) => {
        inject([SearchApiService], async (searchApiService: SearchApiService) => {
            searchApiService.stats('contacts', 'name', 'asdufhaskdfjhfasfdlkjh')
                .subscribe(
                stats => {
                    expect(0).toBeGreaterThan(1)
                },
                err => {
                    expect(err).toEqual('unregistered expression')
                    done()
                },
                () => {
                    expect(1).toBeGreaterThan(2)
                })
        })()
    })

    it('runs tokenization filter', (done) => {
        inject([SearchApiService], async (searchApiService: SearchApiService) => {
            searchApiService.tokenizationTerms('content', 'text')
                .subscribe(terms => {
                    let sample_key = terms[0]

                    let options = {}
                    options['text'] = sample_key[0]
                    options['filterProperty'] = 'text'
                    options['filterBy'] = 'terms'
                    options['datasource'] = 'content'
                    searchApiService.filter(options).subscribe(p => {
                        const items = p.items
                        expect(items[0][options['filterProperty']][0]).toEqual(sample_key[0])
                        done()
                    })
                })
        })()
    })

    it('runs filter', (done) => {
        inject([SearchApiService], async (searchApiService: SearchApiService) => {
            searchApiService.stats('contacts', 'name', Aggregate.FirstLetter)
                .subscribe(stats => {
                    const keys = Object.keys(stats)
                    expect(keys.length).toBeGreaterThan(0)

                    const knownToExist = Object.keys(stats)[0]

                    let options = {}
                    options['text'] = knownToExist
                    options['filterProperty'] = 'name'
                    options['datasource'] = 'contacts'
                    searchApiService.filter(options).subscribe(results => {
                        expect(results.total).toBeGreaterThan(0)
                        expect(typeof results.items[0]).not.toEqual('undefined')
                        expect(results.items.length).toBeLessThanOrEqual(results.total)
                        done()
                    })
                })
        })()
    })

    xit('missing filterProperty on filter', (done) => {
        inject([SearchApiService], async (searchApiService: SearchApiService) => {
            searchApiService.stats('contacts', 'name', Aggregate.FirstLetter)
                .subscribe(stats => {
                    const keys = Object.keys(stats)
                    expect(keys.length).toBeGreaterThan(0)

                    const knownToExist = Object.keys(stats)[0]

                    let options = {}
                    options['text'] = knownToExist
                    // options['filterProperty'] = 'name'
                    options['datasource'] = 'contacts'
                    searchApiService.filter(options).subscribe(
                        results => {
                            expect(0).toBeGreaterThan(1)
                        },
                        err => done()
                    )
                })
        })()
    })

    afterEach(function () {
        TestBed.resetTestingModule()
    })
})
