import { fakeAsync, inject, async, tick, ComponentFixture, TestBed } from '@angular/core/testing'
import { By } from '@angular/platform-browser'
import { DebugElement } from '@angular/core'
import * as Axe from 'axe-core'
import { FormsModule, ReactiveFormsModule } from '@angular/forms'
import { KarmaHelper } from '../../../../karma/karma-helper'
import { RouterTestingModule } from '@angular/router/testing'

import { HttpModule, Http, BaseRequestOptions, XHRBackend } from '@angular/http'
import { MockBackend } from '@angular/http/testing'
import { HomepageComponent} from "./homepage.component";

import { MessageService } from 'primeng/components/common/messageservice'
import { ErrorService } from '../../core/error-messages/error.service'
import { ConfigService } from '../../core/config/config.service'
import { SearchComponent } from '../../core/search/search.component'
import { SearchApiService } from '../../core/search/search-api.service'
import { SearchApiServiceStub } from '../../core/search/search-api.service.stub'
import { Aggregate } from '../../core/search/aggregate'
import { MainContentService } from '../maincontent/maincontent.service'
import {TooltipsService} from "../../mccf-common/tooltips/tooltips.service";
import { CalendarComponent } from '../calendar/calendar.component'
import {EnvironmentService} from '../../core/env/environment.service'

describe('HomepageComponent', () => {
  let component: HomepageComponent
  let fixture: ComponentFixture<HomepageComponent>

  beforeEach(async(() => {
    TestBed.configureTestingModule({
      imports: [
        FormsModule,
        ReactiveFormsModule,
        HttpModule,
        RouterTestingModule
      ],
      declarations: [
        HomepageComponent,
        SearchComponent,
        CalendarComponent
      ],
      providers: [{
          provide: Http, useFactory: (backend, options) => {
            return new Http(backend, options)
          },
          deps: [MockBackend, BaseRequestOptions]
          },
          { provide: SearchApiService, useClass: SearchApiServiceStub },
          HomepageComponent,
          TooltipsService,
          MockBackend,
          BaseRequestOptions,
          ErrorService,
          ConfigService,
          MainContentService,
          MessageService,
          CalendarComponent,
          EnvironmentService
      ]
    })
  }))

  beforeEach(async(() => {
    TestBed.compileComponents()
  }))

  beforeEach(() => {
    fixture = TestBed.createComponent(HomepageComponent)
    component = fixture.componentInstance
    component.ngOnInit()
    fixture.detectChanges()
  })

  afterEach(function () {
    TestBed.resetTestingModule()
  })

  xit('expects search results', inject([SearchApiService], async (searchApiService: SearchApiService) => {
    let de: DebugElement = fixture.debugElement.query(By.css('.homeContactBox > app-search'))
    expect(de).not.toBeNull()
    const searchComponent: SearchComponent = de.componentInstance

    expect(searchComponent.filterProperty).toBeDefined()

    const sample = (await searchApiService._dump(component.searchBoxDataSource))[10]

    searchComponent.filterText = sample[searchComponent.filterProperty][0]

    await searchComponent.applyFilter()

    expect(searchComponent.results.length).toBeGreaterThan(0)

    expect(searchComponent.results[0][searchComponent.filterProperty][0]).toEqual(searchComponent.filterText)
  }))

  xit('expects Error (missing datasource)', inject([SearchApiService], async (searchApiService: SearchApiService) => {
    let de: DebugElement = fixture.debugElement.query(By.css('.homeContactBox > app-search'))
    expect(de).not.toBeNull()
    const searchComponent: SearchComponent = de.componentInstance

    expect(searchComponent.filterProperty).toBeDefined()

    const contacts = (await searchApiService._dump(component.searchBoxDataSource))[10]

    searchComponent.datasource = undefined

    searchComponent.filterText = contacts[searchComponent.filterProperty][0]

    try {
      await searchComponent.filterResults()
      expect(0).toEqual(1)
    } catch (ex) {
      expect(ex).toEqual('datasource is required')
      return
    }

    expect(0).toEqual(1)
  }))

  xit('expects Error (missing filterProperty)', inject([SearchApiService], async (searchApiService: SearchApiService) => {
    let de: DebugElement = fixture.debugElement.query(By.css('.homeContactBox > app-search'))
    expect(de).not.toBeNull()
    const searchComponent: SearchComponent = de.componentInstance

    const contacts = (await searchApiService._dump(component.searchBoxDataSource))[10]

    searchComponent.filterProperty = undefined
    searchComponent.datasource = 'asdfasdf'
    searchComponent.filterText = contacts.name[0]

    try {
      await searchComponent.filterResults()
      expect(0).toEqual(1)
    } catch (ex) {
      expect(ex).toEqual('filterProperty is required')
      return
    }

    expect(0).toEqual(1)
  }))

  xit('expects search results of subset', (done) => {
    inject([SearchApiService], async (searchApiService: SearchApiService) => {
      let de: DebugElement = fixture.debugElement.query(By.css('.homeContactBox > app-search'))
      expect(de).not.toBeNull()
      const searchComponent: SearchComponent = de.componentInstance

      expect(searchComponent.filterProperty).toBeDefined()

      searchApiService.stats(component.searchBoxDataSource, searchComponent.filterProperty, Aggregate.FirstLetter)
        .subscribe(async stats => {
          // + use letter with largest count so subsequent query size will be less than or equal to
          searchComponent.filterText = Object.keys(stats)[0]

          await searchComponent.applyFilter()
          const initialResults = searchComponent.results.slice(0)

          expect(initialResults.length).toBeGreaterThan(0)

          const letter = initialResults[0][searchComponent.filterProperty][1]
          searchComponent.filterText += letter

          await searchComponent.applyFilter()
          const secondResults = searchComponent.results.slice(0)

          expect(secondResults.length).toBeLessThanOrEqual(initialResults.length)

          for (const needsToExist of secondResults) {
            expect(initialResults).toContain(needsToExist)
          }
          done()
        })
    })()
  })

  xit('expects rendered quick contact list', (done) => {
    inject([SearchApiService], fakeAsync(async (searchApiService: SearchApiService) => {
      let de: DebugElement = fixture.debugElement.query(By.css('ul.quick-contact'))
      expect(de).not.toBeNull()

      fixture.whenStable().then(() => {
        fixture.detectChanges()

        expect(component.quickContactData).toBeDefined()
        expect(component.quickContactData.length).toEqual(9)
        expect(de.children.length).toEqual(9)

        // de.children.forEach(p => {
        //   expect(p.nativeElement.textContent.trim().length).toBeGreaterThan(0)
        // })
        done()
      })
    }))()
  })

  it('expects rendered quick news list', (done) => {
    inject([SearchApiService], fakeAsync(async (searchApiService: SearchApiService) => {
      let de: DebugElement = fixture.debugElement.query(By.css('ul.quick-news'))
      expect(de).not.toBeNull()

      fixture.whenStable().then(() => {
        fixture.detectChanges()

        expect(component.quickNewsData).toBeDefined()
        // expect(de.children.length).toEqual(3)

        // de.children.forEach(p => {
        //   // console.log(p.nativeElement.textContent)
        //   expect(p.nativeElement.textContent.trim().length).toBeGreaterThan(0)
        // })
        done()
      })
    }))()
  })

  xit('expects rendered quick updates list', (done) => {
    inject([SearchApiService], fakeAsync(async (searchApiService: SearchApiService) => {
      let de: DebugElement = fixture.debugElement.query(By.css('ul.quick-updates'))
      expect(de).not.toBeNull()

      fixture.whenStable().then(() => {
        fixture.detectChanges()

        expect(component.quickUpdatesData).toBeDefined()
        expect(component.quickUpdatesData.length).toEqual(5)
        expect(de.children.length).toEqual(5)

        // de.children.forEach(p => {
        //   expect(p.nativeElement.textContent.trim().length).toBeGreaterThan(0)
        // })
        done()
      })
    }))()
  })

  it('div 1 should have no accessibility issues', ((done) => {
    // console.error(' @@@@@@@@@@@@@ html='+document.body.innerHTML)

    Axe.run(document, KarmaHelper.axeOptions()).then((result) => {
      KarmaHelper.axeViolationLog(result)
      expect(result.violations.length).toEqual(0)
      done()
    }).catch((error) => {
      console.error('AXE-CORE ERROR:' + error)
    })
  }))
})
