#! /bin/bash
#
#set -x

scriptname=${0}
scriptBase="$( cd "$( dirname "$scriptname" )";  pwd)"
VAULTPASS_PATH="~/.vault_pass"
REPO_FILE="chrome-requirements.yml"
ROOT_DIR="$( cd ../..;  pwd)"
TIMESTAMP=`date "+%Y%m%d_%H%M%S"`
LOG_DIR="${ROOT_DIR}/logs"
LOGFILE="${LOG_DIR}/chromeInstall_${TIMESTAMP}.log"
HOSTS=ci-server
playbooks=("installChrome.yml")

function usage() {
  echo "usage: $scriptname -e ENV"
  echo "      where ENV = AWS_DEV | MAG_ALL | MAG_CI | MAG_CIT | MAG_DEV | MAG_SQA | MAG_UAT | local"
  echo "optional arguments: "
  echo "  -u|--user Bitbucket Username"
  echo "  -p|--pass Bitbucket Password"
  echo "  -l|--logfile logfile"
  echo "  -c|--nocolor (for nocolor)]"
  exit 1
}

# ReadArgs reads the command line arguments passed to it
# and populates shell variables accordingly
ReadArgs() {
  LOGTXT "Entered ReadArg"
  while [[ $# -gt 0 ]] ; do
    key="$1"
    case $key in
      -c | --nocolor )
           UseColor=No
           CLEAR=""
           COL60=""
           NORMAL=""
           WHITE=""
           GREEN=""
           RED=""
           YELLOW=""
           BLUE=""
           shift # past argument
           LOGTXT "Processing without color"
           ;;
      -l | --logfile )
           if [[ $# -ge 2 ]] ; then
             LOGFILE="$2"
           else
             DEBUG "$1 argument requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           LOGTXT "Using LogFile $LOGFILE instead"
           # DEBUG  "Log output redirected to $LOGFILE"
           shift # past argument
           shift # past logfile entry
          ;;
      -u | --user )
           if [[ $# -ge 2 ]] ; then
             USERNAME="$2"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           LOGTXT "USERNAME: $USERNAME "
           # DEBUG  "USERNAME: $USERNAME "
           shift # past argument
           shift # past fqdn entry
           ;;
      -p | --pass )
           if [[ $# -ge 2 ]] ; then
             PASSWD="$2"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           LOGTXT "Using supplied password "
           # DEBUG  "Using supplied password $PASSWD "
           shift # past argument
           shift # past fqdn entry
           ;;
      -s | --skiproles )
           skiproles="true"
           LOGTXT "skiproles: $skiproles, not collecting roles from mccf BB repos"
           DEBUG "skiproles: $skiproles, not collecting roles from mccf BB repos"
           shift # past argument
           ;;
      -e | --env )
           if [[ $# -ge 2 ]] ; then
             ENV="$2"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             exit 9
           fi
           LOGTXT "Environment: $ENV"
           # DEBUG  "Environment $ENV"
           shift # past argument
           shift # past fqdn entry
           ;;
      * )  DEBUG "$1 is not a valid argument"
           LOGTXT "$1 is not a valid argument"
           exit 9
           ;;
    esac
  done
  LOGTXT "Exited ReadArgs"
}

#source the commonfunctions
. "$scriptBase/commonfunctions.sh"

ReadArgs $*

DEBUG "scriptBase: ${scriptBase}"
DEBUG "ROOT_DIR: ${ROOT_DIR}"
DEBUG "logging to: ${LOGFILE}"
DEBUG "vault pass path: ${VAULTPASS_PATH}"

DEBUG "Check if log directory exists and if not, create it"
if [[ ! -d ${LOG_DIR} ]]; then
  mkdir -Z ${LOG_DIR}
fi

DEBUG "Verify ENV was provided"
if [[ -z "${ENV}" ]]; then
  DEBUG "ENV required"
  usage
else
  DEBUG "operating on ENV ${ENV}"
fi
ansibleExtraVars="var_hosts=${HOSTS}"

if is_true "$skiproles" ; then
  DEBUG "skiproles specified, no need to install roles"
else
  DEBUG "Install roles from ${REPO_FILE}"
  getMCCFRoles ${REPO_FILE}
fi

DEBUG "Starting to process playbooks on ${HOSTS} for $ENV environment with vars: ${ansibleExtraVars}"

for item in "${playbooks[@]}"
do
  playBook=${item}
  LOGTXT "execute playbook ${playBook} for ${ENV} ${ansibleExtraVars}"
  runAnsiblePlaybook ${playBook}
  rtnCode=$?
  DEBUG "rtnCode from running ansible playbook is ${rtnCode}"
  if [[ $rtnCode -eq 0 ]]
  then
    DEBUG "${playBook} completed successfully"
    LOGTXT "${playBook} completed successfully for ${ENV} on ${HOSTS}"
  else
    DEBUG "${playBook} failed: returned: ${rtnCode}"
    LOGTXT "${playBook} failed: returned: ${rtnCode}"
  fi
  exit $rtnCode
done

# CONTACT INFO
CONTACT="Lee Benhart: Lee.Benhart@Halfaker.com or PII"

exit
