#!/bin/bash
#set -x

#Don't waste any time -- if you're not root, exit now
if [ "${UID}" != "0" ]; then
  echo "$0 Error: You must run this script as root" >&2
  exit 1
fi

TIMESTAMP=`date +%Y%m%d%H%M%S`
SCRIPTBASE="$(cd "$( dirname "$scriptname" )";  pwd)"
INPUTFILE="${SCRIPTBASE}/downloadRepoFiles.txt"
ROOT_DIR="$(cd ../..;  pwd)"
MCCF_STORAGE_ROOT="/var/www/html/pub/repo/maven/maven2"
LOGDIR="${ROOT_DIR}/logs"
LOGFILE="${LOGDIR}/getMavenCentralFiles-${TIMESTAMP}.log"
ERRLOG="${LOGDIR}/getMavenCentralFiles-${TIMESTAMP}-error.log"
MAVEN_CENTRAL_URL="https://repo.maven.apache.org/maven2"
JFROG_URL="http://repo.jfrog.org/artifactory/plugins-releases-local"
MULESOFT_URL="http://repository.mulesoft.org/releases"
MAVEN2_URL="http://repo1.maven.org/maven2"
MULESOFT_NEXUS_URL="https://repository.mulesoft.org/nexus/content/repositories/public"
WGETOPTS='--user-agent="any_user_agent" --auth-no-challenge -e robots=off -r -nd -nH -np --reject html -P '
CUTCMD="/usr/bin/cut -d/ "
GREPCMD="/usr/bin/grep"
MKDIRCMD="/usr/bin/mkdir -p . "

#LOGTXT writes to logfile
LOGTXT() {
  echo "$*" >> $LOGFILE
} 

#LOGERR writes to error logfile
LOGERR() {
  echo "$*" >> $ERRLOG
} 

# DEBUG sends the arguments passed to the terminal with an EOL at the end
DEBUG() {
  if [ -t 1 ] ; then echo -e "$*" ; fi
}

#rtnCode check
RTNCODE() {
  if [[ $rtnCode -eq 0 ]]; then
    LOGTXT "Successful: WGET dowload completed successfully for ${REPO_URL}/${REPO_PATH}"
    LOGTXT " "
  elif [[ $rtnCode -eq 8 ]]; then
    LOGERR "Error: WGET download for ${REPO_URL}/${REPO_PATH} returned ${rtnCode}.  Verify URL is valid."
    LOGERR " "
  else
    LOGERR "Error: WGET download for ${REPO_URL}/${REPO_PATH} returned ${rtnCode}"
    LOGERR " "
    exit $rtnCode
  fi
}

function readArgs() {
  LOGTXT "readArgs: Entered readArgs `date`"
  while [[ $# -gt 0 ]] ; do
    key="$1"
    case ${key} in
       -l | --logfile )
          if [[ $# -ge 2 ]] ; then
            LOGFILE="$2"
          else
            DEBUG "$1 argument requires another argument after it"
            LOGTXT "Error in $1 argument (no next argument provided)"
            exit 9
          fi
          LOGTXT "Using LogFile ${LOGFILE} instead"
          # DEBUG  "Log output redirected to ${LOGFILE}"
          shift # past argument
          shift # past logfile entry
          ;;
      -q | --quiet )
          if [[ $# -ge 1 ]] ; then
            WGETOPTS='--user-agent="any_user_agent" --auth-no-challenge -e robots=off -q -r -nd -nH -np --reject html -P '
            LOGTXT "Setting Quiet Mode from Command Line"
          else
            DEBUG "Error in $1 argument (no argument provided)"
            LOGTXT "Error in $1 argument (no argument provided)"
            exit 9
          fi
          shift # past argument
          shift # past user entry
          ;;
      * ) DEBUG "$1 is not a valid argument"
          LOGTXT "$1 is not a valid argument"
          exit 9
          ;;
    esac
  done
  LOGTXT "readArgs: Exited readArgs"
  LOGTXT " "
}

#Replace the Mulesoft URL, http://repository.mulesoft.org/releases, with Mulesoft Nexus URL. The Mulesoft repo is a licensed repo and can't be used. 
#Also replace the Maven2 URL, http://repo1.maven.org/maven2, with Maven Central URL. 
function preprocessInputFile() {
  #backup original file
  cp ${INPUTFILE} ${INPUTFILE}.bak
  DEBUG "preprocessInputFile: Replacing ${MULESOFT_URL} with ${MULESOFT_NEXUS_URL}"
  LOGTXT "preprocessInputFile: Replacing ${MULESOFT_URL} with ${MULESOFT_NEXUS_URL}"
  LOGTXT " "
  sed -i "s|${MULESOFT_URL}|${MULESOFT_NEXUS_URL}|g" ${INPUTFILE}
  DEBUG "preprocessInputFile: Replacing ${MAVEN2_URL} with ${MAVEN_CENTRAL_URL}"
  LOGTXT "preprocessInputFile: Replacing ${MAVEN2_URL} with ${MAVEN_CENTRAL_URL}"
  sed -i "s|${MAVEN2_URL}|${MAVEN_CENTRAL_URL}|g" ${INPUTFILE}
  sort -u -o ${INPUTFILE} ${INPUTFILE}
}

function getRepoFiles() {
  LOGTXT "getRepoFiles: Processing ${INPUTFILE}"
  LOGTXT " "
  readarray -t LINES <<< "$($GREPCMD "${REPO_URL}" ${INPUTFILE})"
  for LINE in "${LINES[@]}"; do
    DEBUG "getRepoFiles: $LINE"
    REPO_PATH=$(echo "$LINE" | $CUTCMD$CUT_VAR)
    REPO_PATH="${REPO_PATH}"
    DEBUG "getRepoFiles: $REPO_PATH"
    cd ${MCCF_STORAGE_ROOT}
    if [[ ! -d ${MCCF_STORAGE_ROOT}${REPO_PATH} ]]; then
      ${MKDIRCMD} ${REPO_PATH}
      LOGTXT "getRepoFiles: Downloading ${LINE} to ${MCCF_STORAGE_ROOT}/${REPO_PATH}"
      LOGTXT " "
      /usr/bin/wget ${WGETOPTS} ${MCCF_STORAGE_ROOT}/${REPO_PATH} ${LINE}
      rtnCode=$?
      RTNCODE
    fi
  done
  LOGTXT "getRepoPath: Exited getRepoPath"
  LOGTXT " "
}

function getRepoPath (){
  LOGTXT "getRepoPath: Processing Repo URL"
  LOGTXT " "
  array=( ${MAVEN_CENTRAL_URL} ${JFROG_URL} ${MULESOFT_NEXUS_URL} )
  for i in "${array[@]}"
  do
    REPO_URL=`echo $i`
    case "$i" in
      ${MAVEN_CENTRAL_URL})
        CUT_VAR="-f5-"
        ;;
      ${JFROG_URL})
        CUT_VAR="-f6-"
        ;;
      ${MULESOFT_NEXUS_URL})
        CUT_VAR="-f8-"
        ;;
      *)
        LOGERR "Error: In getRepoPath function, Repo URL ${REPO_URL} is not in ${INPUTFILE}."
        LOGERR" "
        ;;
    esac
    getRepoFiles 
  done
    LOGTXT "getRepoFiles: Exited getRepoFiles"
  LOGTXT " "
}

#Check if log directory exists and if not, create it
if [[ ! -d ${LOGDIR} ]]; then
  ${MKDIRCMD} ${LOGDIR}  
fi

#Check if inputfile exists and size > 0, and if not, log error and exit
if [[ ! -s ${INPUTFILE} ]]; then
  LOGERR "Error: ${INPUTFILE} file does not exist or is not empty"
  LOGERR" "
  exit 1
fi

readArgs $*
preprocessInputFile
getRepoPath
LOGTXT "Completed $0 at `date`"
