#!/bin/bash
#
#set -x

###############################################################################
# name: runPlaybook()
# description: executes ansible playbook
# inputs: 
#    playbook
#    ENV
#    HOSTS
#    vault-password-file
#    LOGFILE
# output: 
###############################################################################
function runAnsiblePlaybook() {
  LOGTXT "Entered runAnsiblePlaybook"
  ansibleRtn=""
  if [ $# -eq 1 ]
  then
    LOGTXT "Running ${ENV} install for ${1}"
    cd "${ROOT_DIR}/playbooks"
    if [[ "${ENV}" == "local" ]]
    then
      DEBUG "ansible-playbook ${1} -i ../environments/${ENV} --extra-vars \"${ansibleExtraVars}\""
      ansible-playbook ${1} -i "../environments/${ENV}" --extra-vars "${ansibleExtraVars}" >> $LOGFILE 2>&1
      ansibleRtn=$?
      DEBUG "ansible-playbook execution returned $ansibleRtn"
      return $ansibleRtn
    else
      DEBUG "ansible-playbook ${1} -i ../environments/${ENV} \"${ansibleExtraVars}\" --vault-password-file ${VAULTPASS_PATH}"
      ansible-playbook ${1} -i "../environments/${ENV}" --extra-vars "${ansibleExtraVars}" --vault-password-file ${VAULTPASS_PATH} >> $LOGFILE 2>&1 
      ansibleRtn=$?
      DEBUG "ansible-playbook execution returned $ansibleRtn"
      return $ansibleRtn
    fi
  else 
    LOGTXT "No playbook specified"
    exit 1
  fi
  LOGTXT "Exited runAnsiblePlaybook (ansibleRtn=$ansibleRtn)"
}

function runAnsiblePlaybookWithTags() {
  LOGTXT "Entered runAnsiblePlaybookWithTags"
  ansibleRtn=""
  if [ $# -eq 1 ]
  then
    LOGTXT "Running ${ENV} install for ${1}"
    cd "${ROOT_DIR}/playbooks"
    if [[ "${ENV}" == "local" ]]
    then
      DEBUG "ansible-playbook ${1} -i ../environments/${ENV} --extra-vars \"${ansibleExtraVars}\" --tags ${ansibleTags}"
      ansible-playbook ${1} -i "../environments/${ENV}" --extra-vars "${ansibleExtraVars}" --tags ${ansibleTags} >> $LOGFILE 2>&1
      ansibleRtn=$?
      DEBUG "ansible-playbook execution returned $ansibleRtn"
      return $ansibleRtn
    else
      DEBUG "ansible-playbook ${1} -i ../environments/${ENV} --extra-vars \"${ansibleExtraVars}\" --tags ${ansibleTags} --vault-password-file ${VAULTPASS_PATH}"
      ansible-playbook ${1} -i "../environments/${ENV}"  --extra-vars "${ansibleExtraVars}" --tags ${ansibleTags} --vault-password-file ${VAULTPASS_PATH} >> $LOGFILE 2>&1 
      ansibleRtn=$?
      DEBUG "ansible-playbook execution returned $ansibleRtn"
      return $ansibleRtn
    fi
  else 
    LOGTXT "No playbook specified"
    exit 1
  fi
  LOGTXT "Exited runAnsiblePlaybookWithTags (ansibleRtn=$ansibleRtn)"
}

###############################################################################
# name: getBBCredentials()
# description: get credentials from Bitbucket
# interactive
# output: USERNAME, PASSWD
###############################################################################
function getBBCredentials(){
  LOGTXT "Entered getBBCredentials"
  DEBUG "Collecting credentials to pull source from bitbucket"
  echo -n "Please enter bitbucket username: "
  read USERNAME
  # 8m is the control code to hide characters
  echo -n "Please enter bitbucket password: "
  read -s PASSWD
  LOGTXT "Exited getBBCredentials"
}

###############################################################################
# name: getMCCFRoles()
# description: collect specified MCCF ansible roles
# interactive
# output: USERNAME, PASSWD
###############################################################################
function getMCCFRoles(){
  LOGTXT "Entered getMCCFRoles"

  if [[ -z "${USERNAME}" ]]; then
    DEBUG "Credentials are required"
    getBBCredentials
  fi

  if [[ -z "${PASSWD}" ]]; then
    DEBUG "Credentials are required"
    getBBCredentials
  fi
  cd "${ROOT_DIR}/playbooks"
  if [[ ! -e ${REPO_FILE} ]]
  then
    DEBUG "${REPO_FILE} not found, exiting"
    exit 1
  fi
  echo "processing: ${REPO_FILE}"
  REPOS=$(sed -n -e '/^\s*name/p' ${REPO_FILE} | cut -d':' -f 2 | sed 's/^ *//;s/ *$//')
  for repo in ${REPOS} ; do
    PROJECT_NAME="$(echo -e "${repo}" | sed -e 's/[[:space:]]*$//')"
    DEBUG "--> ${PROJECT_NAME}"
    PROJECT_PATH="${ROOT_DIR}/roles/${PROJECT_NAME}"
    DEBUG "--> ${PROJECT_PATH}"
    # Make sure the required directories don't exist
    if [ -e "${PROJECT_PATH}" ] ; then
      DEBUG "Role ${PROJECT_NAME} already exists!"
      cd "${ROOT_DIR}/roles"
      rm -rf "${PROJECT_NAME}"
    fi
      # create get the role
    DEBUG "INFO: getting role ${PROJECT_NAME}..."
    DEBUG "===================================================="
    cd "${ROOT_DIR}/roles"
    mkdir ${PROJECT_NAME}

    USERORIGIN="https://${USERNAME}:${PASSWD}@bitbucket.org/halfaker/${PROJECT_NAME}.git"
    git clone ${USERORIGIN} "/tmp/${PROJECT_NAME}"
    DEBUG "sync /tmp/${PROJECT_NAME} to ${PROJECT_NAME}"
    if (( $? == 0 )); then
      rsync -av --exclude ".git*" --exclude ".git/" "/tmp/${PROJECT_NAME}" . >> $LOGFILE 2>&1 
      rm -rf "/tmp/${PROJECT_NAME}"
    else
      DEBUG "problem collecting repo ${PROJECT_NAME} for ${USERNAME}"
      LOGTXT "problem collecting repo ${PROJECT_NAME} for ${USERNAME}"
      DEBUG "please verify you entered the username and password correctly"
      DEBUG "all specified repos are required, exiting"
      exit 1
    fi
    echo
  done
  LOGTXT "Exited getMCCFRoles"
}

# LOGTXT sends arguments passed to the defined logfile
LOGTXT() {
   echo "$(date) $*" >> $LOGFILE
}

# DEBUG sends the arguments passed to the terminal with an EOL at the end
DEBUG() {
   if [ -t 1 ] ; then echo -e "$(date) $*" ; fi
}

# EDEBUG sends the arguments passed to the terminal WITH NO EOL character
EDEBUG() {
   if [ -t 1 ] ; then echo -en "$*" ; fi
}

is_true() {
  if [[ "$1" == "false" ]]; then
    return 1 # false
  fi
  if [[ -z "$1" ]]; then
#    echo "value is empty, therefore return false"
    return 1 # false
  else
    return 0 # true
  fi
}

is_false() {
  if [[ "$1" == "false" ]]; then
    return 0 # true
  fi
  if [[ -z "$1" ]]; then
#    echo "value is empty, therefore return true"
    return 0 # true
  else
    return 1 # false
  fi
}