#! /bin/bash
# ==== Begin Header ====
# mag-init.sh v1.0 2017-Dec-11 PII                      )
#  Initial release notes:
#  Source taken from prior "tas-init.sh". This file does "gold image" creation
#  Other parts of the original "tas-init.sh" remain separately.
#  Known Defects:
#   - No security (McAfee) installed
#   - Centrify SSH service un-tested in MAG environment
#
# These are the available command-line options:
#  -c or --nocolor    : Turn Off Color Debug output
#  -l or --logfile    : Change logfile (must come next)
#  -f or --fqdn       : Use FQDN instead of detecting it (must come next)
#  -s or --satlicense : Use the given key to subscribe to the VA Satellite
#  -u or --user       : Set BitBucket Username to access playbook
#  -p or --pass       : Set BitBucket Password to access playbook
#
# This script assumes a system in Microsoft Azure Government Cloud
#
# The tasks this script performs are essentially to create a "gold image"
# from which VA software can be added to the system.
#
# A log of the progress and results of this script is located at $LOGTXT
# If run interactively, the progress & results (different from the log file)
#      are shown on the screen. If the colorized output of the interactive
#      screen are not working, you can turn off the color with a -c option
#
# ==== End Header ====
# Don't waste any time -- if you're not root, exit now
if [ "${UID}" != "0" ]; then
    echo "$0 Error: You must run this script as root" >&2 
    exit 1
fi

# create and blank the logfile
LOGFILE=/var/log/$0.log ; cp /dev/null ${LOGFILE} ; chmod a+r ${LOGFILE}
MyIP=
MyHostname=
MyDomain=
MyFQDN=
MAGLEVEL=
VARH7BSTRAP="http://DNS.URL/pub/bootstrap/rhel7.sh"
VARH7LICDEV=9-mccf-preprod-el7
VARH7LICNPROD=9-mccf-preprod-el7
VARH7LICPROD=9-mccf-prod-el7
#
# VA DNS Servers & TAS CI servers for MAG Environment
# NOTE: For now, the DNS servers are identical to EDE DNS servers
#   That is expected to change at some time in the future
VAMAGDNS1=IP         
VAMAGDNS2=IP         
#
# Security Playbook Location
SECURITYSRC=bitbucket.org/halfaker/mag_sys_build
SECURITYDIR=mag_sys_build
SECURITYYML=./playbooks/all_high.yml
if [ -f "${SECURITYYML}" ] ; then
    echo "$0 Error: You cannot run this script from here" >&2 
    exit 1
fi
TASSRC=bitbucket.org/halfaker/mccf_devops
TASDIR=mccf_devops
#
# Define colors and terminal escape sequences for intereactive use
UseColor=Yes
CLEAR="\\033c"
COL60="\\033[60G"
NORMAL="\\033[0;39m"
WHITE="\\033[1;39m"
GREEN="\\033[1;32m"
RED="\\033[1;31m"
YELLOW="\\033[1;33m"
BLUE="\\033[1;34m"

SHOWPROCESS="${COL60}${BLUE}Processing${NORMAL}"
SHOWBUILD="${COL60}${BLUE}Building..${NORMAL}"
SHOWINSTALL="${COL60}${BLUE}Installing${NORMAL}"
SHOWSKIP="${COL60}${WHITE}[${YELLOW} SKIP ${WHITE}] ${NORMAL}"
SHOWOK="${COL60}${WHITE}[${GREEN} -OK- ${WHITE}] ${NORMAL}"
SHOWDONE="${COL60}${WHITE}[${GREEN} DONE ${WHITE}] ${NORMAL}"
SHOWPASS="${COL60}${WHITE}[${GREEN} PASS ${WHITE}] ${NORMAL}"
SHOWISSUE="${COL60}${WHITE}[${YELLOW}ISSUE ${WHITE}] ${NORMAL}"
SHOWFAIL="${COL60}${WHITE}[${RED} FAIL ${WHITE}] ${NORMAL}"
#
# LOGTXT sends arguments passed to the defined logfile
LOGTXT() {
   echo "$*" >> ${LOGFILE}
}
# DEBUG sends the arguments passed to the terminal with an EOL at the end
DEBUG() {
   if [ -t 1 ] ; then echo -e "$*" ; fi
}
# EDEBUG sends the arguments passed to the terminal WITH NO EOL character
EDEBUG() {
   if [ -t 1 ] ; then echo -en "$*" ; fi
}

SHOWUSAGE() {
#  -c or --nocolor    : Turn Off Color Debug output
#  -l or --logfile    : Change logfile (must come next)
#  -s or --satlicense : Use the given key to subscribe to the VA Satellite
#  -f or --fqdn       : Use FQDN instead of detecting it (must come next)
  DEBUG " "
  DEBUG "$0 Usage Summary:"
  DEBUG "$0 has the following options:"
  DEBUG "   -c | --nocolor : turns off colorized output to terminal"
  DEBUG "   -l | --logfile {logfile} : changes the default logfile"
  DEBUG "   -f | --fqdn {FQDN} : overrides the system FQDN"
  DEBUG "   -s | --satlicense {satellite license key} : assigns the "
  DEBUG "          VA RedHat Satellite Server Subscription Key"
  DEBUG "   -u | --user : set username for BitBucket pull"
  DEBUG "   -p | --pass : set password for BitBucket pull"
  DEBUG " "
  DEBUG "Correct your command line and try again..."
  DEBUG " "
  DEBUG "$0: Exiting with error. "
  exit 1
}
# ReadArgs reads the command line arguments passed to it
# and polulates shell variables accordingly
ReadArgs() {
  LOGTXT "Entering ReadArgs `date`"
  while [[ $# -gt 0 ]] ; do
    key="$1"

    case ${key} in
      -c | --nocolor ) 
           UseColor=No
           CLEAR=""
           COL60=""
           NORMAL=""
           WHITE=""
           GREEN=""
           RED=""
           YELLOW=""
           BLUE=""
           shift # past argument
           LOGTXT "Processing without color"
           ;;
      -l | --logfile )
           if [[ $# -ge 2 ]] ; then
             LOGFILE="$2"
           else
             DEBUG "$1 argument requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             SHOWUSAGE
           fi
           LOGTXT "Using LogFile ${LOGFILE} instead"
           # DEBUG  "Log output redirected to ${LOGFILE}"
           shift # past argument
           shift # past logfile entry
           ;;
      -f | --fqdn )
           if [[ $# -ge 2 ]] ; then
             MyFQDN="$2"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             SHOWUSAGE
           fi
           LOGTXT "Overriding FQDN lookup. Using ${MyFQDN} instead"
           # DEBUG  "Overriding FQDN lookup. Using ${MyFQDN} instead"
           shift # past argument
           shift # past fqdn entry
           ;;
      -u | --user )
           if [[ $# -ge 2 ]] ; then
             BBUSER="$2"
             LOGTXT "Setting BitBucket User from Command Line"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             SHOWUSAGE
           fi
           shift # past argument
           shift # past user entry
           ;;
      -p | --pass | --password )
           if [[ $# -ge 2 ]] ; then
             BBPASS="$2"
             DEBUG " "
             LOGTXT "Setting BitBucket Password from Command Line"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             SHOWUSAGE
           fi
           shift # past argument
           shift # past password entry
           ;;
      -s | --satlicense )
           if [[ $# -ge 2 ]] ; then
             VARH7LIC="$2"
             LOGTXT "Setting Satellite License from Command Line"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             SHOWUSAGE
           fi
           shift # past argument
           shift # past user entry
           ;;
      -u | --user )
           if [[ $# -ge 2 ]] ; then
             BBUSER="$2"
             LOGTXT "Setting BitBucket User from Command Line"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             SHOWUSAGE
           fi
           shift # past argument
           shift # past user entry
           ;;
      -p | --pass )
           if [[ $# -ge 2 ]] ; then
             BBPASS="$2"
             LOGTXT "Setting BitBucket Password from Command Line"
           else
             DEBUG "argument $1 requires another argument after it"
             LOGTXT "Error in $1 argument (no next argument provided)"
             SHOWUSAGE
           fi
           shift # past argument
           shift # past user entry
           ;;
      * )  DEBUG "$1 is not a valid argument"
           LOGTXT "$1 is not a valid argument"
           SHOWUSAGE
           ;;
    esac
  done
  LOGTXT "Exited ReadArgs"
}

# DetectSystem attempts to identify the system we're on based on IP address
# and PTR record (if any)
DetectSystem() {
  LOGTXT "Entering DetectSystem `date`"
  EDEBUG "Detecting System IP: "
  # This method uses our defined gateway to determine the IP address used
  # for getting out... in most cases, this will be our primary IP address
  InternetIP="`ip route get 8.8.8.8 | awk '{print $NF; exit}'`"
  LOGTXT "InternetIP found to be ${InternetIP}"
  # The hostname -I option returns ALL IPs (other than loopback)
  # In the EDE environment there are often 2 interfaces
  IPs="`hostname -I`"
  LOGTXT "List of all IPs on our interfaces is: ${IPs}"
  MyIP=${InternetIP}
  #
  # If we've been given the FQDN, don't bother detecting it
  if [ ! -z "${MyFQDN}" ] ; then
    LOGTXT "Using provided FQDN ${MyFQDN} and routable IP ${MyIP}"
    EDEBUG "${WHITE}${MyIP}${NORMAL}"
    DEBUG "${SHOWOK}"
  # We weren't told our FQDN, so we have to try to GET it
  # The only other way to get our FQDN is through the VA DNS servers
  elif PTR="`host ${MyIP} ${VAMAGDNS1} | DNS.URL`" ; then
    MyFQDN="`echo ${PTR} | sed 's/.* pointer //'`"
    LOGTXT "${MyIP} has PTR of ${MyFQDN} from ${VAMAGDNS1}"
    EDEBUG "${WHITE}${MyIP}${NORMAL}"
    DEBUG "${SHOWOK}"
  # If our IP isn't listed in the VA DNS server (and it wasn't provided)
  # We have little choice but to exit -- we don't know who we are!
  else 
    LOGTXT "Failed to determine which IP to use"
    EDEBUG "${YELLOW}${IPs}${NORMAL}"
    DEBUG "${SHOWISSUE}"
    exit 10
  fi
  #
  # At this point we should have defined MyFQDN and MyIP
  # Lets verify and set the MyHostname & MyDomain settings
  EDEBUG "Detecting System names: "
  if [ -z "${MyFQDN}" ] ; then
    LOGTXT "Failed to find an FQDN: fix DNS/PTR or specifiy on the command line with -f"
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failed to find your FQDN in DNS"
    DEBUG  " - Fix your DNS (PTR) entries for this IP address, or"
    DEBUG  " - use the --fqdn option to force the value in this script"
    exit 10
  elif ! MyHostname="`echo ${MyFQDN} | awk -F. '{ print $1 }'`" ; then
    LOGTXT "Error getting hostname from ${MyFQDN}"
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failed to get a valid hostname from your FQDN: ${MyFQDN}"
    exit 10
  elif ! MyDomain="`echo ${MyFQDN} | sed s/^${MyHostname}\.//`" ; then
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failed to get a valid domain name from your FQDN: ${MyFQDN}"
    LOGTXT "Error getting domain from ${MyFQDN}"
    exit 10
  fi
  EDEBUG "${WHITE}${MyFQDN}${NORMAL}"
  DEBUG "${SHOWOK}"

  # Next we need to set the hostname
  EDEBUG "Setting Hostname to: "
  if ! hostnamectl set-hostname ${MyFQDN} ; then
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failed to set hostname to ${MyFQDN} - Unknown cause"
    LOGTXT "Error setting hostname to ${MyFQDN}"
    exit 10
  fi
  # Ensure that we have ourselves listed in /etc/hosts
  if ! grep -q ${MyIP} /etc/hosts ; then
    echo ${MyIP} ${MyHostname} ${MyFQDN} >> /etc/hosts
  fi
  EDEBUG "${WHITE}${MyFQDN}${NORMAL}"
  DEBUG "${SHOWOK}"

  # Finally we need to determine our enviroment: MAG, EDE, AWS, or HAC
  EDEBUG "Detected System Environment is: "
  if `echo ${MyFQDN} | grep '.DNS' > /dev/null 2>&1` ; then
    if [ "${MyHostname:2:3}" == "c20" ] ; then
      if [ ${MyHostname:11:1} == "8" ] ; then
        MAGLEVEL=DEV
      elif [ ${MyHostname:11:1} == "4" ] ; then
        MAGLEVEL=NPROD
      elif [ ${MyHostname:11:1} == "2" ] ; then
        MAGLEVEL=PROD
      else
        DEBUG  "Error determining VA MAG install LEVEL -- assuming DEV"
        MAGLEVEL=DEV
      fi
      EDEBUG "${WHITE}MAG-${MAGLEVEL}${NORMAL}"
      DEBUG  "${SHOWOK}"
    else
      DEBUG "${SHOWFAIL}"
      DEBUG  "Error determining VA MAG install environment -- check hostname"
      LOGTXT "Error determining VA MAG install environment -- check hostname"
      exit 10
    fi
  else
    DEBUG "${SHOWFAIL}"
    DEBUG  "Error determining VA MAG install environment -- check domain name"
    LOGTXT "Error determining VA MAG install environment -- check domain name"
    exit 10
  fi
  LOGTXT "Exited ReadArgs"

  LOGTXT "${MyIP} is ${MyHostname}.${MyDomain}"
  LOGTXT "DetectSystem exited"
}
# SetDNS overrides the default DNS servers (known NOT to resolve VA addresses in MAG)
SetDNS() {
  LOGTXT "Entering SetDNS `date`"
  EDEBUG "Setting DNS Servers: "
  EDEBUG "${WHITE}${VAMAGDNS1} ${VAMAGDNS2}${NORMAL}"
  cat > /etc/resolv.conf << -END
search ${MyDomain}
nameserver ${VAMAGDNS1}
nameserver ${VAMAGDNS2}
-END
  DEBUG "${SHOWOK}"
  LOGTXT "Exiting SetDNS"
}

# SetRepos removed unused repos from the yum.repos.d folder, and when we're
# in the VA environment, signs us up for the Satellite repos
SetRepos() {
  LOGTXT "Entering SetRepos `date`"
  EDEBUG "Setting YUM Repo Servers: "
  # Remove any YUM cached data (more than a yum clean)
  rm -rf /var/cache/yum

  # Before anything else, we're going to remove all 32-bit packages
  # This is done silently, and we don't care about any errors
  yum erase "*i386" -y > /dev/null 2>&1
  yum erase "*i586" -y > /dev/null 2>&1
  yum erase "*i686" -y > /dev/null 2>&1

  # we NEVER subscribe to epel repos (but they are subscribed in MAG by default)
  # so, unsubscribe from any epel repos (may be none, but then this is harmless)
  #
  # As it turns out, this script MAY be run on a system that already subscribes
  # to the MCCF repo(s), and so that needs to be removed as well.
  HERE="${PWD}"
  cd /etc/yum.repos.d
  for i in epel* mccf* ; do  mv "$i" ."$i" ; done > /dev/null 2>&1
  cd "${HERE}"
  
  # VA environments use VA SATELLITE repos
  if ! yum repolist 2>/dev/null | grep -q "^va-" > /dev/null 2>&1 ; then
    EDEBUG "${WHITE}Subscribing to VA Satellite ${NORMAL}"
    if [ "${MAGLEVEL}" == "DEV" ] ; then
      VARH7LIC=${VARH7LICDEV}
    elif [ "${MAGLEVEL}" == "NPROD" ] ; then
      VARH7LIC=${VARH7LICNPROD}
    elif [ "${MAGLEVEL}" == "PROD" ] ; then
      VARH7LIC=${VARH7LICPROD}
    fi
    if [ -z "${VARH7LIC}" ] ; then
      DEBUG " "
      EDEBUG "${WHITE}Please enter your License Key for the Satellite Servers: ${YELLOW}"
      read VARH7LIC
      EDEBUG "${NORMAL}"
    fi
    # Add key to bootstrap script provided by sat server admins
    if ! curl -s "${VARH7BSTRAP}" | sed "s#^ACTIVATION_KEYS=.*#ACTIVATION_KEYS=${VARH7LIC}#" > ./vasatsubscribe.sh 2> /dev/null ; then
      DEBUG "${SHOWFAIL}"
      DEBUG  "Failed to obtain VA Satellite Subscription Script"
      LOGTXT "Failed to obtain VA Satellite Subscription Script"
      exit 15
    fi
    if ! chmod 700 ./vasatsubscribe.sh ; then
      DEBUG "${SHOWFAIL}"
      DEBUG  "Failed to obtain VA Satellite Subscription Script"
      LOGTXT "Failed to obtain VA Satellite Subscription Script"
      exit 15
    fi
    ./vasatsubscribe.sh >> "${LOGFILE}" 2>&1
    if [ "$?" -ne 0 ] ; then
      # Oddly, this script sometimes fails the first time, but
      # SUCCEEDS if you just try again...
      DEBUG "${SHOWISSUE}"
      EDEBUG "Retrying Satelliet Subscription..."
      if ! ./vasatsubscribe.sh >> "${LOGFILE}" 2>&1 ; then
        DEBUG "${SHOWFAIL}"
        DEBUG  "Failure in Satellite Subscription -- see logfile"
        LOGTXT "Failure in Satellite Subscription -- see logfile"
        exit 15
      fi
    fi
    rm -f ./vasatsubscribe.sh client-config-overrides.txt client_config_update.py
  else
    EDEBUG "${WHITE}Sat. Subscription already present${NORMAL}"
  fi
  DEBUG "${SHOWOK}"

  EDEBUG "Updating ALL with YUM... be patient (${YELLOW}Up to 30 mins${WHITE}?${NORMAL})"
  if ! yum install deltarpm -y >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failure in YUM install deltarpm -- see logfile"
    LOGTXT "Failure in YUM install deltarpm -- see logfile"
    exit 15

  elif ! yum update -y >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failure in YUM Updates -- see logfile"
    LOGTXT "Failure in YUM Updates -- see logfile"
    exit 15
  fi
  DEBUG "${SHOWOK}"

  LOGTXT "Exiting SetRepos"
}

GetGITAnsible() {
  LOGTXT "Entering GetGITAnsible `date`"
  # The playbooks that complete startup require Ansible and are retrieved with
  # GIT -- so we need to fetch them with YUM
  EDEBUG "Installing git & ansible for future installs..."
  if ! yum install git -y --disablerepo=\*mccf\* >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failure installing git"
    LOGTXT "Failure installing git"
    exit 15
  fi
  if ! yum install ansible -y --disablerepo=\*mccf\* >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}"
    DEBUG  "Failure installing ansible"
    LOGTXT "Failure installing ansible"
    exit 15
  fi
  DEBUG "${SHOWOK}"

  LOGTXT "Exiting GetGITAnsible"
}

#
# InstallSecurity
#
InstallSecurity() {
  LOGTXT "Entering InstallSecurity `date`"
  DEBUG  "Entering InstallSecurity"
  STARTDIR="`pwd`"
  #
  # If BitBucket Username, Password, or Vault Passwords are not provided...
  if [ -z "${BBUSER}" ] ; then
    EDEBUG "${WHITE}Please enter your Atlassian BitBucket Account User Name: ${YELLOW}"
    read BBUSER
  fi
  if [ -z "${BBPASS}" ] ; then
    EDEBUG "${WHITE}Please enter your password for the account ${BBUSER}: ${NORMAL}"
    read -s BBPASS
    DEBUG " "
  fi
  # Clean out any prior downloads
  if [ -d "${SECURITYDIR}" ] ; then rm -rf "${SECURITYDIR}" ; fi
  # Get the REPO from BitBucket
  EDEBUG "Downloading Security Playbook..."
  if ! git clone --depth 1 https://${BBUSER}:${BBPASS}@${SECURITYSRC} >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}"
    DEBUG " Error obtaining playbook -- check the logfile @ ${LOGFILE}"
    DEBUG " However, the most likely error is bad BitBucket permissions"
    LOGTXT "Sorry! git clone failed..."
    exit 20
  else
    DEBUG "${SHOWOK}"
  fi
  if [ ! -d "${SECURITYDIR}" ] ; then
    DEBUG "${SHOWFAIL}"
    DEBUG " Error accessing folder `pwd`/${SECURITYDIR}"
    exit 20
  fi
  cd "${SECURITYDIR}"
  if [ ! -f "${SECURITYYML}" ] ; then
    DEBUG "${SHOWFAIL}"
    DEBUG " Error accessing file ${SECURITYYML}" 
    exit 20
  fi
  DEBUG "${SHOWOK}"
  # Actually RUN the playbook
  EDEBUG "Running Security Playbooks -- be patient! (${YELLOW}15 mins${WHITE}?${NORMAL})"
  if ansible-playbook "${SECURITYYML}" >>${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWOK}"
  else
    DEBUG "${SHOWFAIL}"
    DEBUG "Ansible Playbook Failure -- See above"
    exit 20
  fi
  # Pre-fetch the follow-on TAS build script (with tas-init.sh)
  if ! git clone --depth 1 https://${BBUSER}:${BBPASS}@${TASSRC} >> ${LOGFILE} 2>&1 ; then
    DEBUG "${SHOWFAIL}"
    DEBUG " Error obtaining playbook -- check the logfile @ ${LOGFILE}"
    DEBUG " However, the most likely error is bad BitBucket permissions"
    DEBUG " Still, this is for the follow-on build -- THIS build will continue"
    LOGTXT "Sorry! git clone of ${TASSRC} failed... Continuing with MAG-INIT"
  fi

  cd "${STARTDIR}"
  rm -rf "${SECURITYDIR}"
  LOGTXT "Exiting InstallSecurity"
}
#
# THIS IS THE START OF ACTUAL PROCESSING
#
#
LOGTXT "Starting $0 at `date`"
EDEBUG ${CLEAR}
DEBUG "Welcome to the MAG System Preparation Script."
DEBUG "We need to confirm and/or set some things up before we can integrate"
DEBUG "this system into MAG. Please pay close attention to the output below:"
DEBUG "Additionally, a log file is created at ${LOGFILE}"
DEBUG " "

# We do as much as we can in Ansible Playbooks, which in this script are
# found in the InstallSecurity routine. Everything done BEFORE that is done
# to validate and install git (where we get our playbook from) and ansible
# (which runs the playbook)
ReadArgs $*
# DetectSystem makes sure we're running in the environment we think we are
DetectSystem
# Set DNS is required, because we need to access VA resources to subscribe
# to the VA RedHat Repositories
SetDNS
# With DNS set, we can access the vasat servers to register for RH updates
# If you don't have a license key for this, you are unable to continue, as
# you do not have a source for git and/or ansible -- which are required
SetRepos
# Now that we're connected to the RedHat YUM repos (via the VA Satellite
# server), we can add git and ansible to our system -- which will allow
# us to do all of the IMPORTANT work
GetGITAnsible
# InstallSecurity verifies and changes the system to conform to the VA
# publication "VA Baseline Configuration and Security Standard RHEL 7"
# As-of the writing of this script we are following version 1.1 (2017-08-10)
# HOWEVER, changes made because of updates/modifications to this document
# will actually appear in, and be reflected in, the playbook called in the
# InstallSecurity procedure
InstallSecurity
#
# When we are done, and assuming there wasn't an error (if there was, we
# won't get this far -- the script exits when fatal errors are found)
# we WARN the interactive user that a reboot is coming -- they have 
# time then to cancel (without losing anything) the reboot command.
LOGTXT "Completed $0 at `date`"
DEBUG "All done... everything passed. Rebooting system in 30 seconds for initial use"
DEBUG " NOTE: We have just installed a IDE system that will need to initialized."
DEBUG " The system will take AT LEAST 15 minutes to be available at next boot!"
DEBUG " "
DEBUG "Press CTRL-C to abort the reboot"
sleep 30
rm -f $0
init 6
