import { Component, OnInit, Inject } from '@angular/core';
import { FormControl, FormGroup, Validators } from '@angular/forms';
import { FhirDataService } from '../../mccf-data/fhir-data/fhir-data.service';
import { Resource } from '../../mccf-data/fhir-data/resource';
import { Patient } from '../../mccf-data/fhir-data/patient';
import { Provider } from '../../mccf-data/fhir-data/provider';
import { Search } from './search.interface';
import { PractitionerComponent } from './practitioner/practitioner.component';

@Component({
  selector: 'fhir-search',
  templateUrl: './fhir-search.component.html',
  providers: [FhirDataService],
  styleUrls:  ['./fhir-search.component.css']
})

export class FhirSearchComponent implements OnInit {
  FHIR_Practitioner_Service = 'FHIR Practitioner Service';
  FHIR_Treatment_Facility = 'FHIR Treatment Facility';
  Patient_Data = 'Patient Data (not FHIR)';
  Provider_List = 'Provider List (not FIHR)';
  service: FhirDataService;
  errorMessage: string = '';
  isLoading: boolean = false;
  page = 'select';
  reportType: string = '';
  report: string = '';

  resources : Array<Resource> = [];
  patients : Array<Patient> = [];
  providers : Array<Provider> = [];

  // form controls
  search: FormGroup;
  searchId:FormControl;
  accessCode:FormControl;
  verifyCode:FormControl;
  accessCodeAuthorized: boolean = true;

  constructor(@Inject(FhirDataService) Service) {
    this.service = Service;
  }

  ngOnInit() {
    this.search = new FormGroup({
      searchId: new FormControl('59', Validators.required),
      accessCode: new FormControl('user2##', Validators.required),
      verifyCode: new FormControl('u2123456#', Validators.required)
    });
  }

  onSubmit({ value, valid }: { value: Search, valid: boolean }) {
      this.isLoading = true;

      if(this.reportType === 'prv') {
        this.page = 'practitionerReport';
        let params = btoa(value.accessCode) + '/' +
                    btoa(value.verifyCode) + '/' +
                    value.searchId;

        this.service.getResources(this.reportType,params)
            .subscribe(resources => this.resources = resources,
            e => this.errorMessage = e,
            () => this.isLoading = false);

      } else if(this.reportType === 'tf') {
        this.page = 'treatmentFacilityReport';

        let params = value.searchId;

        this.service.getResources(this.reportType,params)
            .subscribe(resources => this.resources = resources,
            e => this.errorMessage = e,
            () => this.isLoading = false);

      } else if(this.reportType === 'hmppt') {
        this.page = 'patientReport';

        let params = value.searchId;

        this.service.getPatients(params)
            .subscribe(patients => this.patients = patients,
            e => this.errorMessage = e,
            () => this.isLoading = false);
      }

  }

  public back() {
    this.page = 'select'
  }

  public setReport(report:string) {
    this.reportType = report;

    if(this.reportType === 'prv') {
      this.report = this.FHIR_Practitioner_Service;
      this.page = 'search';
    } else if(this.reportType === 'tf') {
      this.report = this.FHIR_Treatment_Facility;
      this.page = 'search';
    } else if(this.reportType === 'hmppt') {
      this.report = this.Patient_Data;
      this.page = 'search';
    } else if(this.reportType === 'pl') {
      this.report = this.Provider_List;
      this.page = 'providerReport';
      this.service.getProviders()
          .subscribe(providers => this.providers = providers,
          e => this.errorMessage = e,
          () => this.isLoading = false);
    }
  }

  public setProvider(id:string) {
    this.page = 'practitionerReport';
    let params = btoa('user2##') + '/' +
                btoa('u2123456#') + '/' + id;

    this.service.getResources('prv',params)
        .subscribe(resources => this.resources = resources,
        e => this.errorMessage = e,
        () => this.isLoading = false);
  }

}
