import { IdGen } from './idgen'
import { WindowRef } from '../windowref'

import { Tokenization } from './tokenization'

let localStorage: any

const m = function (name) {
    console.trace('')
    throw new Error(`missing ${name}`)
}

abstract class MemoryDbMode {
    abstract persist(name: Array<any>)
    abstract load(): Array<any>
    abstract reset()
}

export class MemoryDbVariableMode extends MemoryDbMode {
    table: Array<any>

    persist(database: Array<any>) {
        this.table = database.slice(0)
    }
    load(): Array<any> {
        this.table = this.table || []
        return <Array<any>>this.table.slice(0)
    }
    reset() {
        this.table = []
    }
}

export class MemoryDbLocalStorageMode extends MemoryDbMode {
    _KEY: string

    constructor() {
        super()
        this._KEY = '__$database'
    }
    persist(database: Array<any>) {
        WindowRef.localStorage.setItem(this._KEY, JSON.stringify(database))
    }
    load(): Array<any> {
        return <Array<any>>(JSON.parse(WindowRef.localStorage.getItem(this._KEY)) || [])
    }
    reset() {
        WindowRef.localStorage.setItem(this._KEY, JSON.stringify([]))
    }
}

export class MemoryDb {
    constructor(public strategy: MemoryDbMode) {
    }

    async resetDatabase() {
    }

    import(data: Array<any>) {
        this.strategy.persist(data.slice(0))
    }

    async dumpdb(transform: any = function (_) { return _ }): Promise<any> {
        return transform(this.strategy.load())
    }

    async delete(partition_key: string = m('partition_key'), row_key: string = m('row_key')) {
        const table = this.strategy.load()
        const item = table.find(v => v.partition_key === partition_key && v.row_key === row_key)
        if (!item) {
            throw 404
        }
        else {
            const new_this_table = []
            for (let v of table) {
                if (v.partition_key !== partition_key || v.row_key !== row_key) {
                    new_this_table.push(v)
                }
            }
            this.strategy.persist(new_this_table)
        }
    }

    async insert(partition_key: string = m('partition_key'), row_key: string = m('row_key'), item: any = m('item')): Promise<string> {
        const table = this.strategy.load()
        let store: any = Object.assign({}, item)
        if (typeof store.ts !== 'undefined') {
            //+ helps put items in order
            store.ts += table.length
        }
        store.partition_key = partition_key
        store.row_key = row_key
        table.push(store)
        this.strategy.persist(table)
        return row_key
    }

    async get(partition_key: string = m('partition_key'), row_key: string = m('row_key')): Promise<any> {
        const item = this.strategy.load().find(v => v.partition_key === partition_key && v.row_key === row_key)
        if (item) {
            return Object.assign({}, item)
        }
        else {
            throw 404
        }
    }

    async stats(partition_key: string, process: any): Promise<Array<any>> {
        const data = await this.getAll(partition_key)
        let results
        if (typeof process === 'function') {
            results = data.map(process) || []
            results = results.reduce((groups, n) => {
                groups[n] = groups[n] || 0
                groups[n]++
                return groups
            }, {})
            const keys = Object.keys(results)
            results = keys
                .sort((a, b) => {
                    if (results[a] < results[b])
                        return 1
                    if (results[a] > results[b])
                        return -1
                    return 0
                })
                .reduce(function (result, key) {
                    result[key] = results[key]
                    return result
                }, {})
        }
        return results
    }

    async getAll(partition_key: string = m('partition_key')): Promise<Array<any>> {
        let expression
        const items = this.strategy.load().filter(v => v.partition_key === partition_key)
        items.sort((a, b) => {
            if (a.ts < b.ts)
                return 1
            if (a.ts > b.ts)
                return -1
            return 0
        })
        return items.map(v => Object.assign({}, v))
    }

    async getAllByIds(partition_key: string = m('partition_key'), ids: Array<string>): Promise<Array<any>> {
        if (ids.length === 0) {
            return []
        }
        const items = this.strategy.load()
        let results = []
        for (let id of ids) {
            results.push(Object.assign({}, items.find(v => v.partition_key === partition_key && v.row_key === id)))
        }
        return results
    }

    async changeId(item: any, new_id: string) {
        const table = this.strategy.load()
        if (typeof item.partition_key == 'undefined') {
            m('update|item.partition_key')
        }
        if (typeof item.row_key == 'undefined') {
            m('update|item.row_key')
        }
        await this.delete(item.partition_key, item.row_key)
        item.row_key = new_id
        table.push(Object.assign({}, item))
        this.strategy.persist(table)
    }

    async update(item: any) {
        if (typeof item.partition_key == 'undefined') {
            m('update|item.partition_key')
        }
        if (typeof item.row_key == 'undefined') {
            m('update|item.row_key')
        }
        await this.delete(item.partition_key, item.row_key)
        const table = this.strategy.load()
        table.push(Object.assign({}, item))
        this.strategy.persist(table)
    }

    async sample(partition_key: string = m('partition_key'), count: number = 10) {
        const items = this.strategy.load()
            .filter(p => p.partition_key === partition_key)
            .map(p => Object.assign({}, p))
        if (items.length === 0) {
            return []
        }

        const _sample = count => {
            if (count == 0) {
                return []
            }
            else {
                return _sample(count - 1).concat(items[Math.floor(Math.random() * items.length)])
            }
        }

        return _sample(count)
    }

    async query(partition_key: string = m('partition_key'), query: any = m('query')) {
        const table = this.strategy.load()
            .filter(p => p.partition_key === partition_key)
            .map(p => Object.assign({}, p))
        table.sort((a, b) => {
            if (a.ts < b.ts)
                return 1
            if (a.ts > b.ts)
                return -1
            return 0
        })
        let results = []
        if (typeof query === 'function') {
            results = table.filter(query) || []
        }
        return results.map(v => Object.assign({}, v))
    }

    async updateAll(items: Array<any> = m('items')) {
        for (const item of items) {
            await this.delete(item.partition_key, item.row_key)
        }
        for (const item of items) {
            await this.insert(item.partition_key, item.row_key, item)
        }
    }

    async tokenize(partition_key: string = m('partition_key'), expression: any = m('expression')) {
        const items = this.strategy.load()
            .filter(p => p.partition_key === partition_key)
            .map(p => Object.assign({}, p))

        if (typeof expression !== 'function') {
            throw new Error('Expression must be function')
        }

        let tokenized_data = {}
        items.forEach(item => {
            let p = (expression)(item)
            let split = p.split(' ')
            let reduced = split.reduce((groups, n) => {
                groups[n] = groups[n] || 0
                groups[n]++
                return groups
            }, {})
            const keys = Object.keys(reduced)
            let b = keys
                .sort((a, b) => {
                    if (reduced[a] < reduced[b])
                        return 1
                    if (reduced[a] > reduced[b])
                        return -1
                    return 0
                })
                .reduce(function (result, key) {
                    result[key] = reduced[key]
                    return result
                }, {})
            let breaker = 0
            for (let n in b) {
                let i = 0
                let value
                let v = parseInt(b[n])
                tokenized_data[n] = tokenized_data[n] || {}
                value = tokenized_data[n][v]
                if (typeof value === 'undefined') {
                    tokenized_data[n][v] = { value: item.row_key }
                }
                else {
                    let next = value.next
                    while (typeof next !== 'undefined') {
                        value = next
                        next = value.next
                        if (++breaker > 20) {
                            break
                        }
                    }
                    next = { value: item.row_key }
                    value.next = next
                }
            }
        })
        tokenized_data = Object.keys(tokenized_data)
            .reduce((result, key) => {
                const key_data = tokenized_data[key]
                result[key] = result[key] || {}
                let parent = Object.assign({}, key_data)
                let sum = 0
                const numeric_keys = Object.keys(parent)
                for (let n = 0; n < numeric_keys.length; n++) {
                    let value = parent[numeric_keys[n]]
                    const base = parseInt(numeric_keys[n])
                    let i = 0
                    do {
                        const before = sum
                        sum += base
                        i++
                    } while (typeof (value = value.next) !== 'undefined')
                }
                result[key].sum = sum
                result[key].spread = key_data
                return result
            }, {})
        return new Tokenization(tokenized_data)
    }
}