import { MemoryDbVariableMode, MemoryDb } from './memorydb'
import { Data } from './data'

describe('Database:Data', () => {
    let data: Data

    beforeEach(() => {
        data = new Data(new MemoryDbVariableMode())
    })

    it('expects 40 items (contacts)', async () => {
        const all = await data.all('contacts')
        expect(all.length).toBeGreaterThan(0)
    })

    it('expects stats (contacts)', async () => {
        const stats = await data.stats('contacts', _ => _.name[0])
        const keys = Object.keys(stats)
        expect(keys.length).toBeGreaterThan(0)
    })

    it('expects query for all to equal all set (contacts)', async () => {
        const all = await data.all('contacts')
        const results = await data.query('contacts', p => p.name.length > 0)
        expect(all.length).toBeGreaterThan(0)
        expect(all.length).toEqual(results.length)
    })

    it('get tokenized data (scan)', async () => {
        await data.ensureTokenization('content', p => p.text)

        const terms = await data.tokenizationTerms('content')
        let sample_key = terms[0]
        const scan = await data.tokenizationScan('content', sample_key)
        const top = await data.tokenizationTop('content', sample_key)
        const filtered = await data.tokenizationFilter('content', p => p.startsWith(sample_key[0]))
        const filtered_keys = Object.keys(filtered)
        filtered_keys.forEach(p => {
            expect(p[0]).toEqual(sample_key[0])
        })
    })

    it('get tokenized data (serialize IDs)', async () => {
        await data.ensureTokenization('content', p => p.text)

        const terms = await data.tokenizationTerms('content')
        let sample_key = terms[0]
        const scan = await data.tokenizationScan('content', sample_key)
        const filtered = await data.tokenizationFilter('content', p => p.startsWith(sample_key[0]))
        const ids = await data.tokenizationSerializeIDs(filtered)
        const filtered_keys = Object.keys(filtered)
        expect(ids.length).toBeLessThanOrEqual(filtered_keys.length)
    })

    it('expects sample of 10 items (contacts)', async () => {
        const results = await data.sample('contacts', 10)
        expect(results.length).toEqual(10)
    })

    it('expects sample of 10 items (content)', async () => {
        const results = await data.sample('content', 10)
        expect(results.length).toEqual(10)
    })

    it('expects 0 contacts (contacts)', async () => {
        const results = await data.query('contacts', p => p.name.length == 0)
        expect(results.length).toEqual(0)
    })

    it('expects sample of 10 items (news)', async () => {
        const results = await data.sample('news', 3)
        expect(results.length).toEqual(3)
    })

    it('expects sample of 10 items (updates)', async () => {
        const results = await data.sample('updates', 4)
        expect(results.length).toEqual(4)
    })

    // it('expects sample of 10 items (da:contacts)', async () => {
    //     const results = await data.sample('da:contacts', 10)
    //     expect(results.length).toEqual(10)
    // })

    it('expects 0 results (nonexistent datasource)', async () => {
        const results = await data.sample('asdfdsfafsdasdfasdfasdf', 10)
        expect(results.length).toEqual(0)
    })
})