/********************************************************************
 * Copyriight 2008 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.security;

import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.persistence.Transient;

import org.springframework.security.core.GrantedAuthority;

import gov.va.med.ccht.model.CCHTAuditFields;
import gov.va.med.ccht.model.User;

@Entity
@Table(schema="ht", name="ROLE")
public class Role extends CCHTAuditFields implements GrantedAuthority {

	private static final long serialVersionUID = -4464647987438818049L;

	public static final String DEFAULT_CCHT_APP_ROLE = "CCHT_USER";
	public static final String DEFAULT_OHRS_APP_ROLE = "OHRS_USER";

	private Long id;
	private String name;
	private String description;
	private boolean inactive = false;
	private Date inactiveDate = null;
	private Set<RolePermission> internalPermissions = null;

	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	@Column(name = "ROLE_ID", unique = true, nullable = false)
	public Long getId() {
		return id;
	}

	public void setId(Long id) {
		this.id = id;
	}

	/**
	 * @return the inactive
	 */
	@Transient
	public boolean isInactive() {
		return inactive;
	}

	/**
	 * @param inactive
	 *            the inactive to set
	 */
	public void setInactive(boolean inactive) {
		this.inactive = inactive;
	}

	/**
	 * @return the inactiveDate
	 */
	@Transient
	public Date getInactiveDate() {
		return inactiveDate;
	}

	/**
	 * @param inactiveDate
	 *            the inactiveDate to set
	 */
	public void setInactiveDate(Date inactiveDate) {
		this.inactiveDate = inactiveDate;
	}

	/**
	 * @return the description
	 */
	@Column(name="DESCRIPTION")
	public String getDescription() {
		return description;
	}

	/**
	 * @param description
	 *            the description to set
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * @return the internalPermissions
	 */
	@Transient
	public Set<Permission> getPermissions() {
		Set<Permission> permissions = new HashSet<Permission>();
		for (Iterator i = getInternalPermissions().iterator(); i.hasNext();) {
			permissions.add(((RolePermission) i.next()).getPermission());
		}
		return permissions;
	}
	
	/**
	 * @param internalPermissions
	 *            the internalPermissions to set
	 */
	public void setPermissions(Set<Permission> permissions, String username) {

		Set<Permission> newSet = new HashSet<Permission>();
		newSet.addAll(permissions);

		// old permission map
		Map<Permission, RolePermission> oldMap = new HashMap<Permission, RolePermission>();
		for (Iterator i = getInternalPermissions().iterator(); i.hasNext();) {
			RolePermission rolePermission = (RolePermission) i.next();
			oldMap.put(rolePermission.getPermission(), rolePermission);
		}

		// retain the existing ones and remove the deleted ones
		for (Iterator i = oldMap.keySet().iterator(); i.hasNext();) {
			Permission permission = (Permission) i.next();
			if (newSet.contains(permission)) {
				// an existing permission
				newSet.remove(permission);
			} else {
				// deleted permission
				RolePermission delRolePermission = (RolePermission) oldMap.get(permission);
				boolean removed = getInternalPermissions().remove(delRolePermission);
				delRolePermission.setRole(null);
			}
		}

		// add the new permissions
		for (Iterator i = newSet.iterator(); i.hasNext();) {
			Permission permission = (Permission) i.next();
			RolePermission rolePermission = new RolePermission();
			rolePermission.setPermission(permission);
			rolePermission.setRole(this);
			rolePermission.setRecordCreatedBy(username);
			rolePermission.setRecordModifiedBy(username);
			rolePermission.setRecordCreatedDate(Calendar.getInstance().getTime());
			rolePermission.setRecordModifiedDate(Calendar.getInstance().getTime());
			getInternalPermissions().add(rolePermission);
		}
	}

	/**
	 * @return the internalPermissions
	 */
	@OneToMany(mappedBy="role", targetEntity=RolePermission.class, fetch=FetchType.EAGER, cascade=CascadeType.ALL)
	public Set<RolePermission> getInternalPermissions() {
		if (this.internalPermissions == null)
			this.internalPermissions = new HashSet<RolePermission>();
		return internalPermissions;
	}

	/**
	 * @param internalPermissions
	 *            the internalPermissions to set, ONLY USE TO SET TO TO EMPTY IF NO ROLES SELECTED
	 *             OTHER WISE USE setPermissions
	 */
	public void setInternalPermissions(Set<RolePermission> internalPermissions) {
		this.internalPermissions = internalPermissions;
	}

	/**
	 * @return the name
	 */
	@Column(name="ROLE_NAME")
	public String getName() {
		return name;
	}

	/**
	 * @param name
	 *            the name to set
	 */
	public void setName(String name) {
		this.name = name;
	}

	@Transient
	public String getAuthority() {
		return getName();
	}

	/**
	 * Implement the compare function on the role name
	 */
	public int compareTo(Object obj) {
		if (name == null) {
			return -1;
		} else if (obj != null && Role.class.isAssignableFrom(obj.getClass())) {
			Role target = (Role) obj;
			if (target.name == null) {
				return 1;
			} else {
				return name.compareTo(target.name);
			}
		}
		return 0;
	}
}
