/*****************************************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ****************************************************************************************/
// Package
package gov.va.med.fw.scheduling;

// Java classes
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.log4j.Logger;
import org.quartz.CronScheduleBuilder;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.JobKey;
import org.quartz.JobListener;
import org.quartz.JobPersistenceException;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SchedulerFactory;
import org.quartz.SimpleTrigger;
import org.quartz.Trigger;
import org.quartz.Trigger.TriggerState;
import org.quartz.TriggerBuilder;
import org.quartz.TriggerKey;
import org.quartz.impl.StdScheduler;
import org.quartz.impl.StdSchedulerFactory;
import org.quartz.impl.matchers.GroupMatcher;
import org.quartz.plugins.history.LoggingJobHistoryPlugin;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.config.PropertiesFactoryBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ApplicationObjectSupport;
import org.springframework.stereotype.Service;

import edu.emory.mathcs.backport.java.util.Arrays;
import gov.va.med.ccht.util.ESAPIValidationType;
import gov.va.med.ccht.util.ESAPIValidator;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.trigger.TriggerEvent;
import gov.va.med.fw.util.InvalidConfigurationException;

/**
 * Provides methods to schedule, to unschedule, and to reschedule a job. This
 * service delegates to the underlying Quart Scheduling framework to do the
 * actual scheduling task. This service is aimed to encapsulate the actual
 * implementation and ocnfiguration of a scheduler and to provide the most
 * convenient methods for scheduling a task. For more advance tasks, consider
 * using a scheduler that can be obtained through a getScheduler method.
 * <p/>
 * Project: Framework</br> Created on: 10:52:58 AM </br>
 * 
 * @author DNS
 */
@Service
public class QuartzSchedulingService extends ApplicationObjectSupport implements Serializable,
		SchedulingService, InitializingBean {
	
	private Logger logger = Logger.getLogger(QuartzSchedulingService.class);

	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 3171280925841379597L;

	/**
	 * An instance of scheduler
	 */
	private Scheduler scheduler = null;

	private String immediateTriggerKey;

	private String futureTriggerKey;

	private boolean isClustered = false;
	
	/**
	 * @see gov.va.med.fw.service.AbstractComponent#afterPropertiesSet()
	 */
	@Override
	public void afterPropertiesSet() throws Exception {
		
		immediateTriggerKey = gov.va.med.fw.scheduling.SchedulingService.IMMEDIATE_TRIGGER_NAME;
		futureTriggerKey = gov.va.med.fw.scheduling.SchedulingService.FUTURE_TRIGGER_NAME;
		
		StdSchedulerFactory schedFact = new org.quartz.impl.StdSchedulerFactory();

        Properties props = new Properties();

        // General
        props.put(StdSchedulerFactory.PROP_SCHED_INSTANCE_NAME, "av_sched_clustered");
        props.put(StdSchedulerFactory.PROP_SCHED_INSTANCE_ID, "AUTO");

        // Thread pooling ?
        props.put("org.quartz.threadPool.class", org.quartz.simpl.SimpleThreadPool.class.getName());
        props.put("org.quartz.threadPool.threadCount", "25");
        props.put("org.quartz.threadPool.threadPriority", "5");

        // JobStore
        props.put("org.quartz.jobStore.class", "org.quartz.impl.jdbcjobstore.JobStoreTX");
        props.put("org.quartz.jobStore.driverDelegateClass", "org.quartz.impl.jdbcjobstore.StdJDBCDelegate");
        props.put("org.quartz.jobStore.useProperties", false);
        props.put("org.quartz.jobStore.dataSource", "sched");
        props.put("org.quartz.jobStore.tablePrefix", "QRTZ_");
        props.put("org.quartz.jobStore.isClustered", "true");
        props.put("org.quartz.jobStore.clusterCheckinInterval", "20000");
        props.put("org.quartz.jobStore.misfireThreshold", "60000");
        props.put("org.quartz.jobStore.maxMisfiresToHandleAtATime", "20");

        props.put("org.quartz.dataSource.sched.jndiURL", "jdbc.CchtDataSourceQuartz");
        props.put("org.quartz.dataSource.sched.validationQuery", "SELECT 1");
        props.put("org.quartz.dataSource.sched.validateOnCheckout", true);
        

        try {
			schedFact.initialize(props);
			scheduler = schedFact.getScheduler();
		} catch (SchedulerException e) {
			e.printStackTrace();
		}
		
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#isClustered()
	 */
	public boolean isClustered() {
		return this.isClustered;
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#setIsClustered(boolean)
	 */
	public void setIsClustered(boolean flag) {
		this.isClustered = flag;
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#getScheduler()
	 */
	public Scheduler getScheduler() {
		return scheduler;
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#setScheduler(org.quartz.Scheduler)
	 */
	public void setScheduler(Scheduler scheduler) {
		this.scheduler = scheduler;
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#getFutureTriggerKey()
	 */
	public String getFutureTriggerKey() {
		return futureTriggerKey;
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#setFutureTriggerKey(java.lang.String)
	 */
	public void setFutureTriggerKey(String futureTriggerKey) {
		this.futureTriggerKey = futureTriggerKey;
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#getImmediateTriggerKey()
	 */
	public String getImmediateTriggerKey() {
		return immediateTriggerKey;
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#setImmediateTriggerKey(java.lang.String)
	 */
	public void setImmediateTriggerKey(String immediateTriggerKey) {
		this.immediateTriggerKey = immediateTriggerKey;
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#schedule(org.quartz.Trigger)
	 */
	public Date schedule(Trigger trigger) throws SchedulerException {
		return scheduleJob(trigger);
	}

	private Date scheduleJob(Trigger trigger) throws SchedulerException {
		return scheduleJob(null, trigger);
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#schedule(org.quartz.JobDetail,
	 *      org.quartz.Trigger)
	 */
	public Date schedule(JobDetail jobDetail, Trigger trigger) throws SchedulerException {
		return scheduleJob(jobDetail, trigger);
	}
	
	/**
	 * @param job
	 *            the job
	 * @param trigger
	 *            the trigger
	 * @return the date scheduled
	 * @throws SchedulerException
	 *             if a problem was encountered
	 */
	private Date scheduleJob(JobDetail job, Trigger previousTrigger) throws SchedulerException {
		
		// Create new trigger to ensure unique name
		
		final TriggerKey key = new TriggerKey(previousTrigger.getKey().getName() + System.nanoTime(), previousTrigger.getKey().getName());
		
		Trigger trigger = TriggerBuilder
				.newTrigger()
				.forJob(new JobKey(key.getName(), key.getGroup()))
				.forJob(job)
				.build();
		
		if (job != null) {
			return getScheduler().scheduleJob(job, trigger);
		} else {
			return getScheduler().scheduleJob(trigger);
		}
		
	}


	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#schedule(org.quartz.JobDetail,
	 *      java.util.List, double, java.lang.String, java.lang.String)
	 */
	public Date schedule(JobDetail jobDetail, List contextDataForThisTrigger,
			double numberOfHoursDelayed, String triggerName, String triggerGroup)
			throws SchedulerException {

		long millis = ((long) Math.round(numberOfHoursDelayed * 60)) * 60 * 1000;
		
		final Date date = new Date(System.currentTimeMillis() + millis);
		
		Trigger trigger = TriggerBuilder
		.newTrigger()
		.forJob(jobDetail)
		.startAt(date)
		.build();
		
		if (contextDataForThisTrigger != null && !contextDataForThisTrigger.isEmpty()) {
			trigger.getJobDataMap().put(ScheduledProcessInvocationContext.INVOCATION_ARGUMENTS,
					contextDataForThisTrigger.toArray());
		} else {
			// to be safe...
			trigger.getJobDataMap().remove(ScheduledProcessInvocationContext.INVOCATION_ARGUMENTS);
		}
		
		return date;
	}



	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#triggerImmediately(gov.va.med.fw.scheduling.ScheduledProcessInvocationContext)
	 */
	public void fireJobWithImmediateTrigger(ScheduledProcessInvocationContext invocationContext)
			throws SchedulerException {

		if (logger.isInfoEnabled()) {
			logger.info("Scheduling job[" + invocationContext.getJobName() + "]" + " in group ["
					+ invocationContext.getJobGroup() + "] for immediate execution");
		}
		/*
		 * this is not as flexible since it requires the job to be known and
		 * registered with scheduler
		 * scheduler.triggerJob(invocationContext.getJobName(),
		 * invocationContext.getJobGroup());
		 */
		schedule(getSimpleTrigger(immediateTriggerKey, invocationContext));
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#triggerImmediately(org.quartz.JobDetail)
	 */
	public void fireJobWithImmediateTrigger(JobDetail job) throws SchedulerException {

		Validate.notNull(job, "A job must not be null");
		if (logger.isInfoEnabled()) {
			logger.info("Scheduling job[" + job.getKey().getName() + "]" + " in group [" + job.getKey().getGroup()
					+ "] for immediate execution");
		}
		this.schedule(this.getSimpleTrigger(this.immediateTriggerKey, job, null, null, true));
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#triggerImmediately(gov.va.med.fw.service.trigger.TriggerEvent)
	 */
	public void fireJobWithImmediateTrigger(TriggerEvent event) throws SchedulerException {
		fireJobWithImmediateTrigger((ScheduledProcessInvocationContext) event.getPayload());
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#unschedule(java.lang.String,
	 *      java.lang.String)
	 */
	public boolean unschedule(String triggerName, String groupName) throws SchedulerException {
		Trigger trigger = getTrigger(triggerName, groupName);
		return ((trigger != null) && (this.getScheduler().unscheduleJob(new TriggerKey(triggerName, groupName))));
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#reschedule(java.lang.String,
	 *      java.lang.String, org.quartz.Trigger)
	 */
	public Date reschedule(String triggerName, String groupName, Trigger newTrigger)
			throws SchedulerException {
		return this.getScheduler().rescheduleJob(new TriggerKey(triggerName, groupName), newTrigger);
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#interrupt(java.lang.String,
	 *      java.lang.String)
	 */
	public boolean interrupt(String jobName, String groupName) throws SchedulerException {
		Validate.notNull(jobName, "JobName must not be null.");
		if (logger.isInfoEnabled()) {
						
			 //Fix Fortify Issues
			ESAPIValidator.validateStringInput(jobName, ESAPIValidationType.ScheduledJob_Whitelist);
			ESAPIValidator.validateStringInput(groupName, ESAPIValidationType.ScheduledJob_Whitelist);
			
			logger.info("Interrupting a job[" + jobName + "]" + " in group [" + groupName + "] .");
		}
		return scheduler.interrupt(new JobKey(jobName, groupName));
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#interrupt(org.quartz.JobDetail)
	 */
	public boolean interrupt(JobDetail jobDetail) throws SchedulerException {
		Validate.notNull(jobDetail, "JobDetail must not be null.");
		return interrupt(jobDetail.getKey().getName(), jobDetail.getKey().getGroup());
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#getJobGroupNames()
	 */
	public String[] getJobGroupNames() throws SchedulerException {
		return scheduler.getJobGroupNames().toArray(new String[0]);
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#getJobNames(java.lang.String)
	 */
	public String[] getJobNames(String groupName) throws SchedulerException {
		
		final Set<JobKey> jobKeys = scheduler.getJobKeys(GroupMatcher.groupEquals(groupName));
		
		final List<String> names = new ArrayList<String>();
		names.forEach(name -> {
			names.add(name);
		});
		
		return names.toArray(new String[0]);
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#getTriggerGroupNames()
	 */
	public String[] getTriggerGroupNames() throws SchedulerException {
		return scheduler.getTriggerGroupNames().toArray(new String[0]);
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#getTriggerNames(java.lang.String)
	 */
	public String[] getTriggerNames(String groupName) throws SchedulerException {
		
		final Set<TriggerKey> keys = scheduler.getTriggerKeys(GroupMatcher.groupEquals(groupName));
		
		final List<String> names = new ArrayList<String>();
		
		keys.forEach(key -> {
			names.add(key.getName());
		});
		
		return names.toArray(new String[0]);
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#getJobDetail(java.lang.String,
	 *      java.lang.String)
	 */
	public JobDetail getJobDetail(String jobName, String jobGroup) throws SchedulerException {
		return scheduler.getJobDetail(new JobKey(jobName, jobGroup));
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#getTrigger(java.lang.String,
	 *      java.lang.String)
	 */
	public Trigger getTrigger(String triggerName, String triggerGroup) throws SchedulerException {
		return scheduler.getTrigger(new TriggerKey(triggerName, triggerGroup));
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#getTriggerStatus(java.lang.String,
	 *      java.lang.String)
	 */
	public TriggerStatus getTriggerStatus(String triggerName, String triggerGroup)
			throws SchedulerException {
		Trigger trigger = getTrigger(triggerName, triggerGroup);
		TriggerStatus status = new TriggerStatus();
		if (trigger != null) {
			status.setNextFireTime(trigger.getFireTimeAfter(new Date()));
			// Timezone is supported only in the cron trigger
			if (trigger instanceof CronTrigger) {
				CronTrigger cronTrigger = (CronTrigger) trigger;
				status.setTimeZone(cronTrigger.getTimeZone());
			}

			
			final TriggerState state = scheduler.getTriggerState(new TriggerKey(triggerName, triggerGroup));
			
			if (state == TriggerState.NORMAL)
				status.setStatus(TriggerStatus.NORMAL_STATUS);
			else if (state == TriggerState.ERROR)
				status.setStatus("ERROR");
			else if (state == TriggerState.COMPLETE)
				status.setStatus("COMPLETE");
			else if (state == TriggerState.PAUSED)
				status.setStatus(TriggerStatus.PAUSED_STATUS);
			else if (state == TriggerState.BLOCKED)
				status.setStatus("BLOCKED");
			else
				status.setStatus(StringUtils.EMPTY + state);
		}
		return status;
	}

	public Object getComponent(String name, Class<?> type) throws ServiceException {

		Object component = null;
		try {
			ApplicationContext context = getApplicationContext();
			component = context.getBean(name, type);
		} catch (BeansException e) {
			// Failed to get a component
			throw new ServiceException("Failed to get a component by name " + name + " and type "
					+ type, e);
		}
		return component;
	}

	
	/**
	 * Returns a simple trigger defined in a spring context by the specific key.
	 * A trigger is initialized with a job name, a group name, and a job defined
	 * in a spring appplication context by a job name
	 * 
	 * @param triggerKey
	 *            A key to look up for a trigger in an application context
	 * @param context
	 *            the scheduled process invocation context
	 * @return the simple trigger
	 * @throws SchedulerException
	 *             if a problem was encountered
	 */
	private Trigger getSimpleTrigger(String triggerKey,
			ScheduledProcessInvocationContext context) throws SchedulerException {
		try {
			Object component = getComponent(context.getJobDetailBeanName(), JobDetail.class);
			JobDetail job = component instanceof JobDetail ? (JobDetail) component : null;

			if(job != null)
			{
				Trigger trigger = getSimpleTrigger(triggerKey, job, null, null, true);

				// set dynamic stuff on Trigger (not Job)...this allows to reuse
				// same
				// "core" Job for static and dynamic use
				if (context.getInvocationArguments() != null) {
					trigger.getJobDataMap().put(ScheduledProcessInvocationContext.INVOCATION_ARGUMENTS,
							context.getInvocationArguments());
				} else {
					// to be safe...
					trigger.getJobDataMap().remove(
							ScheduledProcessInvocationContext.INVOCATION_ARGUMENTS);
				}

				// ScheduledProcessInvocationContext.EXECUTION_CONTEXT should always
				// be there
				trigger.getJobDataMap().put(ScheduledProcessInvocationContext.EXECUTION_CONTEXT,
						context.getExecutionContext());

				return trigger;
			}
			else {
				throw new SchedulerException("Unable to trigger task immediately job detail is null");
			}
		} catch (Throwable t) {
			InvalidConfigurationException e = new InvalidConfigurationException(
					"Unable to get a SimpleTrigger for job", t);
			throw new SchedulerException("Unable to trigger task immediately", e);
		}
	}

	/**
	 * Returns a simple trigger defined in a spring context by the specific key.
	 * A trigger is initialized with a job name, a group name, and the specific
	 * job
	 * 
	 * @param triggerKey
	 *            the trigger key
	 * @param job
	 *            the job detail information
	 * @param triggerName
	 *            the trigger name
	 * @param triggerGroup
	 *            the group name
	 * @return the simple trigger
	 * @throws SchedulerException
	 *             if a problem was encountered
	 */
	private Trigger getSimpleTrigger(String triggerKey, JobDetail job, String triggerName,
			String triggerGroup, boolean addJob) throws SchedulerException {
		try {
			
			JobKey key = new JobKey(triggerName + "-" + System.currentTimeMillis(), triggerGroup);
			
			Trigger trigger = TriggerBuilder
					.newTrigger()
					.forJob(key)
					.forJob(job)
					.build();

			// for those jobs that are not already registered with Scheduler, do
			// this
			if (addJob)
				scheduler.addJob(job, true);
			
			return trigger;
			
		} catch (Exception e) {
			InvalidConfigurationException e2 = new InvalidConfigurationException(
					"Unable to get a SimpleTrigger for job", e);
			throw new SchedulerException("Unable to trigger task immediately", e2);
		}
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#pauseJob(java.lang.String,
	 *      java.lang.String)
	 */
	public void pauseJob(String jobName, String jobGroup) throws SchedulerException {
		scheduler.pauseJob(new JobKey(jobName, jobGroup));
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#pauseAllJobs()
	 */
	public void pauseAllJobs() throws SchedulerException {
		scheduler.pauseAll();
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#resumeJob(java.lang.String,
	 *      java.lang.String)
	 */
	public void resumeJob(String jobName, String jobGroup) throws SchedulerException {
		scheduler.resumeJob(new JobKey(jobName, jobGroup));
	}

	/**
	 * @see gov.va.med.fw.scheduling.SchedulingService#resumeAllJobs()
	 */
	public void resumeAllJobs() throws SchedulerException {
		scheduler.resumeAll();
	}

	public void pauseTriggerGroup(String triggerGroup) throws SchedulerException {
		scheduler.pauseTriggers(GroupMatcher.groupEquals(triggerGroup));
	}

	public void resumeTriggerGroup(String triggerGroup) throws SchedulerException {
		scheduler.resumeTriggers(GroupMatcher.groupEquals(triggerGroup));
	}
}