/********************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.rule.impl;

// Java classes
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.Validate;

import gov.va.med.fw.model.AbstractEntity;
import gov.va.med.fw.rule.RuleDataAware;

// EDB classes

/**
 * Project: Framework
 * 
 * @author DNS
 * @version 1.0
 */
public class SimpleRuleDataAware implements RuleDataAware {

	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = -5682465955804417216L;

	/**
	 * Constants for data element's name
	 */
	private static final String PRISTINE_DATA = "PristineData";
	private static final String INCOMING_DATA = "IncomingData";
	private static final String RESULT_DATA = "ResultData";

	/**
	 * An instance of a collection of entities
	 */
	private Map<String, AbstractEntity> data = null;

	/**
	 * A default constructor
	 */
	public SimpleRuleDataAware() {
		super();
		data = new HashMap<String, AbstractEntity>();
	}

	/**
	 * Returns a printine entity stored in an input data collection
	 * 
	 * @return A printine version of an entity
	 */
	public AbstractEntity getPristineData() {
		return getData(PRISTINE_DATA);
	}

	/**
	 * Sets a pristine entity stored in an input data collection
	 * 
	 * @param pristine
	 *            A pristine version of an entity
	 */
	public void setPristineData(AbstractEntity pristine) {
		setData(PRISTINE_DATA, pristine);
	}

	/**
	 * Returns an incoming entity stored in an input data collection
	 * 
	 * @return An incoming version of an entity
	 */
	public AbstractEntity getIncomingData() {
		return getData(INCOMING_DATA);
	}

	/**
	 * Sets an incoming entity stored in an input data collection
	 * 
	 * @param incoming
	 *            An incoming version of an entity
	 */
	public void setIncomingData(AbstractEntity incoming) {
		Validate.notNull(incoming, "An incoming entity must not be null");
		setData(INCOMING_DATA, incoming);
	}

	/**
	 * Returns a result entity stored in an input data collection
	 * 
	 * @return A result version of an entity
	 */
	public AbstractEntity getResultData() {
		return getData(RESULT_DATA);
	}

	/**
	 * Sets a result entity stored in an input data collection
	 * 
	 * @param result
	 *            A result version of an entity
	 */
	public void setResultData(AbstractEntity result) {
		setData(RESULT_DATA, result);
	}

	/**
	 * Sets an entity with the specific name in a collection containing input
	 * and output data used in a rule execution
	 * 
	 * @param key
	 *            A unique name of an entity
	 * @param entity
	 *            An entity to set
	 * @throws IllegalArgumentException
	 *             Thrown if either a key or an entity is null or an entity
	 *             already exists
	 */
	protected void setData(String key, AbstractEntity entity) {
		Validate.notNull(key, "An entity's key must not be null");
		Validate.isTrue(!data.containsKey(key), "An entity with the same key " + key
				+ " already exists ");
		data.put(key, entity);
	}

	/**
	 * Returns an entity with the specific name in a collection containing input
	 * and output data used in a rule execution
	 * 
	 * @param key
	 *            A key to identify an entity in a collection
	 * @return AbstactEntity An entity
	 * @throws IllegalArgumentException
	 *             Thrown if an entity key is null
	 */
	protected AbstractEntity getData(String key) {
		Validate.notNull(key, "An entity's key must not be null");
		return (AbstractEntity) data.get(key);
	}

	/**
	 * @see java.lang.Object#finalize()
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		if (this.data != null) {
			this.data.clear();
		}
		this.data = null;
	}
}