/********************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.fw.report.jasperreport;

// Java classes
import java.io.ByteArrayOutputStream;

import org.apache.commons.lang.Validate;

import gov.va.med.fw.report.AbstractReportExporter;
import gov.va.med.fw.report.ReportConfiguration;
import gov.va.med.fw.report.ReportException;
import gov.va.med.fw.report.ReportTemplate;
import gov.va.med.fw.report.data.ReportData;
import gov.va.med.fw.report.excelreport.ExcelReport;
import gov.va.med.fw.report.jasperreport.data.JasperReportData;
import net.sf.jasperreports.engine.JRDataSource;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;

/**
 * Provides functionalities to fill a report and to export a report to a stream
 * synchronously and asynchronously. To export and to fill a report
 * asynchronously, configure a bean in spring context with a asynchronous
 * property set to true. A default behaviour is to fill a report synchronously.
 * 
 * Project: Framework</br> Created on: 2:27:15 PM </br>
 * 
 * @author DNS
 */
public abstract class AbstractJasperReportExporter extends AbstractReportExporter {

	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = -856131465569866311L;

	/**
	 * A flag to indicate whether or not to fill and to export a report
	 * asynchronously
	 */
	private boolean asynchronous = false;

	/**
	 * A default constructor
	 */
	protected AbstractJasperReportExporter() {
		super();
	}

	/**
	 * @see gov.va.med.fw.service.AbstractComponent#afterPropertiesSet()
	 */
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
		if (this.asynchronous) {
			Validate.notNull(getLoginManager(), "A login manager is missing");
		}
	}

	/**
	 * @return Returns the asynchronous.
	 */
	public boolean isAsynchronous() {
		return asynchronous;
	}

	/**
	 * @param asynchronous
	 *            The asynchronous to set.
	 */
	public void setAsynchronous(boolean asynchronous) {
		this.asynchronous = asynchronous;
	}

	/**
	 * Fills a report with data
	 * 
	 * @param config
	 * @param data
	 * @param template
	 * @return
	 * @throws ReportException
	 */
	protected Object fill(ReportConfiguration config, ReportData data, ReportTemplate template)
			throws ReportException {

		Validate.notNull(template, "A report template must not be NULL ");

		JasperPrint report = null;

		try {
			// Fill a jasper report template with data to create a
			// ready-to-print jasper report object
			report = JasperFillManager.fillReport(this.getJasperReport(config, template), config
					.getResourceMapping(), this.getJasperDataSource(data));
		} catch (Exception e) {
			logger.error("Failed to populate a report ", e);
			throw new ReportException("Failed to populate a report ", e);
		}

		return report;
	}

	/**
	 * Exports a report to a stream
	 * 
	 * @param report
	 *            A generated report
	 * @return A stream of generate report
	 * @throws ReportException
	 *             In case or errors getting an output stream of a generated
	 *             report
	 */
	protected ByteArrayOutputStream export(Object report) throws ReportException {

		if (!(report instanceof JasperPrint)) {
			throw new ReportException("Invalid report type to export: " + report);
		}

		return this.exportJasperReport((JasperPrint) report);
	}

	protected ExcelReport getExcelReport(ReportConfiguration config, ReportTemplate template)
			throws ReportException {
		return (ExcelReport) template.getExcelTemplate(config);
	}

	protected JasperReport getJasperReport(ReportConfiguration config, ReportTemplate template)
			throws ReportException {

		// Get a compiled jasper report template
		Object tpl = template.getCompiledTemplate(config);
		JasperReport jasperReport = tpl instanceof JasperReport ? (JasperReport) tpl : null;

		if (jasperReport == null) {
			throw new ReportException("Invalid type or missing report template");
		}
		return jasperReport;
	}

	protected JRDataSource getJasperDataSource(ReportData data) throws ReportException {

		// Get a jasper report data
		JasperReportData reportData = data instanceof JasperReportData ? (JasperReportData) data
				: null;
		JRDataSource jasperData = reportData != null ? reportData.getJasperDataSource() : null;

		if (jasperData == null) {
			throw new ReportException("Invalid type of report data. jasperData object is null");
		}
		return jasperData;
	}

	/**
	 * Performs the actual task to export a JasperPrint (read-to-print) report
	 * to an output stream
	 * 
	 * @param print
	 *            A report to be printed
	 * @return An output stream of a generated report
	 * @throws ReportException
	 *             In case or errors getting an output stream of a generated
	 *             report
	 */
	protected abstract ByteArrayOutputStream exportJasperReport(JasperPrint print)
			throws ReportException;
}