/********************************************************************
 * Copyriight 2008 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.persistent.ldap;

import java.net.ConnectException;
import java.net.NoRouteToHostException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.naming.NamingException;
import javax.naming.directory.DirContext;
import javax.naming.directory.SearchControls;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.ldap.CommunicationException;
import org.springframework.ldap.core.AttributesMapper;
import org.springframework.ldap.core.ContextMapper;
import org.springframework.ldap.core.DistinguishedName;
import org.springframework.ldap.core.LdapTemplate;
import org.springframework.ldap.core.support.DefaultDirObjectFactory;
import org.springframework.ldap.core.support.LdapContextSource;
import org.springframework.ldap.filter.AndFilter;
import org.springframework.ldap.filter.EqualsFilter;
import org.springframework.ldap.filter.LikeFilter;
import org.springframework.ldap.transaction.compensating.manager.TransactionAwareContextSourceProxy;
import org.springframework.stereotype.Repository;

import gov.va.med.fw.model.ldap.LdapConstants;
import gov.va.med.fw.model.ldap.LdapPerson;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.util.StringUtils;


/**
 * Data Access Object implementation for Ldap Group.
 * 
 * CISS ciss_framework Jun 23, 2008
 * 
 * @author DNS
 */
@Repository
public class LdapPersonDaoImpl extends AbstractComponent implements LdapPersonDao, LdapConstants, InitializingBean{

	private LdapTemplate ldapTemplate;

	private String domainContext = "DC=va,DC=gov";
	
	@Value(value="${ldapReadServerUrl1} ${ldapReadServerUrl2}")
	private String[] ldapServerUrls = null;
	private String groupOU  = "None";
	@Value(value ="${ldapUserId}")
	private String userDn;
	@Value(value="${ldapPassword}")
	private String ldapPassword;

	public void setGroupOU(String groupOU) {
		this.groupOU = groupOU;
	}

	/**
	 * @return the domainContext
	 */
	public String getDomainContext() {
		return domainContext;
	}

	/**
	 * @param domainContext
	 *            the domainContext to set
	 */
	public void setDomainContext(String domainContext) {
		if (StringUtils.isNotEmpty(domainContext)) {
			this.domainContext = domainContext;
		}
	}

	/**
	 * @return the ldapReadServerUrl
	 */
	public String[] getLdapServerUrls() {
		return ldapServerUrls;
	}

	/**
	 * @param ldapReadServerUrl
	 *            the ldapReadServerUrl to set
	 */
	public void setLdapServerUrls(String[] ldapReadServerUrls) {
		this.ldapServerUrls = ldapReadServerUrls;
	}

	private AttributesMapper<LdapPerson> getAttributesMapper() {
		PersonContextMapper personContextMapper = new PersonContextMapper();
		personContextMapper.setGroupOU(groupOU);
		return personContextMapper;
	}
	
	public List findAll() {
		EqualsFilter filter = new EqualsFilter(OBJECT_CLASS, USER);
		//Use ldapTemplate search instead to fix Fortify issue
				return ldapTemplate.search(DistinguishedName.EMPTY_PATH, filter.encode(),
							getAttributesMapper());
	}

	public LdapPerson findByDistinguishedName(String dn) {
		//Use ldapTemplate lookup instead to fix Fortify issue
		return (LdapPerson) ldapTemplate.lookup(dn, getAttributesMapper());
	}

	/**
	 * Searches ActiveDirectory by the given sAMAccountName
	 * 
	 * @param sAMAccountName
	 *            the sAmAccountaName to search by
	 * @return the found LdapPerson
	 */
	public LdapPerson findBySamAccountName(String sAMAccountName) {
		AndFilter andFilter = new AndFilter();
		andFilter.and(new EqualsFilter(OBJECT_CLASS, USER));
		andFilter.and(new EqualsFilter(SAM_ACCOUNT_NAME, sAMAccountName));
		@SuppressWarnings("unchecked")
		//use ldapTemplate to search to fix Fortify issue
		List<LdapPerson> results = ldapTemplate.search(DistinguishedName.EMPTY_PATH, andFilter.encode(), 
				getAttributesMapper());
		
		if (results != null && results.size() > 0)
			return (LdapPerson) results.get(0);
		else
			return null;
	}

	public List find(SearchCriteria criteria) {
		AndFilter andFilter = new AndFilter();
		andFilter.and(new EqualsFilter(OBJECT_CLASS, "user"));

		// Check for minimum required search fields
		if (!isValid(criteria)) {
			return new ArrayList();
		}
		if (StringUtils.isNotEmpty(criteria.getSAMAccountName())) {
			andFilter.and(new EqualsFilter(SAM_ACCOUNT_NAME, criteria.getSAMAccountName()));
		}
		if (StringUtils.isNotEmpty(criteria.getFirstName())) {
			// andFilter.and(new EqualsFilter(GIVEN_NAME,
			// criteria.getFirstName()));
			andFilter.and(new LikeFilter(GIVEN_NAME, criteria.getFirstName()));
		}
		if (StringUtils.isNotEmpty(criteria.getLastName())) {
			andFilter.and(new EqualsFilter(SN, criteria.getLastName()));
		}
		if (StringUtils.isNotEmpty(criteria.getMiddleName())) {
			andFilter.and(new EqualsFilter(MIDDLE_NAME, criteria.getMiddleName()));
		}
		if (StringUtils.isNotEmpty(criteria.getGroupName())) {
			andFilter.and(new EqualsFilter(MEMBER_OF, criteria.getGroupName()));
		}

		// This is a convenience for testing and is not used by the application
		if (StringUtils.isNotEmpty(criteria.getName())) {
			andFilter.and(new LikeFilter(CN, criteria.getName()));
		}

		//use ldaptemplate to search to fix Fortify issue
		return ldapTemplate.search(DistinguishedName.EMPTY_PATH, andFilter.encode(),
						getAttributesMapper());
	}

	/**
	 * Allow userid, lastname or group name search
	 * 
	 * @param criteria
	 * @return
	 */
	private boolean isValid(SearchCriteria criteria) {
		if (StringUtils.isNotEmpty(criteria.getSAMAccountName())
				|| StringUtils.isNotEmpty(criteria.getLastName())
				|| StringUtils.isNotEmpty(criteria.getName())
				|| StringUtils.isNotEmpty(criteria.getGroupName())) {
			return true;
		}
		return false;
	}

	/**
	 * Authenticate by logging into the ldap server
	 * 
	 * @param userDn
	 * @param password
	 * @return
	 */
	public boolean authenticate(String userDn, String password) throws CommunicationException {
		DirContext ctx = null;
		try {
			LdapContextSource context = new LdapContextSource();
			context.setUrls(getLdapServerUrls());
			context.setBase(getDomainContext());
			context.setUserDn(userDn);
			context.setPassword(password);
			context.setPooled(false);
			context.afterPropertiesSet();
			ctx = context.getReadOnlyContext();
			return true;
		} catch (Exception e) {
			logger.error("Unable to authenticate against LDAP", e);
			if(e instanceof CommunicationException && (ExceptionUtils.getRootCause(e) instanceof UnknownHostException || 
					ExceptionUtils.getRootCause(e) instanceof NoRouteToHostException ||
					ExceptionUtils.getRootCause(e) instanceof ConnectException)) {
				throw (CommunicationException) e; 
			}
			// TODO transalte the ldap messaged into user readable error
			// messages

			/*
			 * 525 - user not found 52e - invalid credentials 530 - not
			 * permitted to logon at this time 532 - password expired 533 -
			 * account disabled 701 - account expired 773 - user must reset
			 * password
			 */

			return false;
		} finally {
			close(ctx);
		}
	}

	/**
	 * Close the ldap directory context
	 * 
	 * @param ctx
	 */
	private void close(DirContext ctx) {
		if (ctx != null) {
			try {
				ctx.close();
			} catch (NamingException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
	}
	
	@Override
	public void afterPropertiesSet() throws Exception {
		// Setting the proper LDAP configurations
		final LdapContextSource contextSourceReadTarget = new LdapContextSource();
		contextSourceReadTarget.setUrls(ldapServerUrls);
		contextSourceReadTarget.setBase(domainContext);
		contextSourceReadTarget.setUserDn(userDn);
		contextSourceReadTarget.setPassword(ldapPassword);
		contextSourceReadTarget.setDirObjectFactory(DefaultDirObjectFactory.class);
		contextSourceReadTarget.setPooled(false);
		final Map<String, Object> map = new HashMap<>();
		map.put("java.naming.referral", "ignore");
		contextSourceReadTarget.setBaseEnvironmentProperties(map);
		contextSourceReadTarget.afterPropertiesSet();
		final TransactionAwareContextSourceProxy contextSourceRead = new TransactionAwareContextSourceProxy(contextSourceReadTarget);
		ldapTemplate = new LdapTemplate(contextSourceRead);
		ldapTemplate.setIgnorePartialResultException(true);
	}

}