/********************************************************************
 * Copyriight 2008 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.persistent.ldap;

import java.util.Iterator;
import java.util.List;

import org.springframework.ldap.core.DirContextAdapter;
import org.springframework.ldap.core.DirContextOperations;
import org.springframework.ldap.core.DistinguishedName;
import org.springframework.ldap.core.LdapOperations;
import org.springframework.stereotype.Repository;

import gov.va.med.fw.model.ldap.LdapConstants;
import gov.va.med.fw.model.ldap.LdapGroup;
import gov.va.med.fw.security.Role;

/**
 * Data Access Object implementation for Ldap Group.
 * 
 * CISS ciss_framework Jun 23, 2008
 * 
 * @author DNS
 */
@Repository
public class LdapGroupDaoImpl implements LdapGroupDao, LdapConstants {
	
	LdapOperations ldapWriteOperations;
	private List<String> environmentOuList;

	private DistinguishedName buildDn(String cn) {
		DistinguishedName dn = new DistinguishedName();

		for (Iterator i = environmentOuList.iterator(); i.hasNext();) {
			String orgUnit = (String) i.next();

			/*
			 * When a brand new user is being added to ohrs groups they will
			 * also be added to the CISS User group so the ou for Roles needs to
			 * be changed to CISS Roles rather than OHRS Roles
			 */
			if (cn.equalsIgnoreCase(Role.DEFAULT_CCHT_APP_ROLE)
					&& orgUnit.equalsIgnoreCase(OHRS_ROLES))
				dn.add("ou", CISS_ROLES);
			else
				dn.add("ou", orgUnit);
		}

		dn.add(CN, cn);
		return dn;
	}

	private DistinguishedName buildDnSetSamAccountName(LdapGroup group) {
		DistinguishedName dn = new DistinguishedName();

		for (Iterator i = environmentOuList.iterator(); i.hasNext();) {
			String ou = (String) i.next();
			dn.add("ou", ou);
			// append to the sAM account name if using the test or dev Ldap ou
			if (ou.startsWith(DEV_GROUPS) || ou.startsWith(TEST_GROUPS))
				group.setSamAccountName(buildEnvironmentSamAccountName(group.getSamAccountName(),
						ou));
		}

		dn.add(CN, group.getCnName());
		return dn;
	}

	private String buildEnvironmentSamAccountName(String sAM, String ou) {
		String envString = ou.replace(" ", "_");
		return (sAM + "_" + envString.substring(0, envString.indexOf(GROUPS)).toUpperCase());
	}

	private DirContextOperations setAttributes(DirContextOperations adapter, LdapGroup group) {
		adapter.setAttributeValues(OBJECT_CLASS, new String[] { TOP, GROUP });
		adapter.setAttributeValue(CN, group.getCnName());
		adapter.setAttributeValue("groupType", "-2147483640");
		adapter.setAttributeValue("sAMAccountName", group.getSamAccountName());
		if (group.getMembers() != null) {
			adapter.setAttributeValues(MEMBER, group.getMembers().toArray(new String[0]));
		}
		return adapter;
	}

	public void setLdapWriteOperations(LdapOperations ldapWriteOperations) {
		this.ldapWriteOperations = ldapWriteOperations;
	}


	public void setEnvironmentOuList(List<String> environmentOuList) {
		this.environmentOuList = environmentOuList;
	}

	public void create(LdapGroup group) {
		ldapWriteOperations.bind(buildDnSetSamAccountName(group), setAttributes(
				new DirContextAdapter(), group), null);
	}

	public void delete(LdapGroup group) {
		ldapWriteOperations.unbind(buildDn(group.getCnName()));
	}

}