/********************************************************************
 * Copyriight 2008 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.model.ldap;

import java.io.Serializable;
import java.util.Set;
import java.util.TreeSet;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.springframework.core.style.ToStringCreator;

import gov.va.med.fw.util.StringUtils;

/**
 * Class representing an Ldap person
 * 
 * CISS ciss_framework Jun 23, 2008
 * 
 * @author DNS
 */

public class LdapPerson implements Serializable {
	private static final long serialVersionUID = -8953862790009843528L;
	
	public static String DISABLED_USERS = "OU=Disabled Users";
    public static String DISABLED_ACCOUNTS = "OU=Disabled Accounts";
    
	private String dn;
	private String shortDn; // dn minus base
	private String samAccountName;
	private String userPrincipalName;
	private String fullName;
	private String displayName;
	private String lastName;
	private String firstName;
	private String middleName;
	private String badPasswordCount;
	private String title;
	private String department;
	private String description;
	private String office;
	private String telephoneNumber;
	private String email;
	private Set<String> groups = new TreeSet<String>();
	private Set<String> cissGroups = new TreeSet<String>();
	private Set<String> newGroups = new TreeSet<String>();
	private Set<String> deletedGroups = new TreeSet<String>();

	/**
	 * @return Returns the Distinguished Name of the Ldap Person.
	 */
	public String getDn() {
		return dn;
	}

	/**
	 * @param dn
	 *            The full Distinguished Name to set.
	 */
	public void setDn(String dn) {
		this.dn = dn;
	}

	/**
	 * @return Returns the Distinguished Name of the Ldap Person not including
	 *         the base portion of the DN, specifically:
	 *         DC=vha,DC=med,DC=va,DC=gov.
	 */
	public String getShortDn() {
		return shortDn;
	}

	/**
	 * @param shortDn
	 *            The Distinguished Name to set not including the base portion
	 *            specifically DC=vha,DC=med,DC=va,DC=gov.
	 */
	public void setShortDnString(String shortDn) {
		this.shortDn = shortDn;
	}

	/**
	 * @return Returns the Last Name of the Ldap Person.
	 */
	public String getLastName() {
		return lastName;
	}

	/**
	 * @param lastName
	 *            The Last Name to set.
	 */
	public void setLastName(String lastName) {
		this.lastName = lastName;
	}

	/**
	 * @return Returns the First Name of the Ldap Person.
	 */
	public String getFirstName() {
		return firstName;
	}

	/**
	 * @param firstName
	 *            The First Name to set.
	 */
	public void setFirstName(String firstName) {
		this.firstName = firstName;
	}

	/**
	 * @return Returns the Middle Name of the Ldap Person.
	 */
	public String getMiddleName() {
		return middleName;
	}

	/**
	 * @param middleName
	 *            The Middle Name to set.
	 */
	public void setMiddleName(String middleName) {
		this.middleName = middleName;
	}

	/**
	 * @return Returns the Middle Name of the Ldap Person.
	 */
	public String getBadPasswordCount() {
		return badPasswordCount;
	}

	/**
	 * @param middleName
	 *            The Middle Name to set.
	 */
	public void setBadPasswordCount(String badPasswordCount) {
		this.badPasswordCount = badPasswordCount;
	}

	/**
	 * @return Returns the sAMAccountName of the Ldap Person.
	 */
	public String getSamAccountName() {
		return samAccountName;
	}

	/**
	 * @param samAccountName
	 *            The sAMAccountName to set.
	 */
	public void setSamAccountName(String samAccountName) {
		this.samAccountName = samAccountName;
	}

	/**
	 * @return Returns the sAMAccountName of the Ldap Person.
	 */
	public String getUserPrincipalName() {
		return userPrincipalName;
	}

	/**
	 * @param samAccountName
	 *            The sAMAccountName to set.
	 */
	public void setUserPrincipalName(String userPrincipalName) {
		this.userPrincipalName = userPrincipalName;
	}

	/**
	 * @return Returns the Full Name of the Ldap Person.
	 */
	public String getFullName() {
		return fullName;
	}

	/**
	 * @param fullName
	 *            The Full Name to set.
	 */
	public void setFullName(String fullName) {
		this.fullName = fullName;
	}

	public String getDepartment() {
		return department;
	}

	public void setDepartment(String department) {
		this.department = department;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public String getTitle() {
		return title;
	}

	public void setTitle(String title) {
		this.title = title;
	}

	public String getOffice() {
		return office;
	}

	public void setOffice(String office) {
		this.office = office;
	}

	public String getDisplayName() {
		return displayName;
	}

	public void setDisplayName(String displayName) {
		this.displayName = displayName;
	}

	public String getTelephoneNumber() {
		return telephoneNumber;
	}

	public void setTelephoneNumber(String telephoneNumber) {
		this.telephoneNumber =
		    (StringUtils.isNotBlank(telephoneNumber) && telephoneNumber.length() > 30) ? telephoneNumber.substring(0,30) : telephoneNumber;
	}

	public String getEmail() {
		return email;
	}

	public void setEmail(String email) {
		this.email = email;
	}

	/**
	 * @return Returns a set of Group DNs or which the Ldap Person is a member.
	 */
	public Set<String> getGroups() {
		return groups;
	}

	/**
	 * @param groups
	 *            The set of Group DNs to set.
	 */
	public void setGroups(Set<String> groups) {
		this.groups = groups;
	}

	/**
	 * @return Returns a set of CISS Group CNs of which the Ldap Person is a
	 *         member. The group CN is the same as the group sAMAccountName.
	 */
	public Set<String> getCissGroups() {
		return cissGroups;
	}

	/**
	 * @param groups
	 *            The set of CISS Group CNs to set. The group CN is the same as
	 *            the group sAMAccountName.
	 */
	public void setCissGroups(Set<String> cissGroups) {
		this.cissGroups = cissGroups;
	}

	/**
	 * @return Returns a set of newly added CISS Group CNs of which the Ldap
	 *         Person is a member. The CN is the same as the sAMAccountName.
	 */
	public Set<String> getNewGroups() {
		return newGroups;
	}

	/**
	 * @param groups
	 *            The set of newly added CISS Group CNs to set. The CN is the
	 *            same as the sAMAccountName.
	 */
	public void setNewGroups(Set<String> newGroups) {
		this.newGroups = newGroups;
	}

	/**
	 * @return Returns a set of CISS Group CNs to remove the ldapPerson from.
	 *         The CN is the same as the sAMAccountName.
	 */
	public Set<String> getDeletedGroups() {
		return deletedGroups;
	}

	/**
	 * @param groups
	 *            The set of deleted CISS Group CNs to set. The CN is the same
	 *            as the sAMAccountName.
	 */
	public void setDeletedGroups(Set<String> deletedGroups) {
		this.deletedGroups = deletedGroups;
	}

	public boolean isDisabled() {
		if(groups != null && (groups.contains(DISABLED_USERS) || groups.contains(DISABLED_ACCOUNTS))) {
			return true;
		}
		return false;
	}
	
	public boolean equals(Object obj) {
		if (obj == null || getClass() != obj.getClass()) {
			return false;
		}
		return EqualsBuilder.reflectionEquals(this, obj);
	}

	public int hashCode() {
		return HashCodeBuilder.reflectionHashCode(this);
	}

	public String toString() {
		return new ToStringCreator(this).toString();
	}
}
