/********************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.fw.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.TimeZone;

import javax.persistence.Column;
import javax.persistence.Embedded;
import javax.persistence.MappedSuperclass;
import javax.persistence.Transient;

import org.apache.commons.lang.builder.ToStringBuilder;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;

import gov.va.med.ccht.model.CCHTAuditFields;
import gov.va.med.fw.model.ldap.LdapPerson;
import gov.va.med.fw.security.UserCredentials;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.util.StringUtils;

/**
 * UserPrincipal implementation.
 * 
 * @author Ghazenfer Mansoor
 * @version 1.0
 */
@MappedSuperclass
public class UserPrincipalImpl extends CCHTAuditFields implements UserPrincipal {

	private static final long serialVersionUID = -7044136313562745972L;

	// Authentication attributes
	private UserCredentials userCredentials = new UserCredentials();
	private Collection<GrantedAuthority> authorities = null;
	private Set<String> ldapRoles = null;
	private String userDn = null;
	// User details from ldap
	private String firstName = null;
	private String middleName = null;
	private String lastName = null;
	private String title;
	private String department;
	private String description;
	private String office;
	private String telephoneNumber;
	private String email;
	private TimeZone currentTimeZone;

	/**
	 * Default Constructor.
	 */
	public UserPrincipalImpl() {
		super();
	}

	/**
	 * Construct using name
	 * 
	 * @param name
	 */
	public UserPrincipalImpl(String name) {
		super();
		userCredentials.setUserID(name);
	}

	/**
	 * Construct using name and pwd.
	 * 
	 * @param name
	 * @param pwd
	 */
	public UserPrincipalImpl(String name, String pwd) {
		this(name);
		userCredentials.setPassword(pwd);
	}

	@Embedded
	public UserCredentials getUserCredentials() {
		return userCredentials;
	}

	public void setUserCredentials(final UserCredentials userCredentials) {
		this.userCredentials = userCredentials;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see java.security.Principal#getName()
	 */
	@Transient
	public String getName() {
		return userCredentials.getUserID();
	}

	/**
	 * Set the Principal name
	 * 
	 * @param name
	 */
	public void setName(String name) {
		userCredentials.setUserID(name);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#getPassword()
	 */
	@Transient
	public String getPassword() {
		return userCredentials.getPassword();
	}

	/**
	 * @param password
	 */
	public void setPassword(String password) {
		userCredentials.setPassword(password);
	}

	/**
	 * The first name of a user.
	 */
	@Column(name = "first_name")
	public String getFirstName() {
		return this.firstName;
	}

	/**
	 * @param firstName
	 */
	public void setFirstName(String firstName) {
		this.firstName = firstName;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.vha.fw.security.UserPrincipal#getMiddleName()
	 */
	@Column(name = "middle_name")
	public String getMiddleName() {
		return this.middleName;
	}

	/**
	 * @param middleName
	 */
	public void setMiddleName(String middleName) {
		this.middleName = middleName;
	}

	/**
	 * 
	 * @see gov.vha.fw.security.UserPrincipal#getLastName()
	 */
	@Column(name = "last_name")
	public String getLastName() {
		return this.lastName;
	}

	/**
	 * @param familyName
	 */
	public void setLastName(String familyName) {
		this.lastName = familyName;
	}

	/**
	 * @see gov.va.med.fw.security.UserPrincipal#isAnonymous()
	 */
	@Transient
	public boolean isAnonymous() {
		return userCredentials.isAnonymous();
	}

	/**
	 * Set the anonymous login property
	 * 
	 * @param anonymous
	 */
	public void setAnonymous(boolean anonymous) {
		userCredentials.setAnonymous(anonymous);
	}

	/**
	 * @return the ldapRoles
	 */
	@Transient
	public Set<String> getLdapRoles() {
		if (ldapRoles == null) {
			ldapRoles = new HashSet<String>();
		}
		return ldapRoles;
	}
	
	@Transient
	public Set<String> getCachedRoles() {
		return getLdapRoles();
	}

	/**
	 * @param ldapRoles
	 *            the ldapRoles to set
	 */
	public void setLdapRoles(Set<String> ldapRoles) {
		this.ldapRoles = ldapRoles;
	}

	/**
	 * default implementation
	 */
	@Transient
	public Collection<? extends GrantedAuthority> getAuthorities() {
		 return authorities == null ? new ArrayList() : authorities;
	}

	public void setAuthorities(Collection authorities) {
		if (authorities == null) {
			 this.authorities = new ArrayList();
		} else {
			 this.authorities = authorities;
		}
	}

	public boolean isPermissionGranted(String permission) {
		GrantedAuthority target = new SimpleGrantedAuthority(permission);
		Collection authorities = getAuthorities();
		return (authorities.contains(target));
	}

	@Override
	@Transient
	public boolean isAccountNonExpired() {
	return true;
	}

	@Override
	@Transient
	public boolean isAccountNonLocked() {
	return true;
	}

	@Override
	@Transient
	public boolean isCredentialsNonExpired() {
	return true;
	}

	@Override
	@Transient
	public boolean isEnabled() {
	return true;
	}

	@Transient
	public boolean isRoleGranted(String permission) {
		Set permissions = new HashSet();
		permissions.add(permission);
		return isRoleGranted(permissions);
	}

	@Transient
	public boolean isRoleGranted(Collection permissions) {
		if (permissions != null) {
			Set roles = getLdapRoles();
			for (Iterator iter = permissions.iterator(); iter.hasNext();) {
				if (roles.contains((String) iter.next())) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * @see com.VHA.fw.model.AbstractEntity#buildToString()
	 */
	protected void buildToString(ToStringBuilder builder) {
		// ^^^^^^^^^^^
		// Fix me.
//		super.buildToString(builder);
//		builder.append("name", getName());
//		builder.append("anonymous", isAnonymous());
//		builder.append("givenName", this.givenName);
//		builder.append("middleName", this.middleName);
//		builder.append("familyName", this.familyName);
//		builder.append("title", this.title);
//		builder.append("department", this.department);
//		builder.append("description", this.description);
	}

	/**
	 * @return Returns the logicalName.
	 */
	@Transient
	public String getLogicalName() {
		return userCredentials.getLogicalID();
	}

	/**
	 * Logical name used for all internal anonymous logins
	 * 
	 * @param logicalName
	 *            The logicalName to set.
	 */
	public void setLogicalName(String logicalName) {
		userCredentials.setLogicalID(logicalName);
	}

	/**
	 * equals method
	 */
	public boolean equals(Object o) {
		return ((this == o) || (o instanceof UserPrincipal && this.equals((UserPrincipal) o)));
	}

	/**
	 * Identifies whether user name is same
	 * 
	 * @param userPrincipal
	 * @return
	 */
	protected boolean equals(UserPrincipal userPrincipal) {
		return (userPrincipal != null && this.getName() != null && userPrincipal.getUsername() != null && this
				.getName().equals(userPrincipal.getUsername()));
	}

	/**
	 * Returns user name
	 */
	@Transient
	public String getUsername() {
		return userCredentials.getUserID();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#getFullName()
	 */
	@Transient
	public String getFullName() {

		StringBuilder name = new StringBuilder();
		if (lastName != null) {
			name.append(lastName);
			name.append(", ");
		}
		if (firstName != null) {
			name.append(firstName);
			name.append(" ");
		}

		if (middleName != null)
			name.append(middleName);
		String fullName = name.toString().trim();
		if (StringUtils.isEmpty(fullName)) {
			return getUsername();
		} else {
			return fullName;
		}
	}

	/**
	 * @return the userDn
	 */
	@Transient
	public String getUserDn() {
		return userDn;
	}

	/**
	 * @param userDn
	 *            the userDn to set
	 */
	public void setUserDn(String userDn) {
		this.userDn = userDn;
	}

	/**
	 * @param ldapPerson
	 *            the ldapPerson to set
	 */
	public void setLdapPerson(LdapPerson ldapPerson) {
		if (ldapPerson!=null)
		{	
			// set ldap roles, and user name details
			setUserDn(ldapPerson.getDn());
			setLdapRoles(ldapPerson.getCissGroups());
			setLastName(ldapPerson.getLastName());
			setFirstName(ldapPerson.getFirstName());
			setMiddleName(ldapPerson.getMiddleName());
			setTitle(ldapPerson.getTitle());
			setDepartment(ldapPerson.getDepartment());
			setDescription(ldapPerson.getDescription());
			setOffice(ldapPerson.getOffice());
			setEmail(ldapPerson.getEmail());
			setTelephoneNumber(ldapPerson.getTelephoneNumber());
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#getInactiveDate()
	 */
	@Transient
	public Date getInactiveDate() {
		// TODO Auto-generated method stub
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#getInactiveMessage()
	 */
	@Transient
	public String getInactiveMessage() {
		// TODO Auto-generated method stub
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#isAccountLocked()
	 */
	@Transient
	public boolean isAccountLocked() {
		// TODO Auto-generated method stub
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.security.UserPrincipal#isInactive()
	 */
	@Transient
	public boolean isInactive() {
		// TODO Auto-generated method stub
		return false;
	}

	/**
	 * @return the currentTimeZone
	 */
	@Transient
	public TimeZone getCurrentTimeZone() {
		return currentTimeZone;
	}

	/**
	 * @param currentTimeZone
	 *            the currentTimeZone to set
	 */
	public void setCurrentTimeZone(TimeZone currentTimeZone) {
		this.currentTimeZone = currentTimeZone;
	}

	@Transient
	public String getDepartment() {
		return department;
	}

	public void setDepartment(String department) {
		this.department = department;
	}

	@Transient
	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	@Transient
	public String getTitle() {
		return title;
	}

	public void setTitle(String title) {
		this.title = title;
	}

	@Transient
	public String getOffice() {
		return office;
	}

	public void setOffice(String office) {
		this.office = office;
	}

	@Column(name = "email")
	public String getEmail() {
		return email;
	}

	public void setEmail(String email) {
		this.email = email;
	}

	@Column(name = "telephone_number")
	public String getTelephoneNumber() {
		return telephoneNumber;
	}

	public void setTelephoneNumber(String telephoneNumber) {
		this.telephoneNumber = telephoneNumber;
	}
}