/********************************************************************
 * Copyriight 2005 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.fw.batchprocess;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.springframework.mail.MailException;
import org.springframework.mail.SimpleMailMessage;

import gov.va.med.ccht.util.ESAPIValidationType;
import gov.va.med.ccht.util.ESAPIValidator;
import gov.va.med.fw.mail.MailService;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.config.EnvironmentParamService;

/**
 * Initial javadoc for class ProcessStatisticsMailer. TODO - Add content here
 * 
 * Created Feb 6, 2006 9:43:08 AM
 * 
 * @author DNS   DNS
 */
public class ProcessStatisticsMailer extends AbstractComponent implements ProcessStatisticsHandler {
	public static String DEFAULT_SUBJECT_REPLACEMENT_TOKEN = "@subject@";
	public static String DEFAULT_APP_VERSION_REPLACEMENT_TOKEN = "@appVersion@";
	public static String DEFAULT_APP_ENV_REPLACEMENT_TOKEN = "@appEnv@";

	private MailService mailService;
	private SimpleMailMessage messageTemplate;
	private SimpleMailMessage failedMessageTemplate;
	private String subjectReplacementToken = DEFAULT_SUBJECT_REPLACEMENT_TOKEN;
	private boolean ignoreEmptyStatistics = false;
	private EnvironmentParamService environmentParamService = null;

	public void processStatistics(ProcessStatistics processStatistics) {
		processStatistics(processStatistics, processStatistics.getProcessName());
	}

	public void processStatisticsWithDynamicRecipients(ProcessStatistics processStatistics,
			List<String> emailRecipients) {
		doProcessStatistics(processStatistics, processStatistics.getProcessName(),
				(String[]) emailRecipients.toArray(new String[0]));
	}

	public void processStatistics(ProcessStatistics processStatistics,
			String subjectReplacementValue) {
		doProcessStatistics(processStatistics, subjectReplacementValue, null);
	}

	public void processFailure(String processName, String failureReason, List<String> emailRecipients)
	{
		if (emailRecipients == null)
			doProcessFailureNotification(processName, failureReason, null);
		else
			doProcessFailureNotification(processName, 
				failureReason, (String[]) emailRecipients.toArray(new String[0]));
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.fw.batchprocess.ProcessStatisticsHandler#processStatistics
	 * (gov.va.med.fw.batchprocess.ProcessStatistics)
	 */
	private void doProcessStatistics(ProcessStatistics processStatistics,
			String subjectReplacementValue, String[] dynamicEmailRecipients) {
		if (ignoreEmptyStatistics && processStatistics.getNumberOfTotalRecords() == 0)
			return;

		processStatistics.setExecutedOnServer(getServerName());

		if (messageTemplate.getTo() != null || dynamicEmailRecipients != null) {
			SimpleMailMessage template = messageTemplate;
					
			 //Fix Fortify Issues
			ESAPIValidator.validateStringInput(template.getSubject(), ESAPIValidationType.EmailSubject_Whitelist);
			
			if (StringUtils.isNotBlank(subjectReplacementToken)
					&& StringUtils.isNotBlank(messageTemplate.getSubject())
					&& messageTemplate.getSubject().indexOf(subjectReplacementToken) != -1) {
				template = new SimpleMailMessage(messageTemplate);
				
				template.setSubject(template.getSubject().replaceFirst(subjectReplacementToken,
						subjectReplacementValue));
			}

			if (this.environmentParamService != null
					&& StringUtils.isNotBlank(template.getSubject())) {
				if (template.getSubject().indexOf(DEFAULT_APP_VERSION_REPLACEMENT_TOKEN) != -1) {
					String appVersion = getAppVersion();
					if (StringUtils.isNotEmpty(appVersion))
						template.setSubject(template.getSubject().replaceFirst(
								DEFAULT_APP_VERSION_REPLACEMENT_TOKEN, appVersion));
				}

				if (template.getSubject().indexOf(DEFAULT_APP_ENV_REPLACEMENT_TOKEN) != -1) {
					String appEnv = getDeployedEnvironment();
					if (StringUtils.isNotEmpty(appEnv))
						template.setSubject(template.getSubject().replaceFirst(
								DEFAULT_APP_ENV_REPLACEMENT_TOKEN, appEnv));
				}

			}

			/**
			 * process email,addresses injected string needs to be split to
			 * individual addresses
			 */
			List<String> toAddresses = new ArrayList<String>();

			// use passed-in recipients (if there)
			if (dynamicEmailRecipients != null && dynamicEmailRecipients.length > 0) {
				for (String address : dynamicEmailRecipients)
					toAddresses.add(address);
			}
			// convert injected string of default addresses into a String[]
			else if (messageTemplate.getTo() != null && messageTemplate.getTo().length > 0) {
				String[] pList = StringUtils.split(messageTemplate.getTo()[0], ",");

				for (String address : pList)
					toAddresses.add(address);
			}
			// set the addresses
			template.setTo((String[]) toAddresses.toArray(new String[] {}));

			// don't expose exception just because can not send out statistics
			try {
				mailService.send(processStatistics.toFormattedMailString(), template);
			} catch (MailException e) {
				logger
						.error(
								"Unable to use mailService to send process statistics...logging and ignoring",
								e);
			}
		} else {
			if (logger.isWarnEnabled())
				logger
						.warn("Unable to email ProcessStatistics since there is no recipients configured/passed-in");
		}
	}

	
	private void doProcessFailureNotification(String subjectReplacementValue, 
			String failureReason, String[] dynamicEmailRecipients) {


		if (failedMessageTemplate.getTo() != null || dynamicEmailRecipients != null) {
			SimpleMailMessage template = failedMessageTemplate;
			
		   //Fix Fortify Issues
			ESAPIValidator.validateStringInput(template.getSubject(), ESAPIValidationType.EmailSubject_Whitelist);
			
			if (StringUtils.isNotBlank(subjectReplacementToken)
					&& StringUtils.isNotBlank(failedMessageTemplate.getSubject())
					&& failedMessageTemplate.getSubject().indexOf(subjectReplacementToken) != -1) {
				template = new SimpleMailMessage(failedMessageTemplate);
				template.setSubject(template.getSubject().replaceFirst(subjectReplacementToken,
						subjectReplacementValue));
			}

			if (this.environmentParamService != null
					&& StringUtils.isNotBlank(template.getSubject())) {
				if (template.getSubject().indexOf(DEFAULT_APP_VERSION_REPLACEMENT_TOKEN) != -1) {
					String appVersion = getAppVersion();
					if (StringUtils.isNotEmpty(appVersion))
						template.setSubject(template.getSubject().replaceFirst(
								DEFAULT_APP_VERSION_REPLACEMENT_TOKEN, appVersion));
				}

				if (template.getSubject().indexOf(DEFAULT_APP_ENV_REPLACEMENT_TOKEN) != -1) {
					String appEnv = getDeployedEnvironment();
					if (StringUtils.isNotEmpty(appEnv))
						template.setSubject(template.getSubject().replaceFirst(
								DEFAULT_APP_ENV_REPLACEMENT_TOKEN, appEnv));
				}

			}

			/**
			 * process email,addresses injected string needs to be split to
			 * individual addresses
			 */
			List<String> toAddresses = new ArrayList<String>();

			// use passed-in recipients (if there)
			if (dynamicEmailRecipients != null && dynamicEmailRecipients.length > 0) {
				for (String address : dynamicEmailRecipients)
					toAddresses.add(address);
			}
			// convert injected string of default addresses into a String[]
			else if (failedMessageTemplate.getTo() != null && failedMessageTemplate.getTo().length > 0) {
				String[] pList = StringUtils.split(failedMessageTemplate.getTo()[0], ",");

				for (String address : pList)
					toAddresses.add(address);
			}
			// set the addresses
			template.setTo((String[]) toAddresses.toArray(new String[] {}));

			// don't expose exception just because can not send out statistics
			String messageBody = 
				subjectReplacementValue + " execution on server " + 
				getServerName() + " failed.\n" ;
			if (StringUtils.isNotEmpty(failureReason)) {
				messageBody = messageBody + "Reason: " + messageBody;
			}
			
			
			try {
				mailService.send(messageBody, template);
			} catch (MailException e) {
				logger
						.error(
								"Unable to use mailService to send process failure notification...logging and ignoring",
								e);
			}
		} else {
			if (logger.isWarnEnabled())
				logger
						.warn("Unable to email process failure notification since there is no recipients configured/passed-in");
		}
	}	
	private String getAppVersion() {
		String appVersion = StringUtils.EMPTY;
		if (this.environmentParamService != null) {
			try {
				appVersion = this.environmentParamService.getVersion().getVersion();
			} catch (ServiceException e) {
				logger.error("Unable to determine Application Version for ProcessStatisticsMailer",
						e);
				appVersion = "";
			}
		}
		return appVersion;
	}

	private String getDeployedEnvironment() {
		return environmentParamService == null ? "" : environmentParamService
				.getDeployedEnvironment();
	}

	private String getServerName() {
		try {
			return InetAddress.getLocalHost().getHostName();
		} catch (Exception e) {
			return "Unknown";
		}
	}

	public void afterPropertiesSet() {
		Validate.notNull(mailService, "A MailService is required");
		Validate.notNull(messageTemplate, "A messageTemplate is required");
	}

	/**
	 * @return Returns the mailService.
	 */
	public MailService getMailService() {
		return mailService;
	}

	/**
	 * @param mailService
	 *            The mailService to set.
	 */
	public void setMailService(MailService mailService) {
		this.mailService = mailService;
	}

	/**
	 * @return Returns the messageTemplate.
	 */
	public SimpleMailMessage getMessageTemplate() {
		return messageTemplate;
	}

	/**
	 * @param messageTemplate
	 *            The messageTemplate to set.
	 */
	public void setMessageTemplate(SimpleMailMessage messageTemplate) {
		this.messageTemplate = messageTemplate;
	}

	public SimpleMailMessage getFailedMessageTemplate() {
		return failedMessageTemplate;
	}

	public void setFailedMessageTemplate(SimpleMailMessage failedMessageTemplate) {
		this.failedMessageTemplate = failedMessageTemplate;
	}

	/**
	 * @return Returns the subjectReplacementToken.
	 */
	public String getSubjectReplacementToken() {
		return subjectReplacementToken;
	}

	/**
	 * @param subjectReplacementToken
	 *            The subjectReplacementToken to set.
	 */
	public void setSubjectReplacementToken(String subjectReplacementToken) {
		this.subjectReplacementToken = subjectReplacementToken;
	}

	/**
	 * @return Returns the ignoreEmptyStatistics.
	 */
	public boolean isIgnoreEmptyStatistics() {
		return ignoreEmptyStatistics;
	}

	/**
	 * @param ignoreEmptyStatistics
	 *            The ignoreEmptyStatistics to set.
	 */
	public void setIgnoreEmptyStatistics(boolean ignoreEmptyStatistics) {
		this.ignoreEmptyStatistics = ignoreEmptyStatistics;
	}

	public EnvironmentParamService getEnvironmentParamService() {
		return environmentParamService;
	}

	public void setEnvironmentParamService(EnvironmentParamService environmentParamService) {
		this.environmentParamService = environmentParamService;
	}
}
