/*
 * Copyright 2006-2007 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.va.med.fw.batchprocess;

import java.io.Serializable;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.context.ApplicationContext;
import org.springframework.scheduling.quartz.QuartzJobBean;

import gov.va.med.fw.service.ConfigurationConstants;
import gov.va.med.fw.service.jms.JMSPayload;
import gov.va.med.fw.service.jms.MessageProducerService;
import gov.va.med.fw.util.date.TimeZoneUtils;

/**
 * @author Dave Syer
 * 
 */
public class JobLauncherDetails extends QuartzJobBean {

	/**
	 * Special key in job data map for the name of a job to run.
	 */
	static final String JOB_NAME = "jobName";
	static final int MAX_USER_ID_LENGTH = 30;

	private static Log log = LogFactory.getLog(JobLauncherDetails.class);
	private ApplicationContext applicationContext = null;
	private String jobName = null;
	private String runBatchJobJMSPayloadName;
	private String messageProducerServiceName = null;

	public ApplicationContext getApplicationContext() {
		return applicationContext;
	}

	public void setApplicationContext(ApplicationContext applicationContext) {
		this.applicationContext = applicationContext;
	}

	public String getJobName() {
		return jobName;
	}

	public void setJobName(String jobName) {
		this.jobName = jobName;
	}

	public String getMessageProducerServiceName() {
		return messageProducerServiceName;
	}

	public void setMessageProducerServiceName(String messageProducerServiceName) {
		this.messageProducerServiceName = messageProducerServiceName;
	}

	public String getRunBatchJobJMSPayloadName() {
		return runBatchJobJMSPayloadName;
	}

	public void setRunBatchJobJMSPayloadName(String runBatchJobJMSPayloadName) {
		this.runBatchJobJMSPayloadName = runBatchJobJMSPayloadName;
	}

	protected void executeInternal(JobExecutionContext context) {
		JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
		try {
			sendMessageForexecution();
		} catch (Exception e) {
			log.error("Could not execute job.", e);
		}

	}

	/**
	 * Copy parameters that are of the correct type over to
	 * {@link JobParameters}, ignoring jobName.
	 * 
	 * @return a {@link JobParameters} instance
	 */
	private JobParameters getJobParametersFromJobMap(Map jobDataMap) {

		JobParametersBuilder builder = new JobParametersBuilder();

		for (Iterator iterator = jobDataMap.entrySet().iterator(); iterator.hasNext();) {
			Entry entry = (Entry) iterator.next();
			String key = (String) entry.getKey();
			Object value = entry.getValue();
			if (value instanceof String && !key.equals(JOB_NAME)) {
				builder.addString(key, (String) value);
			} else if (value instanceof Float || value instanceof Double) {
				builder.addDouble(key, (Double) value);
			} else if (value instanceof Integer || value instanceof Long) {
				builder.addLong(key, (Long) value);
			} else if (value instanceof Date) {
				builder.addDate(key, (Date) value);
			} else {
				log.debug("JobDataMap contains values which are not job parameters (ignoring).");
			}
		}
		// Completed jobs can not be run with the same parameters
		// add runDate as parameter to run the jobs
		builder.addDate("runDate", new Date());
		return builder.toJobParameters();

	}

	private void sendMessageForexecution() throws Exception {
		// Get the message producer service
		MessageProducerService messageProducerService = (MessageProducerService) getApplicationContext()
				.getBean(getMessageProducerServiceName());

		Map<String, Object> map = new HashMap<String, Object>();
		map.put(JOB_NAME, jobName);
		// get pay load bean
		JMSPayload jmsPayLoad = (JMSPayload) getApplicationContext().getBean(
				getRunBatchJobJMSPayloadName());
		jmsPayLoad.setPayload((Serializable) map);

		// set header properties
		Map<String, Serializable> props = new HashMap<String, Serializable>();
		String logicalUserName = getJobName();
		if (logicalUserName.length() > MAX_USER_ID_LENGTH) {
			logicalUserName = logicalUserName.substring(0, MAX_USER_ID_LENGTH);
		}
		props.put(ConfigurationConstants.DEFAULT_MESSAGE_TYPE, jmsPayLoad
				.getTargetServiceDescriptor());
		props.put(ConfigurationConstants.DEFAULT_MESSAGE_INITIATER, logicalUserName);
		props.put(ConfigurationConstants.DEFAULT_MESSAGE_ORIGINATING_TIMEZONE, TimeZoneUtils.getTimeZone().getID());

		messageProducerService.send((Serializable) jmsPayLoad.getPayload(), props);
	}
}
