/********************************************************************
 * Copyriight 2006 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.fw.batchprocess;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.quartz.SchedulerException;
import org.springframework.batch.core.BatchStatus;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.configuration.JobLocator;
import org.springframework.batch.core.launch.JobLauncher;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.quartz.JobDetailBean;
import org.springframework.stereotype.Service;

import gov.va.med.ccht.persistent.JobDAO;
import gov.va.med.fw.model.batchprocess.BatchJobExecution;
import gov.va.med.fw.model.batchprocess.JobConfig;
import gov.va.med.fw.model.batchprocess.JobResult;
import gov.va.med.fw.model.batchprocess.JobStatus;
import gov.va.med.fw.scheduling.AuditableScheduledProcess;
import gov.va.med.fw.scheduling.ScheduledProcessInvocationContext;
import gov.va.med.fw.scheduling.ScheduledProcessTriggerEvent;
import gov.va.med.fw.scheduling.SchedulingService;
import gov.va.med.fw.security.LoginManager;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.Reflector;
import gov.va.med.fw.util.SystemUtils;

/**
 * Initial javadoc for class BatchProcessServiceImpl.
 * 
 * Created Sep 29, 2006 1:48:33 PM
 * 
 * @author DNS   DNS
 */
@Service 
public class BatchProcessServiceImpl extends AbstractComponent implements BatchProcessService {
	
	private static String NAMED_QUERY_GET_JOB_RESULTS_BY_STATUS = "jobResultQuery_GetByStatus";
	private static String NAMED_QUERY_GET_JOB_CONFIG_BY_JOB_NAME = "jobConfigQuery_GetByJobName";
	private static String NAMED_QUERY_GET_JOB_CONFIG_BY_JOB_NAME_AND_JOB_GROUP = "jobConfigQuery_GetByJobNameAndJobGroup";

	private static String PARAM_STATUS = "status";

	private static String PARAM_JOB_NAME = "jobName";

	private static String PARAM_JOB_GROUP = "jobGroup";

	private static final String NO_OP = "NOOP";

	private static final String TRANSLATED_NO_OP_STATUS = "Job executed; no data to process";

	@Autowired
	private LoginManager loginManager;

	@Autowired
	private SchedulingService schedulingService;
	
	private JobLauncher jobLauncher;
	private JobLocator jobLocator;
	
	@Autowired
	private JobDAO jobDao;

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.fw.batchprocess.BatchProcessService#saveJobResult(gov.va.med
	 * .fw.batchprocess.model.JobResult)
	 */
	public void saveJobResult(JobResult jr) {
		jobDao.saveJobResult(jr);
	}

	public JobResult getJobResult(final long id) {
		return jobDao.findJobResult(id);
	}

	@Override
	public JobResultData saveAndOverlayJobResult(long id, ProcessStatistics stats,
			int totalDataCount) throws ServiceException {
		JobResult jr = getAndLockJobResult(id);
		// create new ProcessStatistics
		ProcessStatistics updatedStats = null;
		try {
			// create new ProcessStatistics
			updatedStats = (ProcessStatistics) stats.getClass().newInstance();

			updatedStats.importFromCSV(jr.getStatistics()); // seed it with db
			// stats
			// now overlay worker stats
			updatedStats.overlayStats(stats);

			// update status
			if (updatedStats.getNumberOfTotalRecords() >= totalDataCount) {
				updatedStats.setProcessingEndDate(new Date());
				jr.setEndDate(updatedStats.getProcessingEndDate());
				if (updatedStats.getNumberOfErrorRecords() > 0)
					jr.setStatus(JobStatus.COMPLETE_WITH_ERROR);
				else
					jr.setStatus(JobStatus.COMPLETE);
			}
			jr.setStatistics(updatedStats.exportAsCSV());

			saveJobResult(jr);
		} catch (Exception e) {
			throw new ServiceException("Unable to saveAndOverlayJobResult", e);
		}
		return new JobResultData(jr, updatedStats);
	}

	private JobResult getAndLockJobResult(final long id) {
		// FIX TODO LockMode.UPGRADE);
		return jobDao.findJobResult(id);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.fw.batchprocess.BatchProcessService#saveJobConfig(gov.va.med
	 * .fw.batchprocess.model.JobConfig)
	 */
	public void saveJobConfig(JobConfig jc) {
		jobDao.saveJobConfig(jc);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.fw.batchprocess.BatchProcessService#getJobConfig(java.lang
	 * .String)
	 */
	@SuppressWarnings("unchecked")
	public JobConfig getJobConfig(String jobName, String jobGroup) {
		
		if (StringUtils.isEmpty(jobGroup)) { 
			return jobDao.findJobConfigs(jobName).get(0);
		} else {
			return jobDao.findJobConfigs(jobName, jobGroup).get(0);
		}
		
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.fw.batchprocess.BatchProcessService#getJobResults(gov.va.med
	 * .fw.batchprocess.model.JobStatus)
	 */
	public List<JobResult> getJobResults(JobStatus status) {
		return jobDao.findJobResultsByStatus(status.getCode());
	}

	public List<JobResult> getJobResults(BatchStatus status) {
		return jobDao.findJobResultsByStatus(status.toString());
	}

	/*
	 * i.job_name job_name, e.start_time start_time, e.end_time end_time,
	 * e.status status, e.exit_code exit_code, e.exit_message exit_message
	 */
	public List<JobResult> getFinishedJobResults(String jobName, String jobGroup) {
		return jobDao.findFinishedJobResults(jobName, jobGroup);
	}

	public void executeJob(ScheduledProcessTriggerEvent triggerEvent) throws ServiceException {
		executeJob((ScheduledProcessInvocationContext) triggerEvent.getPayload());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.fw.scheduling.SchedulingService#fireJob(gov.va.med.fw.scheduling
	 * .ScheduledProcessInvocationContext)
	 */
	public void executeJob(ScheduledProcessInvocationContext invocationContext)
			throws ServiceException {
		if (logger.isInfoEnabled()) {
			logger.info("Executing job/process [" + invocationContext.getJobName()
					+ "] for executionContext/user [" + invocationContext.getExecutionContext()
					+ "]");
		}

		String auditInfo = null;
		long beforeUsedMemory = SystemUtils.getUsedMemory();
		try {
			/*
			 * this implementation supports simple invocation via a Spring
			 * JobDetailName or from the actual service name and method name.
			 * The point here is we need the serviceName and methodName and
			 * there is more than one way to get it.
			 */
			String serviceName = null;
			String methodName = null;
			if (invocationContext.getJobDetailBeanName() != null) {
				JobDetailBean job = (JobDetailBean) this.getApplicationContext().getBean(
						invocationContext.getJobDetailBeanName(), JobDetailBean.class);
				serviceName = (String) job.getJobDataMap().get("serviceName");
				methodName = (String) job.getJobDataMap().get("methodName");
			} else {
				serviceName = invocationContext.getServiceName();
				methodName = invocationContext.getMethodName();
			}
			Object service = this.getApplicationContext().getBean(serviceName);

			auditInfo = service instanceof AuditableScheduledProcess ? ((AuditableScheduledProcess) service)
					.getAuditInfo(invocationContext)
					: serviceName + "." + methodName;
			getLoginManager().loginAnonymous(auditInfo);

			// see if service supports invocation via a
			// ScheduledProcessInvocationContext
			Object[] args = new Object[] { invocationContext };
			try {
				Reflector.findMethod(service, methodName, args);
			} catch (NoSuchMethodException e) {
				// no "standard" method for ScheduledProcessInvocationContext
				// ....just use wrapped arguments
				Object obj = invocationContext.getInvocationArguments();
				args = obj instanceof Object[] ? (Object[]) obj : new Object[] { obj };
			}

			Reflector.invoke(service, methodName, args);
		} catch (Throwable e) {
			throw new ServiceException(
					"Unable to execute job with ScheduledProcessInvocationContext: "
							+ invocationContext, e);
		} finally {
			getLoginManager().logout();

			SystemUtils.logMemory(true); // note this will suggest GC
			SystemUtils.logMemoryDelta(auditInfo, beforeUsedMemory);
		}
	}

	public void afterPropertiesSet() {
		Validate.notNull(schedulingService, "A schedulingService is required");
		Validate.notNull(loginManager, "A loginManager is required");		
	}

	/**
	 * @return Returns the loginManager.
	 */
	public LoginManager getLoginManager() {
		return loginManager;
	}

	/**
	 * @param loginManager
	 *            The loginManager to set.
	 */
	public void setLoginManager(LoginManager loginManager) {
		this.loginManager = loginManager;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.fw.batchprocess.BatchProcessService#cancelActiveJob(gov.va
	 * .med.fw.model.EntityKey)
	 */
	@Override
	public void cancelActiveJob(final long id) throws ServiceException {
		/*
		 * some jobs are sync and some are async - to cover these scenarios,
		 * must cancel the job in 2 different ways
		 */

		JobResult jobResult = jobDao.findJobResult(id);
		
		if (JobStatus.IN_PROCESS.getCode().equals(jobResult.getStatus().getCode())) {
			// do this first
			jobResult.setStatus(JobStatus.CANCELLED);
			saveJobResult(jobResult);

			// do this second
			try {
				schedulingService.interrupt(jobResult.getName(), jobResult.getGroup());
			} catch (SchedulerException e) {
				throw new ServiceException("Unable to interrupt the job during cancel", e);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.fw.batchprocess.BatchProcessService#errActiveJob(gov.va.med
	 * .fw.model.EntityKey)
	 */
	public void errActiveJob(final long id) {
		JobResult jobResult = jobDao.findJobResult(id);
		if (JobStatus.IN_PROCESS.getCode().equals(jobResult.getStatus().getCode())) {
			jobResult.setStatus(JobStatus.ERROR);
			saveJobResult(jobResult);
		}
	}

	@SuppressWarnings("unchecked")
	public void executeJob(Map jobDetails) throws ServiceException {
		String jobName = (String) jobDetails.get("jobName");
		JobParametersBuilder jobParametersBuilder  = new JobParametersBuilder() ;
		String runBy = (String) jobDetails.get("userId");
		jobParametersBuilder.addString("runBy", runBy == null ? SecurityContextHelper.getUserName() : runBy);
		jobParametersBuilder.addDate("Date", new Date());
				
		JobParameters jobParameters = jobParametersBuilder.toJobParameters();
		try {
			jobLauncher.run(jobLocator.getJob(jobName), jobParameters);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage());
		}
	}

	public JobLauncher getJobLauncher() {
		return jobLauncher;
	}

	public void setJobLauncher(JobLauncher jobLauncher) {
		this.jobLauncher = jobLauncher;
	}

	public JobLocator getJobLocator() {
		return jobLocator;
	}

	public void setJobLocator(JobLocator jobLocator) {
		this.jobLocator = jobLocator;
	}

	/**
	 * @return Returns the schedulingService.
	 */
	public SchedulingService getSchedulingService() {
		return schedulingService;
	}

	/**
	 * @param schedulingService
	 *            The schedulingService to set.
	 */
	public void setSchedulingService(SchedulingService schedulingService) {
		this.schedulingService = schedulingService;
	}

	@SuppressWarnings("unchecked")
	public List<BatchJobExecution> convertResults(List results) throws ServiceException {
		List<BatchJobExecution> jobResults = new ArrayList<BatchJobExecution>();
		try {
			for (int i = 0; i < results.size(); i++) {
				BatchJobExecution result = new BatchJobExecution();
				Object[] objs = (Object[]) results.get(i);
				result.setJobName((String) objs[0]);
				result.setStartDate((Date) objs[1]);
				result.setEndDate((Date) objs[2]);
				result.setStatus((String) objs[3]);
				if (objs[4] != null && ((String) objs[4]).equalsIgnoreCase(NO_OP)) {
					result.setExitStatus(TRANSLATED_NO_OP_STATUS);
				} else
					result.setExitStatus((String) objs[4]);
				result.setExitMessage((String) objs[5]);
				String runBy = (String) objs[6];
				String last_name = (String) objs[7];
				String first_name = (String) objs[8];
				String middle_name = (String) objs[9];
				if (last_name != null || first_name != null) {
					String name = (last_name == null ? "" : last_name + ", ")
							+ (first_name == null ? "" : first_name)
							+ (middle_name == null ? "" : " " + middle_name);
					result.setRunBy(name);
				} else {
					result.setRunBy(runBy);
				}
				result.setStatistics((String) objs[10]);

				jobResults.add(result);
			}
			return jobResults;
		} catch (Exception e) {
			throw new ServiceException("Unable to getJobResults with job name: ", e);
		}
	}

}
