package gov.va.med.config;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.NoResultException;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.event.LoggerListener;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.builders.WebSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.core.userdetails.AuthenticationUserDetailsService;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsByNameServiceWrapper;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.web.access.AccessDeniedHandlerImpl;
import org.springframework.security.web.access.ExceptionTranslationFilter;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.ExceptionMappingAuthenticationFailureHandler;
import org.springframework.security.web.authentication.LoginUrlAuthenticationEntryPoint;
import org.springframework.security.web.authentication.preauth.PreAuthenticatedAuthenticationProvider;
import org.springframework.security.web.authentication.preauth.PreAuthenticatedAuthenticationToken;
import org.springframework.security.web.authentication.preauth.RequestHeaderAuthenticationFilter;

import gov.va.med.ccht.persistent.SecurityDAO;
import gov.va.med.ccht.ui.CchtRequestHeaderAuthenticationFilter;
import gov.va.med.fw.security.AuthenticationManagerImpl;

@Configuration
@EnableWebSecurity
@EnableGlobalMethodSecurity(securedEnabled = true, prePostEnabled = true)
public class WebSecurityConfig extends WebSecurityConfigurerAdapter {
	private static final Log LOGGER = LogFactory.getLog(WebSecurityConfig.class);

	public static final String LOGIN_URL = "/register_1.html";

	private final boolean HTTPS_REQUIRED = true;
	private final String PRINCIPAL_REQUEST_HEADER = "adSamAccountName";

	@Autowired
	private SecurityDAO securityDao;

	@Bean
	@Override
	public AuthenticationManager authenticationManagerBean() throws Exception {
		final List<AuthenticationProvider> providers = new ArrayList<AuthenticationProvider>();
		providers.add(preAuthenticationProvider());
		providers.add(siteMinderPreAuthenticatedProvider());
		AuthenticationManagerImpl manager = new AuthenticationManagerImpl(providers);
		return manager;
	}

	@Bean
	public LoggerListener loggerListener() {
		return new LoggerListener();
	}

	@Override
	protected void configure(HttpSecurity http) throws Exception {
		http.csrf().disable();
		http.authorizeRequests() //
				.antMatchers("/favicon.ico").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/home.html").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/links.html").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/help.html").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/terms.html").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/registrationconfirm.html").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/ccht").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/ccht/searchAllQirs.html").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/searchAllQirs.html").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/manageQirs.html").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/editQir.html").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/newQir.html").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/viewQir.html**").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/qirExcelExport.html").hasAuthority("National Administrator")
				.antMatchers("/framework/**").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/**").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/docs/**").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/reports/**").hasAnyAuthority("CCHT_USER") //
				.antMatchers("/reports").access("!hasAuthority('Vendor')") //

				.and().formLogin()
					.loginPage(LOGIN_URL).permitAll()
					.failureHandler(myAuthenticationFailureHander())
					.defaultSuccessUrl("/home.html", true);

//		http.addFilterBefore(exceptionTranslationFilter(), CchtRequestHeaderAuthenticationFilter.class);
		http.addFilterBefore(devPreAuthFilter(), RequestHeaderAuthenticationFilter.class);
//		http.addFilterBefore(siteMinderFilter(), RequestHeaderAuthenticationFilter.class);
		
		http.logout().invalidateHttpSession(true);

		http.exceptionHandling().authenticationEntryPoint(authenticationEntryPoint()).accessDeniedPage(LOGIN_URL);
	}

	@Override
	public void configure(WebSecurity web) throws Exception {
		web.ignoring().antMatchers("/ccht/images/**");
		web.ignoring().antMatchers("/ccht/packages/**");
		web.ignoring().antMatchers("/ccht/scripts/**");
		web.ignoring().antMatchers("/ccht/css/**");
		web.ignoring().antMatchers("/ccht/loudev/**");
		web.ignoring().antMatchers("/ccht/DataTables/**");
		web.ignoring().antMatchers("/ccht/stylesheets/**");
		web.ignoring().antMatchers("/images/**");
		web.ignoring().antMatchers("/packages/**");
		web.ignoring().antMatchers("/scripts/**");
		web.ignoring().antMatchers("/css/**");
		web.ignoring().antMatchers("/loudev/**");
		web.ignoring().antMatchers("/DataTables/**");
		web.ignoring().antMatchers("/stylesheets/**");
		web.ignoring().antMatchers("/register_1.html");
		web.ignoring().antMatchers("/register_2.html");
		web.ignoring().antMatchers("/registrationConfirm.html");
		
	}

	@Bean
	public AuthenticationFailureHandler myAuthenticationFailureHander() {
		ExceptionMappingAuthenticationFailureHandler h = new ExceptionMappingAuthenticationFailureHandler();
		Map<String, String> failureUrlMap = new HashMap<>();
		failureUrlMap.put(BadCredentialsException.class.getName(), LOGIN_URL);
		failureUrlMap.put(NoResultException.class.getName(), LOGIN_URL);
		failureUrlMap.put(UsernameNotFoundException.class.getName(), LOGIN_URL);

		h.setExceptionMappings(failureUrlMap);
		h.setDefaultFailureUrl(LOGIN_URL);
		return h;
	}

	@Bean
	public AuthenticationProvider preAuthenticationProvider() {

		PreAuthenticatedAuthenticationProvider preAuthenticationProvider = new PreAuthenticatedAuthenticationProvider();
		preAuthenticationProvider.setPreAuthenticatedUserDetailsService(
				new AuthenticationUserDetailsService<PreAuthenticatedAuthenticationToken>() {
					@Override
					public UserDetails loadUserDetails(PreAuthenticatedAuthenticationToken token)
							throws UsernameNotFoundException {
						Object o = token.getPrincipal();
						if (o instanceof UserDetails)
							return (UserDetails) o;
						if (o instanceof String) {
							try {
								return securityDao.loadUserByUsername((String) o);
							}
							catch(Exception e) {
								throw new UsernameNotFoundException("User is not registered yet.", e);
							}
						}
						return null;
					}
				});
		preAuthenticationProvider.setThrowExceptionWhenTokenRejected(false);
		return preAuthenticationProvider;
	}

	@Bean
	public AuthenticationProvider siteMinderPreAuthenticatedProvider() {
		PreAuthenticatedAuthenticationProvider p = new PreAuthenticatedAuthenticationProvider();
		UserDetailsByNameServiceWrapper<PreAuthenticatedAuthenticationToken> uds = new UserDetailsByNameServiceWrapper<>(
				securityDao);
		p.setPreAuthenticatedUserDetailsService(uds);
		return p;
	}

	@Bean
	public CchtRequestHeaderAuthenticationFilter devPreAuthFilter() throws Exception {
		CchtRequestHeaderAuthenticationFilter f = new CchtRequestHeaderAuthenticationFilter();
		f.setAuthenticationManager(authenticationManagerBean());
		f.setExceptionIfHeaderMissing(false);
		f.setPrincipalRequestHeader(PRINCIPAL_REQUEST_HEADER);

		return f;
	}

	@Bean
	public RequestHeaderAuthenticationFilter siteMinderFilter() throws Exception {

		RequestHeaderAuthenticationFilter f = new RequestHeaderAuthenticationFilter() {
			@Override
			public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
					throws IOException, ServletException {
				HttpServletRequest r = (HttpServletRequest) request;
				if (LOGGER.isDebugEnabled()) {
					Enumeration<String> headerNames = r.getHeaderNames();
					while (headerNames.hasMoreElements()) {
						String headerName = (String) headerNames.nextElement();
						LOGGER.debug("header:" + headerName + " " + r.getHeader(headerName));
					}
				}

				super.doFilter(request, response, chain);
			}
		};
		f.setAuthenticationManager(authenticationManagerBean());
		f.setAuthenticationFailureHandler(myAuthenticationFailureHander());
		f.setPrincipalRequestHeader(PRINCIPAL_REQUEST_HEADER);
		f.setExceptionIfHeaderMissing(false);
		return f;
	}

	@Bean
	public LoginUrlAuthenticationEntryPoint authenticationEntryPoint() {
		LoginUrlAuthenticationEntryPoint authenticationEntryPoint = new LoginUrlAuthenticationEntryPoint(LOGIN_URL);
		authenticationEntryPoint.setForceHttps(HTTPS_REQUIRED);
		return authenticationEntryPoint;
	}

	 @Bean
	 public ExceptionTranslationFilter exceptionTranslationFilter() {
		 AccessDeniedHandlerImpl accessDeniedHandler = new AccessDeniedHandlerImpl();
		 accessDeniedHandler.setErrorPage(LOGIN_URL);
		 ExceptionTranslationFilter exceptionTranslationFilter = new ExceptionTranslationFilter(
				 authenticationEntryPoint());
		 exceptionTranslationFilter.setAccessDeniedHandler(accessDeniedHandler);
		 return exceptionTranslationFilter;
	 }
}
